/* install-datebook.c:  Pilot datebook installer
 *
 * Mods 1998 by Jim Rees, University of Michigan
 * Copyright 1997, Tero Kivinen
 * Copyright 1996, Robert A. Kaplan (original install-todos.c)
 *
 * -n: don't actually talk to Palm
 * -L: copy input to output in canonical form
 * -v: verbose output
 *
 * Input file format is
 * begin time|end time|description|note
 *
 * This is free software, licensed under the GNU Public License V2.
 * See the file COPYING for details.
 */

#include <sys/types.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <fcntl.h>

#include "pi-source.h"
#include "pi-socket.h"
#include "pi-dlp.h"
#include "pi-datebook.h"

extern int pilot_connect(char *port);
extern time_t get_date(char *p, time_t *now);
extern time_t parsedate(char * p);

int Lflag, nflag, vflag;

int main(int ac, char *av[])
{
    char *cp, *pilotport = "/dev/cua00";
    int db;
    int sd;
    int c, n;
    struct Appointment apt;
    static unsigned char aptbuf[4096];
    FILE *f;
    struct PilotUser U;
#ifdef foo
    int Appointment_size;
    unsigned char Appointment_buf[0xffff];
    int ret;
    int filelen;
    char *cPtr, *cp;
    char *file_text;
    int fieldno;
#endif
    char field[4][80];
    char buf[1200], note[3000];
    time_t t;
    struct tm *tm;
    int lineno = 0;

    while ((c = getopt(ac, av, "Lnv")) != -1) {
	switch (c) {
	case 'L':
	    Lflag = 1;
	    break;
	case 'n':
	    nflag = 1;
	    break;
	case 'v':
	    vflag = 1;
	    break;
	}
    }
    ac -= optind;
    av += optind;

    if ((cp = getenv("PILOTPORT")) != NULL)
	pilotport = cp;

    if (ac > 2) {
	fprintf(stderr,"usage: %s [file]\n", av[0]);
	exit(2);
    }

    if (!nflag) {
	if ((sd = pilot_connect(pilotport)) < 0) {
	    perror("pi_socket");
	    exit(1);
	}

	/* Ask the pilot who it is. */
	dlp_ReadUserInfo(sd, &U);

	/* Tell user (via Pilot) that we are starting things up */
	dlp_OpenConduit(sd);

	/* Open the Datebook's database, store access handle in db */
	if (dlp_OpenDB(sd, 0, 0x80|0x40, "DatebookDB", &db) < 0) {
	    puts("Unable to open DatebookDB");
	    dlp_AddSyncLogEntry(sd, "Unable to open DatebookDB.\n");
	    pi_close(sd);
	    exit(1);
	}
    }

    if (ac > 1) {
	f = fopen(av[1], "r");
	if (!f) {
	    fprintf(stderr, "can't open %s\n", av[1]);
	    exit(3);
	}
    } else
	f = stdin;

    time(&t);

    while (fgets(buf, sizeof buf, f) != NULL) {
	lineno++;
	if (buf[0] == '#' || buf[0] == '\n')
	    continue;
	n = sscanf(buf, "%76[^|]|%76[^|]|%76[^|]|%76s", field[0], field[1], field[2], field[3]);
	if (n < 3) {
	    fprintf(stderr, "line %d: Bad format: %s", lineno, buf);
	    continue;
	}

	bzero(&apt, sizeof apt);
	apt.repeatType = repeatNone;

	/* read in the description */
	apt.description = field[2];
	if (apt.description[0] == '\0')
	    /* no description */
	    continue;

	if (n > 3 && !strcasecmp(field[3], "note")) {
	    /* Read in a note */
	    note[0] = '\0';
	    while (fgets(buf, sizeof buf, f) != NULL) {
		char dummy[80];
		lineno++;
		if (sscanf(buf, "%76[^|]|%76[^|]|%76[^|]|", dummy, dummy, dummy) == 3)
		    fprintf(stderr, "line %d: missing END?\n", lineno);
		if (!strcasecmp(buf, "end\n"))
		    break;
		if (strlen(note) + strlen(buf) + 1 < sizeof note)
		    strcat(note, buf);
	    }
	    /* strip trailing newline */
	    if (note[strlen(note) - 1] == '\n')
		note[strlen(note) - 1] = '\0';
	    apt.note = note;
	} else
	    apt.note = NULL;

	/* parse date fields */
	if (!strcasecmp(field[0], "none")) {
	    /* no start time */
	    apt.event = 1;
	} else {
	    apt.event = 0;
	    t = get_date(field[0], &t);
	    if (t == -1) {
		fprintf(stderr, "line %d: Invalid start date or time: %s\n", lineno, field[0]);
		continue;
	    }
	    apt.begin = *localtime(&t);
	    if (Lflag) {
		tm = &apt.begin;
		printf("%4d-%02d-%02d %02d:%02d|",
		       tm->tm_year+1900, tm->tm_mon+1, tm->tm_mday, tm->tm_hour, tm->tm_min);
	    }

	    t = get_date(field[1], &t);
	    if (t == -1) {
		fprintf(stderr, "line %d: Invalid end date or time: %s\n", lineno, field[1]);
		continue;
	    }
	    apt.end = *localtime(&t);
	    if (Lflag) {
		tm = &apt.end;
		printf("%4d-%02d-%02d %02d:%02d|%s|",
		       tm->tm_year+1900, tm->tm_mon+1, tm->tm_mday, tm->tm_hour, tm->tm_min, apt.description);
		if (apt.note)
		    printf("note\n%s\nEND\n", apt.note);
		else
		    printf("\n");
	    }
	}

	if (vflag) {
	    /* Do this in pieces because asctime returns a pointer to static */
	    printf("Begin: %s", asctime(&apt.begin));
	    printf("End: %sDescription: %s\n", asctime(&apt.end), apt.description);
	    if (apt.note)
		printf("%s\n", apt.note);
	}

	/* Add it to the datebook */
	if (!nflag) {
	    n = pack_Appointment(&apt, aptbuf, sizeof aptbuf);
	    dlp_WriteRecord(sd, db, 0, 0, 0, aptbuf, n, 0);
	}
    }

    if (!nflag) {
	/* Close the database */
	dlp_CloseDB(sd, db);

	/* Tell the user who it is, with a different PC id. */
	U.lastSyncPC = 0x00010000;
	U.successfulSyncDate = time(NULL);
	U.lastSyncDate = U.successfulSyncDate;
	dlp_WriteUserInfo(sd, &U);

	dlp_AddSyncLogEntry(sd, "Loaded USENIX data\n");

	/* All of the following code is now unnecessary, but harmless */

	dlp_EndOfSync(sd, 0);
	pi_close(sd);
    }
    exit(0);
}
