/*
 * Written by Jim Rees at University of Michigan.
 * March 26, 2002
 *
 * usage
 * -p port
 * -D: delete from database
 * -n: don't transfer files
 * -q: don't prompt for hotsync
 * -t: use photo title as file name
 * -v: be verbose
 */

/*
copyright 2002
the regents of the university of michigan
all rights reserved

permission is granted to use, copy, create derivative works 
and redistribute this software and such derivative works 
for any purpose, so long as the name of the university of 
michigan is not used in any advertising or publicity 
pertaining to the use or distribution of this software 
without specific, written prior authorization.  if the 
above copyright notice or any other identification of the 
university of michigan is included in any copy of any 
portion of this software, then the disclaimer below must 
also be included.

this software is provided as is, without representation 
from the university of michigan as to its fitness for any 
purpose, and without warranty by the university of 
michigan of any kind, either express or implied, including 
without limitation the implied warranties of 
merchantability and fitness for a particular purpose. the 
regents of the university of michigan shall not be liable 
for any damages, including special, indirect, incidental, or 
consequential damages, with respect to any claim arising 
out of or in connection with the use of the software, even 
if it has been or is hereafter advised of the possibility of 
such damages.
*/

#include <sys/types.h>
#include <sys/stat.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <fcntl.h>

#include "pi-source.h"
#include "pi-socket.h"
#include "pi-dlp.h"

extern int pilot_connect(char *port);

#define MAXPHOTO 1000
#define TIMEDIFF 2082844800L

#define hinib(x) ((x) >> 4)
#define lonib(x) ((x) & 0xf)

int isdup(unsigned long ts);

char *pilotport = "/dev/pilot";

int Dflag, nflag, qflag, tflag, vflag;
unsigned long tstab[MAXPHOTO];

struct {
    char name[24];
    unsigned char pad1[2];
    unsigned char timestamp[4];
    unsigned char pad2[6];
} hdr;

int
main(int ac, char *av[])
{
    int i, sd, db, id, fd, len, attr, cat, x, y, b;
    char *cp, fname[32];
    FILE *f;
    unsigned char buf[8 * 1024], px;
    struct pi_sockaddr addr;
    unsigned long tstamp;
    time_t t;
    struct tm *tmp;

    if ((cp = getenv("PILOTPORT")) != NULL)
	pilotport = cp;

    while ((i = getopt(ac, av, "p:Dnqtv")) != -1) {
	switch (i) {
	case 'p':
	    pilotport = optarg;
	    break;
	case 'D':
	    Dflag = 1;
	    break;
	case 'n':
	    nflag = 1;
	    break;
	case 'q':
	    qflag = 1;
	    break;
	case 't':
	    tflag = 1;
	    break;
	case 'v':
	    vflag = 1;
	    break;
	}
    }

    sd = pilot_connect(pilotport);
    if (sd < 0)
	exit(1);

    if(dlp_OpenDB(sd, 0, dlpOpenReadWrite, "WQVLinkDB", &db) < 0) {
	fprintf(stderr, "Unable to open WQVLinkDB\n");
	dlp_AddSyncLogEntry(sd, "Unable to open MailDB.\n");
	exit(1);
    }

    for (id = 0; id < MAXPHOTO; id++) {
	if ((len = dlp_ReadRecordByIndex(sd, db, id, buf, 0, 0, &attr, &cat)) < 0)
	    break;
	if (attr & (dlpRecAttrDeleted | dlpRecAttrArchived))
	    continue;
	bcopy(buf, &hdr, sizeof hdr);
	bcopy(hdr.timestamp, &tstamp, sizeof tstamp);
	t = ntohl(tstamp) - TIMEDIFF;
	while (isdup(t))
	    t += 60;
	if (!tflag || hdr.name[0] == '\0') {
	    tmp = gmtime(&t);
	    sprintf(fname, "%04d%02d%02d%02d%02d.pgm",
		    tmp->tm_year + 1900, tmp->tm_mon + 1, tmp->tm_mday,
		    tmp->tm_hour, tmp->tm_min);
	} else
	    sprintf(fname, "%s.pgm", hdr.name);
	if (vflag)
	    printf("got rec %d title \"%s\" file \"%s\"\n", id, hdr.name, fname);
	if (nflag)
	    continue;
	if ((fd = open(fname, (O_WRONLY | O_CREAT | O_EXCL), 0666)) < 0)
	    continue;
	if ((f = fdopen(fd, "w")) == NULL) {
	    perror(fname);
	    continue;
	}
	fprintf(f, "P5\n# WQV %s\n120 120\n16\n", fname);
	for (y = 0; y < 120; y++) {
	    for (x = 0; x < 120; x++) {
		b = buf[sizeof hdr + (y * 60 + x / 2)];
		px = 15 - ((x % 2) ? lonib(b) : hinib(b));
		if (fwrite(&px, 1, 1, f) != 1) {
		    perror(fname);
		    fclose(f);
		    goto ng;
		}
	    }
	}
	if (fclose(f) == EOF) {
	    perror(fname);
	    goto ng;
	}
	if (Dflag)
	    dlp_DeleteRecord(sd, db, 0, id);
    ng:
    }

    dlp_CloseDB(sd, db);
    dlp_AddSyncLogEntry(sd, "WQVLink");
    pi_close(sd);
    exit(0);
}

int
isdup(unsigned long ts)
{
    int i;

    for (i = 0; i < MAXPHOTO && tstab[i]; i++)
	if (ts == tstab[i])
	    return 1;
    tstab[i] = ts;
    return 0;
}
