/*
 * Simple morse code sender.
 * Jim Rees University of Michigan April 1998
 */

#pragma pack(2)

#include <Common.h>
#include <System/SysAll.h>
#include <System/DataMgr.h>
#include <System/SoundMgr.h>
#include <System/SysEvtMgr.h>
#include <UI/UIAll.h>

#include <System/Unix/unix_stdio.h>
#include <System/Unix/unix_stdlib.h>
#include <System/Unix/sys_types.h>

#include "morse.h"
#include "smorse.h"

#define SmorseAppID    'sMRS'

#define DFLTWPM	20
#define MAXWPM	80

#define DIT	1
#define DAH	3
#define IBG	1	/* inter-bit gap */
#define ICG	3	/* inter-character gap */
#define IWG	7	/* inter-word gap */

static Boolean smorse(EventPtr event);

extern struct morse tab[];

FieldPtr wpmfield, infield, immfield;
char *mbits[128];
int wpm, ticksperdit, msperdit;

DWord
PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{
    int err;

    if (cmd == sysAppLaunchCmdNormalLaunch) {
	err = StartApplication();
	if (err)
	    return err;
	EventLoop();
	return 0;
    } else
	return sysErrParamErr;
}

StartApplication()
{
    struct morse *mtp;

    for (mtp = tab; mtp->c; mtp++)
	mbits[mtp->c] = mtp->bits;

    FrmGotoForm(formID_smorse);

    return 0;
}

EventLoop()
{
    short err;
    int formID;
    FormPtr form;
    EventType event;

    do {
	EvtGetEvent(&event, 200);

	if (SysHandleEvent(&event))
	    continue;
	if (MenuHandleEvent((void *)0, &event, &err))
	    continue;

	if (event.eType == frmLoadEvent) {
	    formID = event.data.frmLoad.formID;
	    form = FrmInitForm(formID);
	    FrmSetActiveForm(form);
	    switch (formID) {
	    case formID_smorse:
		FrmSetEventHandler(form, (FormEventHandlerPtr) smorse);
		break;
	    }
	}
	FrmDispatchEvent(&event);
    } while (event.eType != appStopEvent);
}

static Boolean
smorse(EventPtr event)
{
    FormPtr form = FrmGetActiveForm();
    Word field = FrmGetFocus(form);
    int id, handled = 0;

    switch (event->eType) {
    case frmOpenEvent:
	CtlSetValue(FrmGetObjectPtr(form, FrmGetObjectIndex(form, 2000 + DFLTWPM)), 1);
	wpm = DFLTWPM;
	wpmfield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_wpm));
	infield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_in));
	immfield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_imm));
	FrmDrawForm(form);
	FrmSetFocus(form, FrmGetObjectIndex(form, fieldID_in));
	handled = 1;
	break;

    case ctlSelectEvent:
	id = event->data.ctlEnter.controlID;
	if (id == buttonID_send) {
	    FrmSetFocus(form, FrmGetObjectIndex(form, fieldID_in));
	    msendfield(infield);
	    handled = 1;
	} else if (id / 1000 == 2) {
	    /* speed select pushbutton */
	    wpm = id % 1000;
	    handled = 1;
	}
	break;

    case keyDownEvent:
	if (field == FrmGetObjectIndex(form, fieldID_in) && event->data.keyDown.chr == '\n') {
	    msendfield(infield);
	    handled = 1;
	} else if (field == FrmGetObjectIndex(form, fieldID_imm)) {
	    if (msetspeed() >= 0)
		msend(event->data.keyDown.chr);
	    FldDelete(immfield, 0, 1);
	}
	break;

    case menuEvent:
	switch (event->data.menu.itemID) {
	case menuitemID_about:
	    FrmAlert(alertID_about);
	    break;
	case menuitemID_copy:
	    if (field != noFocus)
		FldCopy(FrmGetObjectPtr(form, field));
	    break;
	case menuitemID_cut:
	    if (field != noFocus)
		FldCut(FrmGetObjectPtr(form, field));
	    break;
	case menuitemID_paste:
	    if (field != noFocus)
		FldPaste(FrmGetObjectPtr(form, field));
	    break;
	}
	handled = 1;
	break;

    }
    return handled;
}

msendfield(FieldPtr field)
{
    char *s;
    int pos;

    if (FldGetTextLength(field) <= 0)
	return 0;
    if (msetspeed() < 0)
	return -1;
    s = FldGetTextPtr(field);
    FldSetInsertionPoint(field, pos = 0);
    while (*s) {
	if (msend(*s++) < 0)
	    break;
	FldSetInsPtPosition(field, ++pos);
    }
    return 0;
}

msend(int c)
{
    char *cp;

    if (c & 0x80)
	return;
    if (c >= 'A' && c <= 'Z')
	c |= 0x20;

    for (cp = mbits[c]; cp && *cp; cp++) {
	switch (*cp) {
	case '.':
	    mtone(DIT);
	    break;
	case '-':
	    mtone(DAH);
	    break;
	case ' ':
	    mwait(IWG - 2 * ICG);
	    break;
	}
	mwait(IBG);
	if (EvtSysEventAvail(true))
	    return -1;
    }
    mwait(ICG - IBG);
    return 0;
}

msetspeed()
{
    int n = wpm;
    char *s;

    if (n == 99) {
	/* custom speed */
	s = FldGetTextPtr(wpmfield);
	n = s ? atoi(s) : 0;
    }
    if (n < 1 || n > MAXWPM) {
	FrmAlert(alertID_errspeed);
	return -1;
    } else {
	ticksperdit = (SysTicksPerSecond() * 5) / (n * 6);
	msperdit = 5000 / (n * 6);
    }
    return 0;
}

mtone(int n)
{
    SndCommandType cmd;

    cmd.cmd = sndCmdFreqDurationAmp;
    cmd.param1 = 1000;
    cmd.param2 = msperdit * n;
    cmd.param3 = sndDefaultAmp;

    SndDoCmd(NULL, &cmd, 0);
}

mwait(int n)
{
    SysTaskDelay(ticksperdit * n);
}

/*
copyright 1998
the regents of the university of michigan
all rights reserved

permission is granted to use, copy, create derivative works 
and redistribute this software and such derivative works 
for any purpose, so long as the name of the university of 
michigan is not used in any advertising or publicity 
pertaining to the use or distribution of this software 
without specific, written prior authorization.  if the 
above copyright notice or any other identification of the 
university of michigan is included in any copy of any 
portion of this software, then the disclaimer below must 
also be included.

this software is provided as is, without representation 
from the university of michigan as to its fitness for any 
purpose, and without warranty by the university of 
michigan of any kind, either express or implied, including 
without limitation the implied warranties of 
merchantability and fitness for a particular purpose. the 
regents of the university of michigan shall not be liable 
for any damages, including special, indirect, incidental, or 
consequential damages, with respect to any claim arising 
out of or in connection with the use of the software, even 
if it has been or is hereafter advised of the possibility of 
such damages.
*/
