/*
 * Simple database routine wrappers
 *
 * Jim Rees, University of Michigan CITI, November 2000
 */

#ifdef __palmos__
/*
 * PalmOS version
 */
#pragma pack(2)

#if (__GNUC_MINOR__ > 7)
#include <PalmOS.h>
#else
#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>
#endif
#include <System/DateTime.h>
#include <System/MemoryMgr.h>
#include <System/DataMgr.h>
#include <System/Unix/sys_types.h>
#include <System/Unix/unix_stdio.h>
#include <System/Unix/unix_stdlib.h>
#include <string.h>

#include "field.h"
#include "tdb.h"

DmOpenRef tdb_DB;
char tdb_mode;
int tdb_nrecs;
UInt tdb_idx;
VoidHand tdb_hand;
ULong tdb_offset;

int tdb_open(char *name, char *crname, char *how)
{
    long creator;

    memcpy(&creator, name, 4);
    tdb_mode = how[0];

    tdb_DB = DmOpenDatabaseByTypeCreator('DATA', creator, (tdb_mode != 'r') ? dmModeReadWrite : dmModeReadOnly);

    if (!tdb_DB && tdb_mode == 'c') {
	if (DmCreateDatabase(0, crname, creator, 'DATA', false)) {
	    printf("can't create %s\n", crname);
	    return -1;
	}
	tdb_DB = DmOpenDatabaseByTypeCreator('DATA', creator, dmModeReadWrite);
    }

    if (!tdb_DB) {
	printf("can't open %s\n", name);
	return -1;
    }

    tdb_nrecs = DmNumRecords(tdb_DB);
    tdb_idx = 0;
    return 0;
}

int tdb_delete(char *name)
{
    LocalID id;
    UInt cardno;

    if (tdb_open(name, NULL, "r") < 0)
	return -1;
    if (DmOpenDatabaseInfo(tdb_DB, &id, NULL, NULL, &cardno, NULL))
	return -1;
    tdb_close();
    if (DmDeleteDatabase(cardno, id))
	return -1;
    return 0;
}

void *tdb_read()
{
    VoidHand h = NULL;
    Ptr p;

    while (tdb_idx < tdb_nrecs) {
	h = DmQueryRecord(tdb_DB, tdb_idx++);
	if (h)
	    break;
    }
    if (!h)
	return NULL;
    tdb_hand = h;
    p = MemHandleLock(h);
    return p;
}

void tdb_create(int len)
{
    tdb_idx = tdb_nrecs;
    tdb_hand = DmNewRecord(tdb_DB, &tdb_idx, len);
    tdb_offset = 0;
    tdb_nrecs++;
}

void tdb_write(void *p, int len)
{
    Ptr rp;

    if (tdb_hand) {
	rp = MemHandleLock(tdb_hand);
	DmWrite(rp, tdb_offset, p, len);
	tdb_offset += len;
    }
}

void tdb_release()
{
    if (tdb_hand) {
	MemHandleUnlock(tdb_hand);
	tdb_hand = NULL;
	if (tdb_mode != 'r')
	    DmReleaseRecord(tdb_DB, tdb_idx, 1);
    }
}

void tdb_close()
{
    tdb_release();
    DmCloseDatabase(tdb_DB);
    tdb_DB = NULL;
}
#else
/*
 * Unix version
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "tdb.h"

static FILE *dbf;

int tdb_open(char *name, char *crname, char *how)
{
    dbf = fopen(crname, how[0] != 'r' ? "w" : "r");
    return dbf ? 0 : -1;
}

int tdb_delete(char *name)
{
    printf("delete %s\n", name);
}

void *tdb_read()
{
    short len;
    static void *p;

    if (p) {
	free(p);
	p = NULL;
    }
    if (!dbf || fread(&len, sizeof len, 1, dbf) < 1)
	return NULL;
    p = (void *) malloc(len);
    fread(p, len, 1, dbf);
    return p;
}

void tdb_create(int len)
{
    short n = len;

    fwrite(&n, sizeof n, 1, dbf);
}

void tdb_write(void *p, int len)
{
    fwrite(p, len, 1, dbf);
}

void tdb_release()
{
}

void tdb_close()
{
    if (dbf) {
	fclose(dbf);
	dbf = NULL;
    }
}
#endif
