/*
 * Simple http (web) server.
 * Jim Rees University of Michigan
 * Mark Eichin <eichin@thok.org> The Herd Of Kittens
 * Original version April 1998
 */

#pragma pack(2)

#if (__GNUC_MINOR__ > 7)
#include <PalmOS.h>
#else
#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>
#endif
#include <System/DataMgr.h>
#include <System/NetMgr.h>
#include <System/SysEvtMgr.h>

#include <System/Unix/unix_stdio.h>
#include <System/Unix/unix_stdlib.h>
#include <System/Unix/unix_string.h>
#include <System/Unix/sys_types.h>
#include <System/Unix/sys_socket.h>
#include <System/Unix/netinet_in.h>
#include <System/Unix/arpa_inet.h>
#include <System/Unix/netdb.h>

#include "httpd.h"
#include "resid.h"
#include "lib/field.h"

int StartApplication();
void StopApplication();
void EventLoop();
void serve();
static Err OpenNet(void);
static Err CloseNet(void);
static Boolean httpd(EventPtr event);

FieldPtr printfield;
int netisopen, sfd = -1;

/* I'm not sure what these are doing here. */
Err errno, h_errno;
long AppNetTimeout;
NetHostInfoBufType AppHostInfo;
NetServInfoBufType AppServInfo;
UInt16 AppNetRefnum;

#if (__GNUC_MINOR__ > 7)
UInt32 PilotMain (UInt16 cmd, void *cmdPBP, UInt16  launchFlags)
#else
DWord PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
#endif
{
    int err;

    if (cmd == sysAppLaunchCmdNormalLaunch) {
	err = StartApplication();
	if (err)
	    return err;
	EventLoop();
	StopApplication();
	return 0;
    } else
	return sysErrParamErr;
}

int
StartApplication()
{
    FrmGotoForm(formID_httpd);

    return 0;
}

static Err
OpenNet(void)
{
    Err err;
    unsigned short ifErrs;
    struct sockaddr_in saddr;

    if (netisopen)
	return 0;

    err = SysLibFind("Net.lib", &AppNetRefnum);
    if (err) {
	prerr("SysLibFind", err);
	return err;
    }

    err = NetLibOpen(AppNetRefnum, &ifErrs);
    if ((err && err != netErrAlreadyOpen) || ifErrs) {
	prerr("NetLibOpen", err);
	return err;
    }

    err = 0;
    AppNetTimeout = SysTicksPerSecond() * 4;

    if ((sfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	err = errno;
	prerr("socket", err);
	goto done;
    }

    /* Bind it to the http port */

    bzero(&saddr, sizeof saddr);
    saddr.sin_family = AF_INET;
    saddr.sin_port = htons(PORT);
    saddr.sin_addr.s_addr = INADDR_ANY;

    if (bind(sfd, (struct sockaddr *) &saddr, sizeof saddr) < 0) {
	err = errno;
	close(sfd);
	prerr("bind", err);
	goto done;
    }

#ifdef BROKEN
    /* This seems to be unimplemented */

    opt = 1;
    if (setsockopt(sfd, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof opt) < 0)
	prerr("setsockopt SO_REUSEADDR", errno);
#endif

    /* Set the backlog */

    if (listen(sfd, 1) < 0) {
	err = errno;
	close(sfd);
	prerr("listen", err);
	goto done;
    }

    netisopen++;

 done:
    if (err)
	NetLibClose(AppNetRefnum, false);
    return err;
}

static Err
CloseNet(void)
{
    if (!netisopen)
	return 0;

    close(sfd);
    NetLibClose(AppNetRefnum, false);
    netisopen = 0;
    return 0;
}

void
StopApplication()
{
    CloseNet();
}

void
EventLoop()
{
    short err;
    int formID;
    FormPtr form;
    EventType event;

    do {
	EvtGetEvent(&event, evtWaitForever);

	if (SysHandleEvent(&event))
	    continue;
	if (MenuHandleEvent((void *)0, &event, &err))
	    continue;

	if (event.eType == frmLoadEvent) {
	    formID = event.data.frmLoad.formID;
	    form = FrmInitForm(formID);
	    FrmSetActiveForm(form);
	    switch (formID) {
	    case formID_httpd:
		FrmSetEventHandler(form, (FormEventHandlerPtr) httpd);
		break;
	    }
	}
	FrmDispatchEvent(&event);
    } while (event.eType != appStopEvent);
}

static Boolean
httpd(EventPtr event)
{
    FormPtr form;
    int handled = 0;

    switch (event->eType) {
    case frmOpenEvent:
	form = FrmGetActiveForm();
	printfield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_msg));
	FrmDrawForm(form);
	handled = 1;
	break;

    case ctlSelectEvent:  // A control button was pressed and released.
	if (event->data.ctlEnter.controlID == buttonID_go) {
	    serve();
	    handled = 1;
	} else if (event->data.ctlEnter.controlID == buttonID_stop) {
	    /* nothing special */
	    handled = 1;
	}
	break;

    case keyDownEvent:
	handled = scrollfield(printfield, event->data.keyDown.chr);
	break;

    case menuEvent:
	switch (event->data.menu.itemID) {
	case menuitemID_about:
	    FrmAlert(alertID_about);
	    break;
	}
	handled = 1;
	break;

    default:
	handled = 0;
	break;
    }
    return handled;
}

void
serve()
{
    int fd = -1, len, n;
    XFILE *f = NULL;
    char buf[96], cmd[12], host[20], url[80], *u;
    struct sockaddr_in saddr;
    NetMasterPBType ifinfo;

    /* Init net */

    if (OpenNet())
	return;

    /* See if we can figure out our ip address */

    NetLibMaster(AppNetRefnum, netMasterInterfaceInfo, &ifinfo, AppNetTimeout);
    NetLibAddrINToA(AppNetRefnum, ifinfo.param.interfaceInfo.ipAddr, host);
    printf("Listening on %s\n", host);

    while (1) {
	if (f) {
	    xclose(f);
	    f = NULL;
	}
	if (fd >= 0) {
	    close(fd);
	    fd = -1;
	}

	/* Accept connections */
	len = sizeof saddr;
	AppNetTimeout = SysTicksPerSecond() * 1;
	if ((fd = accept(sfd, (struct sockaddr *) &saddr, &len)) < 0) {
	    if (errno == netErrTimeout) {
		if (!EvtSysEventAvail(true))
		    continue;
	    } else
		prerr("accept", errno);
	    break;
	}
	AppNetTimeout = SysTicksPerSecond() * 4;

	/* I'm not sure this is necessary */
	EvtResetAutoOffTimer();

	/* I can't get inet_ntoa() to work */
	NetLibAddrINToA(AppNetRefnum, saddr.sin_addr.s_addr, host);

	f = xfdopen(fd, "r");

	if ((n = xgets(f, buf, sizeof buf)) < 0) {
	    printf("%s read %d errno %d\n", host, n, errno);
	    continue;
	}
	/* I've seen this from lynx. */
	if (n >= 2 && buf[n-2] == 0xd)
	    buf[n-2] = '\0';
	else
	    buf[n-1] = '\0';

	if (sscanf(buf, "%11s %79s", cmd, url) != 2) {
	    printf("%s bad http \"%s\"\n", host, buf);
	    continue;
	}

	/* Unescape the URL */
	u = url;
	while (*u) {
	    char n1, n2;
	    switch (*u) {
	    case '+':
		*u = ' ';
		break;
	    case '%':
		n1 = *(u+1); n2 = *(u+2);
		n1 = (n1 > 96 && n1 < 123) ? (n1-32) : n1;
		n2 = (n2 > 96 && n2 < 123) ? (n2-32) : n2;
		if (n1 > 47 && n1 < 57) n1 -= 48;
		if (n1 > 64 && n1 < 71) n1 -= 55;
		if (n1 > 15) break;
		if (n2 > 47 && n2 < 57) n2 -= 48;
		if (n2 > 64 && n2 < 71) n2 -= 55;
		if (n2 > 15) break;
	    
		*u = 16 * n1 + n2;

		strcpy(u+1, u+3);
		break;
	    }
	    u++;
	}

	if (strcmp(cmd, "GET")) {
	    printf("%s bad http cmd \"%s\"\n", host, cmd);
	    continue;
	}

	/* Read and discard remaining http headers */
	while ((n = xgets(f, buf, sizeof buf)) >= 0 && buf[0] > ' ')
	    ;
	xflush(f);

	printf("%s %s\n", host, url);

	if (!strcmp(url, "/"))
	    send_root_index(f);
	else if (!strncmp(url, "/addr/", strlen("/addr/")))
	    send_addr_file(f, url);
	else if (!strncmp(url, "/date/", strlen("/date/")))
	    send_date_file(f, url);
	else if (!strncmp(url, "/memo/", 6))
	    send_memo_file(f, url + 6, 0);
	else if (!strncmp(url, "/doc/", strlen("/doc/")))
	    send_doc_file(f, url);
	else if (!strcmp(url, "/debug/"))
	    send_debug_index(f);
	else if (!strcmp(url, "/about/"))
	    send_about_index(f);
	else if (!strcmp(url, "/httpd-icon.gif"))
	    send_icon(f, httpd_icon, httpd_icon_size);
	else
	    send_404(f, url);
    }

    printf("stopped\n");
}

char html0[] = "HTTP/1.0 200 OK\nMIME-version: 1.0\nContent-type: %s\n\n";

static char html_root[] =
    "<title>Pilot Server</title>\n"
    "<h1>Pilot Server</h1>\n"
    "<img width=32 height=20 src=\"httpd-icon.gif\"><p>\n"
    "Contents:<ul>\n"
    "<li> <a href=\"addr/\">Address Book</a>\n"
    "<li> <a href=\"date/\">Datebook</a>\n"
    "<li> <a href=\"memo/\">Memos</a>\n"
    "<li> <a href=\"doc/\">Doc Files</a>\n"
    "<li> <a href=\"debug/\">Debug</a>\n"
    "<li> <a href=\"about/\">About httpd</a>\n"
    "</ul>\n"
;

char html_go_home[] =
    "<a href=\"/\"><img width=32 height=20 align=middle src=\"/httpd-icon.gif\"></a>\n"
    "Home.<p>\n"
;

int send_root_index(XFILE *f)
{
    /* If there is a memo called "pilot.html" we send that */
    if (!send_memo_file(f, "pilot.html", 1))
	return 0;

    wrstr(f, html0, "text/html");
    wrstr(f, html_root);
    return 0;
}

static char about_text[] =
    "<h1>About httpd</h1>\n"
    "Written by\n"
    "<a href=\"http://www.citi.umich.edu/u/rees/\">Jim Rees</a>,\n"
    "<a href=\"http://www.thok.org/~eichin\">Mark Eichin</a>,\n"
    "and <a href=\"mailto:misty@drrobin.yi.org\">Robin Kirkman</a>.\n"
    "Original version April 1998. Current version and other goodies at Jim's\n"
    "<a href=\"http://www.citi.umich.edu/u/rees/pilot/\">PalmPilot Stuff</a>\n"
    "page.<p>\n"
;

int send_about_index(XFILE *f)
{
    wrstr(f, html0, "text/html");
    xputs(f, about_text);
    wrstr(f, html_go_home);
    return 0;
}

int send_icon(XFILE *f, char *icon, int size)
{
    wrstr(f, html0, "image/gif");
    xwrite(f, icon, size);
    return 0;
}

char html_notfound[] =
    "HTTP/1.0 404 Not Found\n"
    "Content-type: text/html\n\n"
    "<html><head><title>404 File Not Found</title></head>\n"
    "<body><h1>File Not Found</h1>\n"
    "%s not found.</body></html>\n"
;

int send_404(XFILE *f, char *url)
{
    return wrstr(f, html_notfound, url);
}

int wrstr(XFILE *f, const char *fmt, ...)
{
    va_list args;
    int i;
    char buf[400];

    va_start(args, fmt);
    i = vsprintf(buf, fmt, args);
    va_end(args);
    return xputs(f, buf);
}

void prerr(char *s, int err)
{
    printf("%s: %s\n", s, geterr(err));
}

/*
copyright 1998
the regents of the university of michigan
all rights reserved

permission is granted to use, copy, create derivative works
and redistribute this software and such derivative works
for any purpose, so long as the name of the university of
michigan is not used in any advertising or publicity
pertaining to the use or distribution of this software
without specific, written prior authorization.  if the
above copyright notice or any other identification of the
university of michigan is included in any copy of any
portion of this software, then the disclaimer below must
also be included.

this software is provided as is, without representation
from the university of michigan as to its fitness for any
purpose, and without warranty by the university of
michigan of any kind, either express or implied, including
without limitation the implied warranties of
merchantability and fitness for a particular purpose. the
regents of the university of michigan shall not be liable
for any damages, including special, indirect, incidental, or
consequential damages, with respect to any claim arising
out of or in connection with the use of the software, even
if it has been or is hereafter advised of the possibility of
such damages.
*/
