/*
 * Simple http (web) server.
 * Jim Rees University of Michigan April 1998
 * Doc page by Mark Eichin <eichin@thok.org> The Herd Of Kittens
 * July 1998
 */

#pragma pack(2)

#if (__GNUC_MINOR__ > 7)
#include <PalmOS.h>
#else
#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>
#endif
#include <System/DataMgr.h>

#include <System/Unix/unix_stdio.h>
#include <System/Unix/unix_stdlib.h>
#include <System/Unix/unix_string.h>

#include "httpd.h"
#include "lib/field.h"

char html_doc_intro[] = 
"<title>DOC Files</title>\n"
"<h1>DOC Files</h1>\n"
"DOC is a format generated by tools like makedoc7, screwdriver.com, and\n"
"the TopGun Wingman gateway.  DOC files can be read on the pilot by\n"
"AportisDOC, J-DOC, TealDOC, QED, and other programs.\n"
"(All of these use type REAd creator TEXt.)\n"
"<br>\n";

char html_doc_sorry[] =
"<h2>No DOC files found</h2>\n"
"Sorry, there don't appear to be any DOC files installed here.\n"
"To scan all databases, check the <a href=\"/debug/\">debug page</a>.<p>\n";

int send_doc_index(XFILE *f)
{
    Err st;
    DmSearchStateType dmst;
    UInt16 cardno = 0;
    LocalID thisdbid;

    wrstr(f, html0, "text/html");
    wrstr(f, html_doc_intro);

    st = DmGetNextDatabaseByTypeCreator(1, /* Boolean newSearch */
					&dmst, /* stateInfoP */
					'TEXt',	/* UInt32 type */
					'REAd',	/* UInt32 creator */
					0, /* Boolean onlyLatestVers */
			 		&cardno, /* UIntPtr cardNoP */
					&thisdbid /* LocalID* dbIDP */);

    if (st) {
	wrstr(f, html_doc_sorry);
	wrstr(f, html_go_home);
	return 0;
    }

    wrstr(f, "<h1>DOC files available</h1>\n<ul>\n");
    while(!st) {
	/* for now, snarf debug code */
	char namebuf[32];

	namebuf[0] = 0;
	st = DmDatabaseInfo(cardno, thisdbid, namebuf,
			    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
	if (st) {
	    wrstr(f, "<li> dmdbi(0x%x) failed (0x%x)\n", 
		  (int)thisdbid, (int)st);
	    continue;
	}
	wrstr(f, "<li><a href=\"/doc/%lx\">%s</a>\n", thisdbid, namebuf);
	/* end of snarf */
	st = DmGetNextDatabaseByTypeCreator(0, /* Boolean newSearch */
					    &dmst, /*  stateInfoP */
					    'TEXt', /* UInt32 type */
					    'REAd', /* UInt32 creator */
					    0, /* Boolean onlyLatestVers */
					    &cardno, /* UIntPtr cardNoP */
					    &thisdbid /* LocalID* dbIDP */);
	/* loop back around with new sample data */
    }
    wrstr(f, "</ul><p>\n");
    wrstr(f, html_go_home);
    return 0;
}



typedef struct DocHeader {
    unsigned char unknown0;
    unsigned char compressed;
    UInt16 x1;
    UInt32 truelength;
    UInt16 nrecs;
    UInt16 recsize;
    UInt32 pad;
} DocHeader, *DocHeaderP;

char *doc_decomp_2(unsigned char *inbuf, unsigned char *outbuf, int inbytes);
void put_some_text(XFILE *f, char *bp, int n);

int send_doc_file(XFILE *f, char* url)
{
    LocalID dbid;
    Err st;
    char namebuf[32];
    DmOpenRef docref;
    int i;
    void *hh;
    DocHeaderP dhp;
    int compressflag, ndocrecs, recsize;
    UInt32 truelength;
    unsigned char *outbuf, *outptr;

    if (!strcmp(url, "/doc/"))
	return send_doc_index(f);

    if (sscanf(url, "/doc/%lx", &dbid) != 1) {
	send_404(f, url);
	return 0;
    }

    wrstr(f, html0, "text/html");
    namebuf[0] = 0;
    st = DmDatabaseInfo(0, dbid, namebuf,
			NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
    if (st) {
	printf("DmDatabaseInfo(0x%x) failed (0x%x)\n", (int)dbid, (int)st);
	return 0;
    }
    wrstr(f, "<title>%s</title>\n<h1>%s</h1>\n", namebuf, namebuf);
#ifdef DOCDEBUG
    wrstr(f, "dbid %lx:\n<br>\n", dbid);
    st = DmDatabaseSize(0, /* UInt cardNo */
			dbid, /* LocalID dbID */
			&numrecs, /* ULongPtr numRecordsP */
			&numbytes, /* ULongPtr totalBytesP */
			&databytes /* ULongPtr dataBytesP */);
    if (st) {
	wrstr(f, "dmdbsz(0x%x) failed (0x%x)\n", (int)dbid, (int)st);
	return 0;
    }
    wrstr(f, "DOC %ld recs, %ld/%ld bytes<p>\n", 
	  numrecs, databytes, numbytes);
#endif

    docref = DmOpenDatabase(0, /* UInt cardNo */
			    dbid, /* LocalID dbID */
			    dmModeReadOnly /* UInt mode*/);

    if (!docref) {
	wrstr(f, "open failed: docref NULL\n");
	wrstr(f, html_go_home);
	return 0;
    }

    hh = DmQueryRecord(docref, /* DmOpenRef dbP */
		       0 /* UInt index */);
    dhp = (DocHeaderP)MemHandleLock(hh);
    if (!dhp) {
	wrstr(f, "DmQueryRecord: no handle\n");
	wrstr(f, html_go_home);
	return 0;
    } else {
	/* later, just ref don't copy */
	compressflag = dhp->compressed;
	truelength = dhp->truelength;
	ndocrecs = dhp->nrecs;
	recsize = dhp->recsize;
    }
    MemHandleUnlock(hh);

    switch (compressflag) {
    case 1:
	for (i = 1; i <= ndocrecs; i++) {
	    char *chp;
	    hh = DmQueryRecord(docref, /* DmOpenRef dbP */
			       i /* UInt index */);
	    chp = (char*)MemHandleLock(hh);
	    if (chp)
		put_some_text(f, chp, MemPtrSize(chp));
	    MemHandleUnlock(hh);
	}
	break;
    case 2:
	outptr = outbuf = (unsigned char*)MemPtrNew(recsize);
	if (!outbuf) { 
	    wrstr(f, "memptrnew failed\n");
	    break; 
	}
	for (i = 1; i <= ndocrecs; i++) {
	    unsigned char *chp;
	    hh = DmQueryRecord(docref, /* DmOpenRef dbP */
			       i /* UInt index */);
	    chp = (unsigned char*)MemHandleLock(hh);
	    
	    if (!chp) {
		wrstr(f, "%d: no handle\n", i);
	    } else {
		outptr = doc_decomp_2(chp, outbuf, MemPtrSize(chp));
		put_some_text(f, outbuf, outptr - outbuf);
	    }
	    MemHandleUnlock(hh);
	}
	MemPtrFree(outbuf);
	break;

    default:
	wrstr(f, "compress(%d) not yet handled\n", compressflag);
	break;
    }

    st = DmCloseDatabase(docref /* DmOpenRef dbP */);
    if (st) {
	wrstr(f, "<p> db close failed (0x%x)\n", (int)st);
    }

    xputs(f, "<p>");
    wrstr(f, html_go_home);
    return 0;
}

void put_some_text(XFILE *f, char *bp, int n)
{
    char c;

    while (n--) {
	c = *bp++;
	if (c == '\n')
	    xputs(f, "<p>");
	if (f->p >= f->base + XBUFSIZ - 1)
	    xflush(f);
	*(f->p)++ = c;
    }
}

char *doc_decomp_2(unsigned char *inbuf, unsigned char *outbuf, int inbytes)
{
    unsigned char *outptr = outbuf;
    unsigned int m, n, c, c2;

    while(inbytes > 0) {
	c = *(inbuf++); inbytes--;
	if (c && c < 9) {
	    /* copy 1..9 */
	    memcpy(outptr, inbuf, c);
	    inbuf += c; outptr += c; inbytes -= c;
	} else if (c < 128) {
	    /* self */
	    *(outptr++) = c;
	} else if ((c & 0300) == 0300) {
	    /* space lead */
	    *(outptr++) = ' ';
	    *(outptr++) = c & 0177;
	} else if ((c & 0300) == 0200) {
	    /* the tricky copying one */
	    c2 = *(inbuf++); inbytes--;
	    /* do this the clever way... */
	    m = ((c & 077) << 5)
		+ ((c2 & 0377) >> 3);
	    n = (c2 & 07) + 3;

	    if (n <= m) {
		memcpy(outptr, outptr-m, n);
		outptr += n;
	    } else {
		while (n >= m) {
		    memcpy(outptr, outptr-m, m);
		    outptr += m; n -= m; 
		}
		/* now n < m but n is still a distance... */
		if(n > 0) {
		    memcpy(outptr, outptr-m, n);
		    outptr += n;
		}
	    }
	} else {
	    static char MSG[] = "Invalid Byte in Decompression\n";
	    memcpy(outptr, MSG, (n = strlen(MSG)));
	    outptr += n;
	    break;
	}
    }
    return outptr;
}
