/*
 *  Eringson v0.4.2
 *  Copyright (C) 1999, Martin von Weissenberg
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <System/SysAll.h>
#include <UI/UIAll.h>

#include <Pilot.h>
#include <SerialMgr.h>
#include <ctype.h>
#include <irlib.h>

#include "eringson.h"


/*****************
 * Various defines
 */
#define king 1
#define the (
#define lives )

#define noSongSelected 0xffff

#define BUFSIZE 2048
#define BAUDRATE 38400L

// mode constants
#define MODE_TEXT    1
#define MODE_GRAPHIC 2

// button commands
#define BUTTON_HIDE  1
#define BUTTON_SHOW  2
#define BUTTON_AUTO  3

// The SH888 has memory allocated for 40 notes
#define SONGLENGTH 40

/*****************
 * Debugging aids
 */
#define PRINTF(format, args...) { StrPrintF(errTemp, format, ## args); FrmCustomAlert(frmErrorAlert, errTemp, NULL, NULL); }
#define MessageDisplay(format, args...) { StrPrintF(errTemp, format, ## args); FrmCustomAlert(frmErrorAlert, errTemp, NULL, NULL); }


/*****************
 * Globals
 */


static Word curForm = frmMain;

static UInt comRefNum;

// various static buffers
static char comBuf[BUFSIZE],
  errTemp[BUFSIZE],
  curSong[SONGLENGTH];

// curSongLen refers to the first free note position in curSong[],
// i.e. the terminating null char.
static short curSongLen = 0;
// curSongLeft is the index of the leftmost note on the screen
static short curSongLeft = 0;

static short uiMode = MODE_TEXT;

static WinHandle winH[BITMAPS];  // contains the note bitmaps


static Word
ConvertSongPtoI (CharPtr srcP, CharPtr dstP,
                 Word dstLen, WordPtr insPtP)
     // a function converting from phone song format to internal
     // format.  returns the number of notes converted, i.e. the size
     // of dstP.
{
  // the internal format is as follows: each byte is one note.
  // bit values:
  // 0x0F note value where 1 = c up to 14 = +h.  15 = pause
  // 0x10 set if flat note
  // 0x20 set if sharp note
  // 0x40 set if long note

  Boolean oneDown = false,
    flat = false, sharp = false, octave = false,
    longNote = false;
  Word notesDone = 0,
    ip = 0;
  CharPtr cP,
    ipP = NULL;
  char cur = '\0';
  
  // empty the buffer
  for (cP=dstP; cP<(dstP+dstLen); cP++) *cP=0;

  cP=srcP;
  if (insPtP!=NULL) ipP=srcP+(*insPtP);

  while ((*cP != '\0') && (notesDone < dstLen))
    {
      if ((cP==ipP)&&(insPtP!=NULL)) *insPtP=notesDone;
      
      switch (*cP) {
        
      case 'C':
      case 'D':
      case 'E':
      case 'F':
      case 'G':
        cur = (*cP) - 'B';
        longNote=true;
        oneDown=true;
        break;
        
      case 'c':
      case 'd':
      case 'e':
      case 'f':
      case 'g':
        cur = (*cP) - 'b';
        oneDown=true;
        break;
        
      case 'A':
        longNote=true;
      case 'a':
        cur = 6;
        oneDown=true;
        break;
        
      case 'H':
        longNote=true;
      case 'h':
        cur=7;
        oneDown=true;
        break;
        
      case 'p':
        cur=15;
        oneDown=true;
        break;
        
      case '+':
        octave=true;
        break;
        
      case '#':
        sharp=true;
        break;
        
      case 'b':
        flat=true;
        break;
      }
      
    //    PRINTF("peeking at '%c': cur=%x", *cP, (int)cur);
      
      *cP++;
      if (oneDown)
        {
          if ((flat) && ((cur==2)||(cur==3)||(cur==5)
                         ||(cur==6)||(cur==7)))
            cur |= 0x10;

          if ((sharp) && ((cur==1)||(cur==2)||(cur==4)
                          ||(cur==5)||(cur==6)))
            cur |= 0x20;
          
          if ((octave) && ((cur&0x0F)>=1) && ((cur&0x0F)<=7)) cur += 7;

          if (longNote) cur |= 0x40;
          
          *dstP++ = cur;
          
          //      PRINTF("got %x", (int)cur);
          
          oneDown = false;
          flat = sharp = false;
          octave = longNote = false;
          cur = 0;
          //      if (*cP) notesDone++;
          notesDone++;
        }
    }

  return notesDone;
}




static Word
ConvertSongItoP (CharPtr srcP, CharPtr dstP,
                 Word dstLen, WordPtr insPtP)
     // a function converting from internal song format to phone
     // format.  returns the number of characters converted, i.e. the
     // size of dstP.  0 means overflow
{
  CharPtr cP, ipP = NULL;
  char c, dump[4], notes[] = "cdefgah";
  Word di=0,
    numChars=0,
    dumpLen=0;

  // empty the buffer
  for (cP=dstP; cP<(dstP+dstLen); cP++) *cP=0;

  if (insPtP!=NULL) ipP=srcP+(*insPtP);

  for (cP=srcP; *cP != '\0'; cP++)
    {
      if ((cP==ipP)&&(insPtP!=NULL)) *insPtP=numChars;

      c=*cP;
      for (di=0; di<4; di++) dump[di]='\0';
      di=0;
      
      if ((c&0x0F) == 15) // pause
        {
          dump[di] = 'p';
        }
      else
        {
          
          if (((c&0x0F)>=8) && ((c&0x0F)<=14)) // octavo
            dump[di++]='+';
      
          if (c&0x10) // flat
            {
              dump[di++]='b';
            }
          else if (c&0x20) // sharp
            {
              dump[di++]='#';
            }

          // prefixes have now been handled, so forget about them
          if (((c&0x0F)>=8) && ((c&0x0F)<=14)) // octavo
            c -= 7;
          
          if (c&0x40) // long note
            {
              dump[di] = notes[(c&0x0F)-1] - 32;
            }
          else
            {
              dump[di] = notes[(c&0x0F)-1];
            }      
        }

      dumpLen=StrLen(dump);
      numChars+=dumpLen;

      if (numChars<=dstLen)
        {
          StrCat(dstP, dump);
          dstP += dumpLen;
        }
      else // buffer overflow, return error
        {
          return 0;
        }
    }

  return numChars;
}




static Err
SendCommand(CharPtr cmdP)
     // very simple procedure for sending data to the serial port
{
  Err err = 0;
  UInt sent = 0;

  SerReceiveFlush(comRefNum, 0);
  sent = SerSend(comRefNum, cmdP, StrLen(cmdP), &err);

  return err;
}




static long
GetLine(Long timeout, CharPtr respBufP, UInt respBufSize, int level)
     // the procedure is used to fetch data from the serial port.  if
     // no data bytes have arrived within an initial timeout, give up.
     // otherwise fetch any additional data in a tighter loop until
     // the buffer is used up or data stops arriving.  this is because
     // the phone's replies are sent in bursts.
{
  ULong oldNumBytes=0,
    numBytes = 0;
  UInt i;
  Err err;
  Char* cP;

  // empty the buffer
  for (cP=respBufP; cP<(respBufP+respBufSize); cP++) *cP=0;

  // wait for the first reply to arrive
  // empirically determined to happen within 2 sec or so
  err = SerReceiveWait(comRefNum, respBufSize, timeout);
  if (err)
    {
      if (err==serErrTimeOut)
        {
          SerClearErr(comRefNum);
        }
      else
        {
          PRINTF("GetLine A: err %x", (int)err);
          SerClearErr(comRefNum);
          return -err;
        }
    }

  // paranoia check
  SerReceiveCheck(comRefNum, &numBytes);
  if (numBytes >= respBufSize) numBytes = respBufSize-1;
  
  // now get the bytes from the receive buffer
  numBytes = SerReceive(comRefNum, respBufP, numBytes, 0, &err);
  if (err)
    {
      if (err == serErrLineErr)
        {
          err = SerClearErr(comRefNum);
        }
    }
  else
    {
      respBufP[numBytes] = '\0';
    }
  
  // repeat several times with a timeout of 1/4 sec until there's no
  // more data to be found.  this is very inefficient.  fixit
  if ((level>0) && (numBytes > 0) && (numBytes < respBufSize-1)) 
    numBytes += GetLine(SysTicksPerSecond()/4, (respBufP+numBytes),
                        (respBufSize-numBytes), level-1);
  
  return numBytes;
}




static int
ATDialog(Long timeout, CharPtr bufP, UInt bufSize)
     // this function sends data to the serial port and if there's a
     // reply within the timeout range, it returns the received data.
     // note that bufP is reused for the reply
{
  long res;
  int att;

  res = SendCommand(bufP);
  if (res) return res;
  
  res=0; att=0;
  while ((res==0) && (att<3))
    {
      att++;
      res = GetLine(timeout, bufP, bufSize, 7);
      if (res<0)
        return -res;
      else if (res>0) 
        return res; 
    }

  return 0;
}





static Err
ComsClose(void)
     // shut down the serial port.  forget any errors.
{
  DWord err = 0;

  err = SerSetReceiveBuffer (comRefNum, NULL, 0);
  if (err == serErrLineErr)  // small matter
    {
      err = SerClearErr(comRefNum);
    }
  err = 0;

  err = SerClose(comRefNum);
  comRefNum = NULL;

  return err;
}




static Err
ComsOpen(long serSpeed)
     // open the serial port with all due respect and error checking
{
  DWord err = 0;
  SerSettingsType serSettings;
  DWord value = 0;
  Word valueLen = sizeof(value);

  err = SysLibFind("Serial Library", &comRefNum);
  // other possible libs: Serial, IrSerial, SerIrComm 

  if (err)
    {
      PRINTF("Open: Error %x on SysLibFind.", (int)err);
      comRefNum = 0;
      return err;
    }

  // open the port.  if it's already open, restore parity and give up
  err = SerOpen(comRefNum, 0, serSpeed);
  if (err)
    {
      if (err==serErrAlreadyOpen)
        SerClose(comRefNum);
      
      PRINTF("Open: Error %x on SerOpen.", (int)err);
      return err;
    }

  // the 256 byte buffer provided by PalmOS doesn't suffice far.
  // BUFSIZE should be 1024 bytes or more for this application
  err = SerSetReceiveBuffer (comRefNum, &comBuf, BUFSIZE);
  if (err == serErrLineErr) // small matter
    err = SerClearErr(comRefNum);
  return 0;
  
 OPEN_FAILURE:
  ComsClose();
  return err;
}




static Err
InstallMelody(CharPtr melody, Short melLen)
     // install the given melody on the phone
{
  Err comErr = 0;
  Char data[BUFSIZE];
  Char *buf;
  ULong res;
  int attempts, yo;
  Boolean success;

  // make sure we have data to send
  if ((melody==NULL) || (melLen==0) || (StrLen(melody)==0))
    {
      MessageDisplay("Hey, there's no melody to send!");
      return 1;
    }


  // open the port
  comErr = ComsOpen(BAUDRATE);
  if (comErr != 0)
    {
      MessageDisplay("Error %x, could not open the serial port.", comErr);
      return comErr;
    }

  // probe the phone and check that the serial channel is working
  data[0] = '\0';
  StrCopy(data, "\rat\r");
  yo = ATDialog(300, (CharPtr)&data, BUFSIZE);

  if (yo!=0) // have we got data?
    {
      if (StrStr(data, "OK")) // have we got the right data?
        {
          SerReceiveFlush(comRefNum, 100);
        }
      else // not OK
        {
          yo=0; // force the if (yo==0) clause below
        }
    }

  if (yo==0)
    {
      MessageDisplay("Hey, the phone doesn't react to AT commands?");
      ComsClose();
      return 1;
    }

  // build the melody AT command and send it
  data[0] = '\0';
  StrCopy(data, "at*esom=1,\"");
  // fixit -- must weed out control chars from srcP first
  StrCat(data, melody);
  StrCat(data, "\"\r");


  //  PRINTF("Sending '%s' [%d]", data, StrLen(data));
  yo = ATDialog(300, (CharPtr)&data, BUFSIZE);
  if (yo!=0) // have we got data?
    {
      if (StrStr(data, "OK")) // have we got the right data?
        {
          
          // play back the melody
          data[0] = '\0';
          StrCopy(data, "\rat*erip=6,31\r");
          yo = ATDialog(300, (CharPtr)&data, BUFSIZE);
          if (yo!=0) // have we got data?
            {
              if (StrStr(data, "OK")) // have we got the right data?
                {
                  SerReceiveFlush(comRefNum, 100);
                }
              else
                {
                  yo=0; // force the if (yo==0) clause below
                }
            }
        }
      else if (StrStr(data, "ERROR"))
        {
          MessageDisplay("Failure! The phone rejected your melody!");
        }
      else
        {
          yo=0; // force the if (yo==0) clause below
        }
    }


  if (yo==0)
    {
      MessageDisplay("Phone didn't reply to the melody install and playback command.");
      ComsClose();
      return 1;
    }
  
  // clean up
  comErr = ComsClose();
}




static Err
FetchMelody(CharPtr melody, Short melLen)
     // fetch the current melody from the phone
{
  Err comErr = 0;
  Char data[BUFSIZE];
  Char *start, *stop, *cP;
  Handle hdl, oldhdl;
  int i, yo;

  for (cP=melody; cP<(melody+melLen); cP++) *cP='\0';

  // open the port
  comErr = ComsOpen(BAUDRATE);
  if (comErr != 0)
    {
      MessageDisplay("Error %x, could not open the serial port.", comErr);
      return comErr;
    }

  // probe the phone
  data[0] = '\0';
  StrCopy(data, "\rat\r");
  yo = ATDialog(300, (CharPtr)&data, BUFSIZE);
  if (yo!=0) // have we got data?
    {
      if (StrStr(data, "OK")) // have we got the right data?
        {
          SerReceiveFlush(comRefNum, 100);
        }
      else
        {
          yo=0; // force the if (yo==0) clause below
        }
    }

  if (yo==0)
    {
      PRINTF("Hey, the phone doesn't react to AT commands?");
      ComsClose();
      return 1;
    }

  // build the AT command and send it
  StrCopy(data, "at*esom?\r");
  yo = ATDialog(300, (CharPtr)&data, BUFSIZE);
  if (yo!=0) // have we got data?
    {
      if (StrStr(data, "OK")) // have we got the right data?
        {
          MessageDisplay("Success! I got the melody!");
          
          // fetch the payload
          start = StrStr(data, "1,\"");
          start += 3;
          stop = StrStr(start, "\"");
          
          *stop = '\0';
          
          // data now in *start.  put it into the edit field
          
          StrCat(melody, start);
          
        }
      else if (StrStr(data, "ERROR"))
        {
          MessageDisplay("Yuk! There's a bug in this program, or some data got garbled!");
        }
      else
        {
          yo=0; // force the if (yo==0) clause below
        }
    }

  if (yo==0)
    {
      PRINTF("Hey, the phone didn't answer the melody query?");
      ComsClose();
      return 1;
    }

  // clean up
  ComsClose();
}




static void
PlayMelody()
{
  Err comErr = 0;
  Char data[BUFSIZE];
  int yo;

  // open the IR port
  comErr = ComsOpen(BAUDRATE);
  if (comErr != 0) 
    {
      MessageDisplay("Error %x, could not open the serial port.", comErr);
      return;
    }

  // no probing needed, just order the phone to play back the current
  // melody
  data[0] = '\0';
  StrCopy(data, "\rat*erip=6,31\r");
  yo = ATDialog(300, (CharPtr)&data, BUFSIZE);
  if (yo!=0) // have we got data?
    {
      if (StrStr(data, "OK")) // have we got the right data?
        {
          SerReceiveFlush(comRefNum, 100);
        }
      else
        {
          yo=0; // force the if (yo==0) clause below
        }
    }

  if (yo==0)
    {
      PRINTF("Hey, the phone doesn't react to AT commands?");
    }

  // clean up
  ComsClose();
}



static void
PasteBitmap(int winIndex, Short x, Short y, ScrOperation mode)
{
  RectangleType r;

  ErrFatalDisplayIf(((winIndex>=BITMAPS)||(winIndex<0)),
                    "Unsupported bitmap");
  ErrFatalDisplayIf((winH[winIndex] == NULL),
                    "Empty bitmap handle");

  MemMove(&r, &(winH[winIndex]->windowBounds), sizeof(r));
  WinCopyRectangle(winH[winIndex], 0, &r, x, y, mode);
}





static void
DrawNote(char note, char prevNote,
         short staffLeft, short staffTop,
         short position, Boolean clearArea)
     // this procedure contains lots of hardcoded values
{
  short noteBmp=0,
    yPos=0,
    xPos=0,
    tone=(note&0x0F),
    i;
  RectangleType r;

  // find out what kind of note we should draw and where we should
  // draw it
  if (tone==0) // end
    {
      return;
    }
  else if (tone==15) // pause
    {
      noteBmp=restBitmap;
      yPos=38;
    }
  else
    {
      if ((tone>=1)&&(tone<=6)) //  handle up
        {
          if ((note&0x40)!=0) // long note
            {
              noteBmp=lnUpBitmap;
            }
          else // short note
            {
              noteBmp=shUpBitmap;
            }
          yPos = staffTop + 27 - (tone*4);
        }
      else // handle down
        {
          if ((note&0x40)!=0) // long note
            {
              noteBmp=lnDnBitmap;
            }
          else // short note
            {
              noteBmp=shDnBitmap;
            }
          yPos = staffTop + 50 - (tone*4);
        }
    }
  xPos = staffLeft + 42 + 26*position;

  // clear and redraw part of the staff
  if (clearArea)
    {
      RctSetRectangle(&r, xPos-10, staffTop-14, 24, 90);
      WinEraseRectangle(&r, 0);
      for (i=(staffTop+12); i<=(staffTop+44); i+=8)
        WinDrawLine(xPos-10, i, xPos+14, i);
    }

  PasteBitmap(noteBmp, xPos, yPos, scrOR);
  
  if (tone==1)
    {
      WinDrawLine(xPos, staffTop+52, xPos+12, staffTop+52);
    }
  else if ((tone>=13)&& (tone<=14))
    {
      WinDrawLine(xPos, staffTop+4, xPos+12, staffTop+4);
    }


  if ((note&0x10)) // flat note
    {
      switch (tone)
        {
        case 1:
        case 4:
        case 8:
        case 11:
        case 15:
          break;
        default:
          xPos -= 10;
          yPos = staffTop + 38 - (tone*4);
          PasteBitmap(flatBitmap, xPos, yPos, scrOR);
          break;
        }
    }
  else if ((note&0x20)) // sharp note
    {
      switch (tone)
        {
        case 3:
        case 7:
        case 10:
        case 13:
        case 15:
          break;
        default:
          xPos -= 8;
          yPos = staffTop + 42 - (tone*4);
          PasteBitmap(sharpBitmap, xPos, yPos, scrOR);
          break;
        }
    }
  else if ((tone!=15)&&(tone==(prevNote&0x0F))
             &&(prevNote&0x30)) // normalize
    {
      xPos -= 8;
      yPos = staffTop + 42 - (tone*4);
      PasteBitmap(normalBitmap, xPos, yPos, scrOR);
    }
}




static void
DrawScrollButtons(short visL, short visR)
  // draw the scroll buttons
{
  Word oi;
  ControlPtr btnP = NULL;
  FormPtr frmP = FrmGetActiveForm();
  Boolean vis = false;

  vis = (visL == BUTTON_AUTO)
    ? (curSongLeft>0) : (visL==BUTTON_SHOW);
  btnP = FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP, btnMainL));
  if (vis)
    CtlShowControl(btnP);
  else
    CtlHideControl(btnP);

  vis = (visR == BUTTON_AUTO)
    ? (curSongLeft<(curSongLen-1)) : (visR==BUTTON_SHOW);
  btnP = FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP, btnMainR));
  if (vis)
    CtlShowControl(btnP);
  else
    CtlHideControl(btnP);

  return;
}




static void
DrawStaff(short first, Boolean clearArea)
{
  short y=32, x=5, i, j, k;
  char prevNote, noteNum[16];
  RectangleType r;
  FormPtr frmP = FrmGetActiveForm();

  // get gadget bounds and erase the area
  FrmGetObjectBounds(frmP, FrmGetObjectIndex(frmP, gdgMainGraphic), &r);
  x = r.topLeft.x;
  y = r.topLeft.y+10;
  if (clearArea) WinEraseRectangle(&r, 0);

  // draw the staff
  PasteBitmap(gBitmap, x+2, y, scrOR);
  for (i=(y+12); i<=(y+44); i+=8)
    WinDrawLine(x+1, i, x+r.extent.x-1, i);

  // make sure that there's at least one note on screen
  i=0;
  while ((i<SONGLENGTH) && (curSong[i]!=0)) i++;
  if (first>(i-1)) first=(i-1);
  if (first<0) first=0;
  curSongLeft=first;

  DrawScrollButtons(BUTTON_AUTO, BUTTON_AUTO);

  // draw four notes 
  i=first; j=0;
  while ((curSong[i]!=0) && (j<4)
         && (i<curSongLen) && (i<SONGLENGTH))
    {
      prevNote='\0';
      k=(i-1);
      while (k>=first)
        {
          if ((curSong[k] & 0x0F)==(curSong[i] & 0x0F))
            {
              if (curSong[k] & 0x30)
                {
                  prevNote=(curSong[i] | 0x20);  // force normal
                  break;
                }
              else
                {
                  prevNote=0;
                  break;
                }
            }
          k--;
        }
      DrawNote(curSong[i], prevNote, x, y, j, false);
      i++;
      j++;
    }  

  // print the number of the first note shown
  FntSetFont(stdFont);
  StrIToA((CharPtr)&noteNum, (first+1));
  i = StrLen(noteNum);
  WinDrawChars((CharPtr)&noteNum, i,
       (x+16)-(FntCharsWidth((CharPtr)&noteNum, i)/2),
       (y-FntBaseLine()));
}




void
InsertNote(Char what, SWord where)
{
  int i;

  if (curSongLen>=SONGLENGTH) return;
  if (what==0) return;
  if (where<0) return;
  if (where>curSongLen) where = curSongLen;

  curSongLen++;

  for (i=curSongLen; i>where; i--) curSong[i]=curSong[i-1];
  curSong[where]=what;

  return;
} 





void
RemoveNote(SWord where)
{
  int i;

  //  i=0;
  //  while ((i<SONGLENGTH) && (curSong[i]!=0)) i++;
  
  if ((where<0)||(where>=(SONGLENGTH))) return;

  for (i=where; i<curSongLen; i++) curSong[i]=curSong[i+1];
  curSong[curSongLen]='\0';
  curSongLen--;
  if (curSongLeft >= curSongLen) 
    curSongLeft = (curSongLen-1);
} 





static SWord
XtoLoc(SWord x)
{
  SWord res = (((x - (42 - 26)) / 26) * 2) - 1;
  if ((((x - 19) % 26)>14)) res++;
  if (res<0) res=0;
  return res;
}




static SWord
YtoLoc(SWord y)
{
  SWord res = ((63 - y) / 4) + 1; 
  if (res<0) res=0;
  if (res>14) res=14;
  return res;
}



static void
GadgetTapped(EventPtr e)
     // here's where all the graphical editing happens.  first we get
     // the tap and drag movements and convert them into staff system
     // coordinates.  then we check what kind of movement the user
     // did.  we try to prefer x movement, because an unwitting y
     // movement may change data so subtly that the user may not even
     // notice the missing accidental
{
  SWord x, y, dx = -1, dy = -1,
    where = 0, whereto=0, i=0;
  RectangleType r;
  Boolean penDown = true;
  FormPtr frmP = FrmGetActiveForm();
  
  PenGetPoint (&x, &y, &penDown);

  while (penDown)
    {
      PenGetPoint (&dx, &dy, &penDown);
    }

  // Simplify the movement
  dx = dx - x;
  dy = dy - y;

  if (abs(dy)<abs(dx))
    dy=0;
  else
    dx=0;

  // convert x,y to staff positions
  FrmGetObjectBounds(frmP, FrmGetObjectIndex(frmP, gdgMainGraphic), &r);
  x = XtoLoc(e->screenX - r.topLeft.x);
  y = YtoLoc(e->screenY - r.topLeft.y);

  //  PRINTF("down at %d,%d delta %d,%d", x, y, dx, dy);

  if ((abs(dx)<4) && (abs(dy)<4)) // tap at location x,y
    {
      where = curSongLeft + ((x-1)/2);
      if ((x<((curSongLen-curSongLeft)*2)) &&
          ((x % 2) == 1)) // tapped within visible notes
        {
          if ((x>0) && ((curSong[where]&0x0F)!=0))
            {
              if ((curSong[where] & 0x0F) == y) // tapped on a nate
                {
                  if ((curSong[where] & 0x40) == 0)
                    {
                      curSong[where] |= 0x40;
                    }
                  else
                    {
                      curSong[where] &= 0xBF;
                    }
                }
              else
                {
                  if ((curSong[where] & 0x0F) == 15) // tapped on a pause
                    {
                      curSong[where] = y;
                    }
                  else
                    {
                      curSong[where] = 15;
                    }
                }
            }
        }

      else if (curSongLen<SONGLENGTH)
        {
          if (((x % 2) == 0) && (x>0)) where += 1;
          if (y!=0) InsertNote(y, where);
        }

      DrawStaff(curSongLeft, true);
    }

  else // drag
    {
      if ((x % 2) == 0) // dragging from a space in the staff
        {
          if (dx>0) // dragging from a space to the right
            {
              // insert a pause
              
              where = curSongLeft + (x/2);
              InsertNote(15, where);
              
            }
          else if (dx<0) // dragging from a space to the left
            {

              // delete the previous note
              if (x>1)
                {
                  where = curSongLeft + ((x-1)/2);
                  RemoveNote(where);
                }
            }
          DrawStaff(curSongLeft, true);
        }
      else // dragging from a note
        {
          where = curSongLeft + ((x-1)/2);
          if (dx>0) // dragging a note to the right
            {

              InsertNote(15, where);
              DrawStaff(curSongLeft, true);
              
            }
          else if (dx<0) // dragging a note to the left
            {
      
              RemoveNote(where-1);
              DrawStaff(curSongLeft, true);
              
            }
          else if (dy<0) // dragging a note upwards
            {

              if (curSong[where] & 0x10) // flat
                {
                  curSong[where] &= 0xEF; // remove flat
                }
              else if (curSong[where] & 0x20) // sharp
                {
                  if ((curSong[where] & 0x0F) < 13)
                    {
                      curSong[where] += 1; // up one
                      curSong[where] &= 0xDF; // remove sharp
                    }
                }
              else // neither flat or sharp
                {
                  switch (curSong[where] & 0x0F)
                    {
                    case 3: // these are moved one step up
                    case 7:
                    case 10:
                      curSong[where] += 1;
                      break;
                    case 13: // sorry, can do nothing here
                    case 15:
                      break;
                    default:
                      curSong[where] |= 0x20;
                      break;
                    }
                }
              
              DrawStaff(curSongLeft, true);
              
            }
          else if (dy>0) // dragging a note downwards
            {

              if (curSong[where] & 0x20) // sharp
                {
                  curSong[where] &= 0xDF; // remove sharp
                }
              else if (curSong[where] & 0x10) // flat
                {
                  if ((curSong[where] & 0x0F) > 1)
                    {
                      curSong[where] -= 1; // down one
                      curSong[where] &= 0xEF; // remove flat
                    }
                }
              else // neither flat or sharp
                {
                  switch (curSong[where] & 0x0F)
                    {
                    case 1: // sorry, nothing to be done
                    case 15:
                      break;
                    case 4: // these are moved one step down
                    case 8:
                    case 11:
                      curSong[where] -= 1;
                      break;
                    default:
                      curSong[where] |= 0x10;
                      break;
                    }
                }
              
              DrawStaff(curSongLeft, true);
              
            }
        }
    }
}





static void
SwitchModes(short newMode)
     // Used to switch between ui modes
{
  FormPtr  frmP = FrmGetFormPtr(frmMain);
  FieldPtr fldP = NULL;
  CharPtr txtP = NULL;
  Handle oldhdl = NULL,
    txtH = NULL;
  RectangleType r;
  Word insPt=0;

  if (newMode==0)
    {
      if (uiMode==MODE_TEXT)
        newMode=MODE_GRAPHIC;
      else
        newMode=MODE_TEXT;
    }

  if (newMode==MODE_GRAPHIC)
    {
      
      for (txtP=&curSong[0]; txtP<=&curSong[SONGLENGTH]; txtP++)
        *txtP='\0';
      
      fldP = (FieldPtr)(FrmGetObjectPtr(frmP,
                  (FrmGetObjectIndex(frmP, fldMainContent))));
      insPt = FldGetInsPtPosition(fldP);

      curSongLen = ConvertSongPtoI(FldGetTextPtr(fldP),
                                   &curSong[0], SONGLENGTH, &insPt);

      FrmHideObject(frmP, FrmGetObjectIndex(frmP, fldMainContent));
      FrmShowObject(frmP, FrmGetObjectIndex(frmP, gdgMainGraphic));
      
      curSongLeft=insPt;
      DrawStaff(curSongLeft, true);
      
    }
  else
    {

      DrawScrollButtons(BUTTON_HIDE, BUTTON_HIDE);
      
      FrmHideObject(frmP, FrmGetObjectIndex(frmP, gdgMainGraphic));
      RctSetRectangle(&r, 5, 16, 150, 116);
      WinEraseRectangle(&r, 0);
      
      if (curSongLen>0)
        txtH=(Handle)MemHandleNew(3*(curSongLen+1));
      else
        txtH=(Handle)MemHandleNew(16);
      ErrFatalDisplayIf((txtH==NULL), "Out of memory."); 
      txtP=MemHandleLock((VoidHand)txtH);
      
      insPt = curSongLeft;
      
      if (curSongLen)
        ConvertSongItoP(&curSong[0], txtP, 3*curSongLen, &insPt);
      else
        *txtP='\0';
      
      MemPtrUnlock((VoidPtr)txtP);
      txtP=NULL;
      
      fldP = (FieldPtr)(FrmGetObjectPtr(frmP,
                (FrmGetObjectIndex(frmP, fldMainContent))));
      oldhdl = FldGetTextHandle(fldP);
      FldSetTextHandle(fldP, txtH);
      if (oldhdl) MemHandleFree((VoidHand)oldhdl);
      FldCompactText(fldP);
      
      FrmShowObject(frmP, FrmGetObjectIndex(frmP, fldMainContent));
      FldSetInsPtPosition(fldP, insPt);
      FldGrabFocus(fldP);
    }
  
  uiMode=newMode;
}



static Boolean
MainFormHandleMenu (EventPtr e)
{
  Boolean handled = false;  
  /*  switch (e->data.menu.itemID)
      {
      default:
      handled=false;
      break;
      }
  */
  return handled;
}



static Boolean
MainFormHandleEvent (EventPtr e)
{
  Boolean handled = false;
  FormPtr  frmP = NULL;
  FieldPtr fldP = NULL;
  Handle oldhdl = NULL,
    txtH = NULL;
  CharPtr txtP = NULL;
  char buffer[SONGLENGTH*3];
  Short n;
  Err err = 0;
  RectangleType r;

  switch (e->eType)
    {
    case frmOpenEvent:
      FrmDrawForm(FrmGetActiveForm());
      uiMode=MODE_TEXT;
      DrawScrollButtons(BUTTON_HIDE, BUTTON_HIDE);
      
      // fixit -- remove this snippet...
      frmP = FrmGetFormPtr(frmMain);
      fldP = (FieldPtr)(FrmGetObjectPtr(frmP,
                  (FrmGetObjectIndex(frmP, fldMainContent))));
      FldPaste(fldP);
      FldSetInsPtPosition(fldP, 0);
      FldDrawField(fldP);
      /// ...end
      
      handled=true;
      break;
      
    case frmUpdateEvent:
      frmP = FrmGetActiveForm();
      FrmDrawForm(frmP);
      if (uiMode==MODE_GRAPHIC) DrawStaff(curSongLeft, true);
      handled=true;
      break;
      
      
    case frmCloseEvent:
    case appStopEvent:
      // ABSOLUTELY NO PRINTF:S ALLOWED HERE
      if (uiMode==MODE_GRAPHIC)
        {
          SwitchModes(MODE_TEXT);
        }
      
      frmP = FrmGetFormPtr(frmMain);
      if (frmP)
        {
          fldP = (FieldPtr)(FrmGetObjectPtr(frmP,
                     (FrmGetObjectIndex(frmP, fldMainContent))));
          if (fldP)
            {
              txtP = FldGetTextPtr(fldP);
              ClipboardAddItem(clipboardText, txtP, StrLen(txtP));
            }
        }
      
      handled=true;
      break;
      
      
    case menuEvent:
      handled = MainFormHandleMenu(e);
      break;
      
    case ctlSelectEvent:
      switch (e->data.ctlSelect.controlID)
        {
        case btnMainInstall:
          if (uiMode==MODE_TEXT)
            {
              
              frmP = FrmGetFormPtr(frmMain);
              fldP = (FieldPtr)(FrmGetObjectPtr(frmP,
                           (FrmGetObjectIndex(frmP, fldMainContent))));
              txtH = (Handle)FldGetTextHandle(fldP);
              txtP = MemHandleLock((VoidHand)txtH);
              n = StrLen(txtP);
              if (n>sizeof(buffer))
                {
                  n = sizeof(buffer);
                  StrNCopy((CharPtr)&buffer, txtP, n);
                }
              else
                {
                  StrCopy((CharPtr)&buffer, txtP);
                }
              MemHandleUnlock((VoidHand)txtH);
              txtP = NULL;
              
            }
          else
            {

              ConvertSongItoP(&curSong[0], (CharPtr)&buffer,
                              sizeof(buffer), NULL);
              
            }
          
          err = InstallMelody((CharPtr)&buffer, StrLen(buffer));
          handled=true;
          break;
          
        case btnMainFetch:
          err = FetchMelody((CharPtr)&buffer, sizeof(buffer));
          if (!err) 
            {
              if (uiMode==MODE_TEXT)
                {
                  frmP = FrmGetFormPtr(frmMain);
                  fldP = (FieldPtr)(FrmGetObjectPtr(frmP,
                             (FrmGetObjectIndex(frmP, fldMainContent))));
                  
                  txtH = (Handle)MemHandleNew((ULong)StrLen((CharPtr)&buffer)+1);
                  txtP = MemHandleLock((VoidHand)txtH);
                  StrCopy(txtP, (CharPtr)&buffer);
                  MemHandleUnlock((VoidHand)txtH);
                  txtP = NULL;
                  oldhdl = FldGetTextHandle(fldP);
                  FldSetTextHandle(fldP, txtH);
                  FldDrawField(fldP);
                  if (oldhdl != NULL) MemHandleFree((VoidHand)oldhdl);
                  FldSetInsPtPosition (fldP, 0);
                }
              else
                {
                  curSongLen = ConvertSongPtoI((CharPtr)&buffer, &curSong[0],
                                               SONGLENGTH, NULL);
                  curSongLeft=0;
                  DrawStaff(curSongLeft, true);
                }
            }
          handled=true;
          break;
          
        case btnMainPlay:
          PlayMelody();
          handled=true;
          break;
          
        case btnMainSwitch:
          SwitchModes(0);
          handled=true;
          break;
          
        case btnMainR:
        case btnMainL:
          if (uiMode==MODE_GRAPHIC)
            {
              
              DrawStaff(curSongLeft +
                        ((e->data.ctlSelect.controlID == btnMainR) ? 1 : -1),
                        true);
        
            }
          else
            {
              DrawScrollButtons(BUTTON_HIDE, BUTTON_HIDE);
            }
          handled=true;
          break;
          
        default:
          handled=false;
          break;
        }
      break;
      
    case penDownEvent:
      if (uiMode==MODE_GRAPHIC)
        {
          frmP = FrmGetActiveForm();
          FrmGetObjectBounds(frmP,
                             FrmGetObjectIndex(frmP, gdgMainGraphic), &r);
          if (RctPtInRectangle (e->screenX, e->screenY, &r)) {
            GadgetTapped(e);
            handled=true;
          }
        }
      break;

    default:
      handled=false;
      break;
    }

  return handled;
}




static Boolean
ApplicationHandleEvent (EventPtr e)
{
  Word formID;
  FormPtr frm;
  
  if (e->eType == frmLoadEvent)
    {
      formID = e->data.frmLoad.formID;
      frm = FrmInitForm (formID);
      FrmSetActiveForm (frm);         
      
      switch (formID)
        {
        case frmMain:
          FrmSetEventHandler (frm, MainFormHandleEvent);
          break;
        }
      return (true);
    }
  return (false);
}





static void
EventLoop() 
{
  Word err;
  EventType e;
  Boolean handled = false;
  
  do
    {
      EvtGetEvent (&e, evtWaitForever);
      if (! SysHandleEvent (&e))
        {
          
          if (uiMode==MODE_TEXT) 
            handled = MenuHandleEvent (MenuGetActiveMenu(), &e, &err);
          else
            handled = false;
          
          if (! handled)
            if (! ApplicationHandleEvent (&e))
              FrmDispatchEvent (&e); 
        }
    }
  while (e.eType != appStopEvent);
}




static Word
StartApplication()
{
  int i;
  Word err;
  Handle bmpH;
  BitmapPtr bmpP;
  WinHandle saveWinH = WinGetDrawWindow();

  // we need to use windows to be able to paste in the bitmaps
  // "transparently" (bitwise OR, actually) in the melody editor.
  // the idea of making windows out of bitmaps is gleaned from
  // HardBall.c by 3Com, but no code was copied directly
  for (i=0; i<BITMAPS; i++)
    {
      bmpH = (Handle)DmGetResource(bitmapRsc, i);
      bmpP = (BitmapPtr)MemHandleLock((VoidHand)bmpH);
      winH[i] = WinCreateOffscreenWindow(bmpP->width, bmpP->height,
                       screenFormat, &err);
      ErrFatalDisplayIf(err, "Yawp! Got an error trying to load images!");
      WinSetDrawWindow(winH[i]);
      WinDrawBitmap(bmpP, 0, 0);
      MemHandleUnlock((VoidHand)bmpH);
      DmReleaseResource((VoidHand)bmpH);
    }

  WinSetDrawWindow(saveWinH);
  
  return 0;
}




static void
StopApplication()
{
  int i;

  if (uiMode==MODE_GRAPHIC)
    {
      SwitchModes(MODE_TEXT);
    }
  
  // release the windows
  for (i=0; i<BITMAPS; i++)
    if (winH[i])
      WinDeleteWindow(winH[i], false);

  // you never know...
  if (comRefNum) ComsClose();
}




DWord PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{
  short err;
  Err eErr;
  EventType e;
  DWord value;
  VoidHand clipHandle;

  if (cmd == sysAppLaunchCmdNormalLaunch)
    {
      err = StartApplication();
      if (err) return (err);
      
      FrmGotoForm(curForm);
      EventLoop();
      StopApplication();
    }
  return 0;
}

