/*
 * Bus schedule generator
 *
 * Jim Rees, University of Michigan CITI, November 2000
 */

#ifdef __palmos__
#pragma pack(2)

#include <Common.h>
#include <System/SysAll.h>
#include <System/Unix/unix_stdio.h>
#include <System/Unix/unix_stdlib.h>
#include <System/Unix/sys_types.h>
#include <System/Unix/sys_socket.h>
#include <UI/UIAll.h>
#include <string.h>

#include "field.h"
#include "resource.h"
#else
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#endif

#include "xio.h"
#include "bus.h"
#include "tdb.h"

extern int openconn(char *hostname, int port, int flags);

#define DBVERSION 2

int nruns;
struct run *runtab;

int nstops;
struct stop *stoptab;

int nroutes;
struct route *routetab;

static char default_url[] = "http://www.citi.umich.edu/u/rees/pilot/busman-tables";
char *url, *version, *info;

char dbcrt[] = "mBUS";
char dbname[] = "busmanDB";

int
read_tables()
{
    unsigned char *p;
    int rtsize, stsize, rusize;
    struct route *rt;
    struct stop *st;
    struct run *ru;

    if (tdb_open(dbcrt, dbname, "r") < 0)
	return -1;

    while ((p = tdb_read()) != NULL) {
	switch (p[0]) {
	case RT_URL:
	    if (!url)
		url = strdup(p + 2);
	    break;
	case RT_VERSION:
	    if (!version)
		version = strdup(p + 2);
	    break;
	case RT_INFO:
	    if (!info)
		info = strdup(p + 2);
	    break;
	case RT_SIZE:
	    /* sizes */
	    nroutes = nstops = nruns = 0;
	    if (p[2] != DBVERSION)
		printf("Warning: Incompatible db version %d (expecting %d)\n", p[2], DBVERSION);
	    rtsize = p[3];
	    stsize = p[4];
	    rusize = p[5];
	    if (routetab)
		free(routetab);
	    if (stoptab)
		free(stoptab);
	    if (runtab)
		free(runtab);
	    routetab = (struct route *) malloc(rtsize * sizeof (struct route));
	    stoptab = (struct stop *) malloc(stsize * sizeof (struct stop));
	    runtab = (struct run *) malloc(rusize * sizeof (struct run));
	    break;
	case RT_ROUTE:
	    /* route */
	    rt = &routetab[nroutes++];
	    memcpy(rt, p + 2, sizeof (struct route) - sizeof (char *));
	    rt->name = strdup(p + 2 + sizeof (struct route) - sizeof (char *));
	    break;
	case RT_STOP:
	    /* stop */
	    st = &stoptab[nstops++];
	    memcpy(st, p + 2, sizeof (struct stop) - sizeof (char *));
	    st->name = strdup(p + 2 + sizeof (struct stop) - sizeof (char *));
	    break;
	case RT_RUN:
	    /* run */
	    ru = &runtab[nruns++];
	    memcpy(ru, p + 2, sizeof (struct run));
	    break;
	}
	tdb_release();
    }
    tdb_close();
    if (vflag)
	printf("read %d routes\n", nroutes);
    return 0;
}

int
fetch_tables()
{
    XFILE *f;
    int fd, r;
    char *s, buf[100];

    if (!url)
	url = strdup(default_url);

    s = url;
    if (!strncmp(s, "http://", 7))
	s += 7;
    sscanf(s, "%[^/]", buf);
    s += strlen(buf);
    if (vflag)
	printf("host \"%s\" file \"%s\"\n", buf, s);
    fd = openconn(buf, 80, vflag);
    if (fd < 0)
	return -1;
    sprintf(buf, "GET %s\r\n", s);
    write(fd, buf, strlen(buf));
    f = xfdopen(fd, "r");
    r = read_table_file(f);
    xclose(f);
    close(fd);
    if (r >= 0 && vflag)
	printf("fetched %d routes\n", nroutes);
    return r;
}

int
read_table_file(XFILE *f)
{
    int t0, t1, t2, t3, rtsize, stsize, rusize;
    char buf[80], s0[16], s1[64], *cp;
    struct route *rt = NULL;
    struct stop *st;
    struct run *ru;

    while (xgets(f, buf, sizeof buf) >= 0) {
	if (buf[0] == '#' || sscanf(buf, "%15s", s0) != 1)
	    continue;
	if (buf[0] == '<') {
	    /* Looks like html, maybe error message? */
	    printf("%s", buf);
	    return -1;
	}
	if (!strcmp(s0, "size")) {
	    nroutes = nstops = nruns = 0;
	    sscanf(buf, "%*s %d %d %d", &rtsize, &stsize, &rusize);
	    if (routetab)
		free(routetab);
	    if (stoptab)
		free(stoptab);
	    if (runtab)
		free(runtab);
	    routetab = (struct route *) malloc(rtsize * sizeof (struct route));
	    stoptab = (struct stop *) malloc(stsize * sizeof (struct stop));
	    runtab = (struct run *) malloc(rusize * sizeof (struct run));
	} else if (!strcmp(s0, "version")) {
	    sscanf(buf, "%*s \"%63[^\"]\"", s1);
	    version = strdup(s1);
	    printf("table version %s\n", s1);
	} else if (!strcmp(s0, "info")) {
	    sscanf(buf, "%*s \"%63[^\"]\"", s1);
	    info = strdup(s1);
	    if (vflag)
		printf("%s\n", s1);
	} else if (!strcmp(s0, "route")) {
	    rt = &routetab[nroutes++];
	    bzero(rt, sizeof (struct route));
	    sscanf(buf, "%*s \"%63[^\"]\"", s1);
	    rt->name = strdup(s1);
	    if (vflag)
		printf("route %s\n", s1);
	} else if (!strcmp(s0, "stop")) {
	    if (!rt)
		continue;
	    sscanf(buf, "%*s \"%63[^\"]\" %d", s1, &t0);
	    st = &stoptab[nstops];
	    bzero(st, sizeof (struct stop));
	    st->name = strdup(s1);
	    st->time = t0;
	    rt->stops[rt->nstops++] = nstops++;
	    if (vflag)
		printf("stop %s\n", s1);
	} else if (!strcmp(s0, "run")) {
	    if (!rt)
		continue;
	    ru = &runtab[nruns];
	    bzero(ru, sizeof (struct run));
	    sscanf(buf, "%*s %d %d %d %d %s", &t0, &t1, &t2, &t3, s1);
	    ru->days = t0;
	    ru->start = t1;
	    ru->freq = t2;
	    ru->end = t3;
	    ru->stops = 0;
	    if (s1[0] == '*')
		ru->stops = 0xffff;
	    else {
		for (cp = s1; *cp; cp++) {
		    t0 = *cp;
		    if (t0 >= '0' && t0 <= '9')
			t0 -= '0';
		    else if (t0 >= 'a' && t0 <= 'z')
			t0 -= 'a' - 10;
		    ru->stops |= (1 << t0);
		}
	    }
	    if (vflag)
		printf("run %d\n", nruns);
	    rt->runs[rt->nruns++] = nruns++;
	}
    }
    return 0;
}

void
store_tables()
{
    int i;
    unsigned char buf[8];

    tdb_delete(dbcrt);
    if (tdb_open(dbcrt, dbname, "c") < 0)
	return;

    buf[0] = RT_SIZE;
    buf[1] = 0;
    buf[2] = DBVERSION;
    buf[3] = nroutes;
    buf[4] = nstops;
    buf[5] = nruns;
    tdb_create(sizeof buf);
    tdb_write(buf, sizeof buf);
    tdb_release();

    buf[0] = RT_URL;
    buf[1] = strlen(url);
    tdb_create(2 + strlen(url) + 1);
    tdb_write(buf, 2);
    tdb_write(url, buf[1] + 1);
    tdb_release();

    buf[0] = RT_VERSION;
    buf[1] = strlen(version);
    tdb_create(2 + strlen(version) + 1);
    tdb_write(buf, 2);
    tdb_write(version, buf[1] + 1);
    tdb_release();

    buf[0] = RT_INFO;
    buf[1] = strlen(info);
    tdb_create(2 + strlen(info) + 1);
    tdb_write(buf, 2);
    tdb_write(info, buf[1] + 1);
    tdb_release();

    for (i = 0; i < nroutes; i++) {
	buf[0] = RT_ROUTE;
	buf[1] = strlen(routetab[i].name);
	tdb_create(2 + sizeof (struct route) - sizeof (char *) + buf[1] + 1);
	tdb_write(buf, 2);
	tdb_write(&routetab[i], sizeof (struct route) - sizeof (char *));
	tdb_write(routetab[i].name, buf[1] + 1);
	tdb_release();
    }
    for (i = 0; i < nstops; i++) {
	buf[0] = RT_STOP;
	buf[1] = strlen(stoptab[i].name);
	tdb_create(2 + sizeof (struct stop) - sizeof (char *) + buf[1] + 1);
	tdb_write(buf, 2);
	tdb_write(&stoptab[i], sizeof (struct stop) - sizeof (char *));
	tdb_write(stoptab[i].name, buf[1] + 1);
	tdb_release();
    }
    for (i = 0; i < nruns; i++) {
	buf[0] = RT_RUN;
	buf[1] = 0;
	tdb_create(2 + sizeof (struct run));
	tdb_write(buf, 2);
	tdb_write(&runtab[i], sizeof (struct run));
	tdb_release();
    }
    tdb_close();
}

void
print_tables(struct route *rt)
{
    int i;
    struct stop *st;
    struct run *ru;

    for (i = 0; i < rt->nstops; i++) {
	st = &stoptab[rt->stops[i]];
	printf("stop \"%s\" %d\n", st->name, st->time);
    }
    for (i = 0; i < rt->nruns; i++) {
	ru = &runtab[rt->runs[i]];
	printf("run %d %4d %2d %4d %x\n", (int) ru->days, (int) ru->start, (int) ru->freq, (int) ru->end, (int) ru->stops);
    }
}

void
print_info()
{
    if (info)
	printf("%s\n", info);
    printf("%d routes, %d stops, %d runs\n", nroutes, nstops, nruns);
    if (version)
	printf("%s\n", version);
    if (url)
	printf("%s\n", url);
}

/*
copyright 2000
the regents of the university of michigan
all rights reserved

permission is granted to use, copy, create derivative works 
and redistribute this software and such derivative works 
for any purpose, so long as the name of the university of 
michigan is not used in any advertising or publicity 
pertaining to the use or distribution of this software 
without specific, written prior authorization.  if the 
above copyright notice or any other identification of the 
university of michigan is included in any copy of any 
portion of this software, then the disclaimer below must 
also be included.

this software is provided as is, without representation 
from the university of michigan as to its fitness for any 
purpose, and without warranty by the university of 
michigan of any kind, either express or implied, including 
without limitation the implied warranties of 
merchantability and fitness for a particular purpose. the 
regents of the university of michigan shall not be liable 
for any damages, including special, indirect, incidental, or 
consequential damages, with respect to any claim arising 
out of or in connection with the use of the software, even 
if it has been or is hereafter advised of the possibility of 
such damages.
*/
