/*
 * Bus schedule generator
 *
 * Jim Rees, University of Michigan CITI, November 2000
 */

#pragma pack(2)

#include <Common.h>
#include <System/SysAll.h>
#include <System/DateTime.h>
#include <System/MemoryMgr.h>
#include <System/DataMgr.h>
#include <System/SysEvtMgr.h>
#include <System/Unix/unix_stdio.h>
#include <System/Unix/unix_stdlib.h>
#include <System/Unix/sys_types.h>
#include <System/Unix/sys_socket.h>
#include <UI/UIAll.h>
#include <string.h>

#include "field.h"
#include "xio.h"
#include "conn.h"

#include "bus.h"
#include "resource.h"

int vflag;
int r, w, s1, s3, evID;

FieldPtr printfield, routefield, stop1field, stop3field, wanted;

int StartApplication();
void StopApplication();
void EventLoop();
static Boolean busman(EventPtr event);
void print_desired_stuff(int evID);
void display_route_list(int r);
void display_stop_list(FieldPtr f, int n);
void display_list(char **names, int nnames, FieldPtr fld, int n);
int vrf_routes(int r, int needroute);
void get_input(int *rp, int *s1p, int *s2p);
void set_input(FieldPtr f, int n);
void do_fetch();

DWord
PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{
    int err;

    if (cmd == sysAppLaunchCmdNormalLaunch) {
	err = StartApplication();
	if (err)
	    return err;
	EventLoop();
	StopApplication();
	return 0;
    } else
	return sysErrParamErr;
}

int
StartApplication()
{
    FrmGotoForm(formID_list);
    return 0;
}

void
StopApplication()
{
    FrmCloseAllForms();
    CloseNet();
}

void
EventLoop()
{
    short err;
    int formID;
    FormPtr form;
    EventType event;

    do {
	EvtGetEvent(&event, 200);

	if (SysHandleEvent(&event))
	    continue;
	if (MenuHandleEvent((void *)0, &event, &err))
	    continue;

	if (event.eType == frmLoadEvent) {
	    formID = event.data.frmLoad.formID;
	    form = FrmInitForm(formID);
	    FrmSetActiveForm(form);
	    switch (formID) {
	    case formID_print:
	    case formID_list:
		FrmSetEventHandler(form, (FormEventHandlerPtr) busman);
		break;
	    }
	}
	FrmDispatchEvent(&event);
    } while (event.eType != appStopEvent);
}

static Boolean
busman(EventPtr event)
{
    FormPtr form = FrmGetActiveForm();
    Word item;
    FieldPtr field;
    static int did_read_tables;
    int handled;

    item = FrmGetFocus(form);
    if (item == noFocus)
	field = printfield;
    else
	field = FrmGetObjectPtr(form, item);

    switch (event->eType) {
    case frmOpenEvent:
	if (FrmGetActiveFormID() == formID_list) {
	    routefield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_route));
	    stop1field = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_stop1));
	    stop3field = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_stop3));
	    printfield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_print1));
	    if (!did_read_tables) {
		read_tables();
		did_read_tables = 1;
	    } else {
		/* Returning from timetable; set sticky stops */
		set_input(stop1field, s1);
		set_input(stop3field, s3);
	    }
	    display_route_list(r);
	} else if (FrmGetActiveFormID() == formID_print) {
	    printfield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_print2));
	    print_desired_stuff(evID);
	}
	handled = 1;
	break;

    case ctlSelectEvent:
	evID = event->data.ctlEnter.controlID;
	handled = 1;
	switch (evID) {
	case buttonID_Routes:
	    if (vrf_routes(-1, 0))
		display_route_list(r);
	    break;
	case buttonID_from:
	    display_stop_list(stop1field, s1);
	    break;
	case buttonID_to:
	    display_stop_list(stop3field, s3);
	    break;
	case buttonID_Wday:
	case buttonID_Sat:
	case buttonID_Sun:
	    w = evID - buttonID_Wday;
	    print_desired_stuff(evID);
	    break;
	case buttonID_back:
	    FrmGotoForm(formID_list);
	    break;
	default:
	    handled = 0;
	    break;
	}
	break;

    case lstSelectEvent:
	set_input(wanted, (int) (event->data.lstSelect.selection));
	handled = 1;
	break;

    case keyDownEvent:
	handled = scrollfield(printfield, event->data.keyDown.chr);
	break;

    case menuEvent:
	evID = event->data.menu.itemID;
	switch (evID) {
	case menuitemID_info:
	case menuitemID_tables:
	case menuitemID_fetch:
	    print_desired_stuff(evID);
	    break;
	case menuitemID_warn:
	    FrmHelp(textID_warn);
	    break;
	case menuitemID_about:
	    FrmAlert(alertID_about);
	    break;
	case menuitemID_help:
	    FrmHelp(textID_help);
	    break;
	case menuitemID_cut:
	    FldCut(field);
	    break;
	case menuitemID_copy:
	    FldCopy(field);
	    break;
	case menuitemID_paste:
	    FldPaste(field);
	    break;
	}
	handled = 1;
	break;

    default:
	handled = 0;
	break;

    }
    return handled;
}

/* Switch to print form, and print whatever (based on last event) */

void print_desired_stuff(int evID)
{
    FormPtr form = FrmGetActiveForm();
    struct route *rt;

    if (FrmGetActiveFormID() == formID_list) {
	get_input(&r, &s1, &s3);
	FrmGotoForm(formID_print);
	return;
    }

    switch (evID) {
    case buttonID_Wday:
    case buttonID_Sat:
    case buttonID_Sun:
	if (vrf_routes(r, 1)) {
	    rt = &routetab[r];
	    hidefield(fieldID_print2);
#ifndef DEBUG
	    setfield(printfield, NULL);
#endif
	    makesched(rt, w, s1, -1, s3);
	    FldSetScrollPosition(printfield, 0);
	    showfield(fieldID_print2);
	}
	break;
    case menuitemID_info:
	print_info();
	break;
    case menuitemID_tables:
	if (vrf_routes(r, 1))
	    print_tables(&routetab[r]);
	break;
    case menuitemID_fetch:
	do_fetch();
	break;
    }
    FrmDrawForm(form);
}

void display_route_list(int r)
{
    int i;
    static char **routenames;

    if (!nroutes)
	return;
    if (!routenames)
	routenames = (char **) malloc(nroutes * sizeof (char **));

    for (i = 0; i < nroutes; i++)
	routenames[i] = routetab[i].name;

    display_list(routenames, nroutes, routefield, r);
}

void display_stop_list(FieldPtr f, int n)
{
    int i;
    struct route *rt;
    static char **stopnames;

    if (!stopnames)
	stopnames = (char **) malloc(MAXSTOPS * sizeof (char **));

    get_input(&r, &s1, &s3);
    if (!vrf_routes(r, 1))
	return;
    rt = &routetab[r];
    for (i = 0; i < rt->nstops; i++)
	stopnames[i] = stoptab[rt->stops[i]].name;

    display_list(stopnames, rt->nstops, f, n);
}

void display_list(char **names, int nnames, FieldPtr fld, int n)
{
    FormPtr form = FrmGetActiveForm();
    ListPtr lp;

    lp = (ListPtr) FrmGetObjectPtr(form, FrmGetObjectIndex(form, listID_data));
    LstSetListChoices(lp, names, nnames);
    LstSetSelection(lp, n);
    set_input(fld, n);
    wanted = fld;
    FrmDrawForm(form);
}

static char url_msg[] = "Enter url below and select <fetch> again:\n";

void
do_fetch()
{
    char *s;
    int url_msglen = strlen(url_msg);

    /* Get the url to fetch */
    s = FldGetTextPtr(printfield);
    if (s && !strncmp(s, url_msg, url_msglen)) {
	/* Got it */
	if (url)
	    free(url);
	url = strdup(s + url_msglen);
    } else {
	/* Prompt user for url, supplying default */
	setfield(printfield, NULL);
	if (!url)
	    url = strdup("http://");
	printf("%s%s", url_msg, url);
	return;
    }
    vflag = 1;
    OpenNet();
    if (fetch_tables() >= 0)
	store_tables();
    vflag = 0;
}

void get_input(int *rp, int *s1p, int *s2p)
{
    char *s;

    s = FldGetTextPtr(routefield);
    *rp = s ? (atoi(s) - 1) : 0;
    s = FldGetTextPtr(stop1field);
    *s1p = s ? (atoi(s) - 1) : 0;
    s = FldGetTextPtr(stop3field);
    *s2p = s ? (atoi(s) - 1) : 0;
}

void set_input(FieldPtr f, int n)
{
    char buf[8];

    sprintf(buf, "%d", n + 1);
    setfield(f, buf);
}

/*
copyright 2000
the regents of the university of michigan
all rights reserved

permission is granted to use, copy, create derivative works 
and redistribute this software and such derivative works 
for any purpose, so long as the name of the university of 
michigan is not used in any advertising or publicity 
pertaining to the use or distribution of this software 
without specific, written prior authorization.  if the 
above copyright notice or any other identification of the 
university of michigan is included in any copy of any 
portion of this software, then the disclaimer below must 
also be included.

this software is provided as is, without representation 
from the university of michigan as to its fitness for any 
purpose, and without warranty by the university of 
michigan of any kind, either express or implied, including 
without limitation the implied warranties of 
merchantability and fitness for a particular purpose. the 
regents of the university of michigan shall not be liable 
for any damages, including special, indirect, incidental, or 
consequential damages, with respect to any claim arising 
out of or in connection with the use of the software, even 
if it has been or is hereafter advised of the possibility of 
such damages.
*/
