/*
 * Copyright 2002 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Niels Provos.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#import "Controller.h"

#include <sys/types.h>
#include <sys/param.h>
#include <limits.h>
#include <libgen.h>
#include <pwd.h>
#include <unistd.h>

#define errx(w, l)  do { \
    fprintf(stderr, "%s\n", l); \
    exit(w); \
} while(0)

static char home[MAXPATHLEN];		/* Home directory of user */
static char username[MAXLOGNAME];	/* Username: predicate match and expansion */

char *
freadline(char *line, int size, FILE *fp)
{
	char *p;

	if (fgets(line, size, fp) == NULL)
		return (NULL);

	if ((p = strchr(line, '\n')) == NULL)
		return (line);	/* just terminate */
	*p = '\0';

	return (line);
}

char *
strrpl(char *str, size_t size, char *match, char *value)
{
	char *p, *e;
	int len, rlen;

	p = str;
	e = p + strlen(p);
	len = strlen(match);

	/* Try to match against the variable */
	while ((p = strchr(p, match[0])) != NULL) {
		if (!strncmp(p, match, len) && !isalnum(p[len]))
			break;
		p += len;

		if (p >= e)
			return (NULL);
	}

	if (p == NULL)
		return (NULL);

	rlen = strlen(value);

	if (strlen(str) - len + rlen > size)
		return (NULL);

	memmove(p + rlen, p + len, strlen(p + len) + 1);
	memcpy(p, value, rlen);

	return (p);
}

void
free_list(void *data, void *arg)
{
	free(data);
}

int
make_policy(char **type, char **data, int count, char **presult, int derive)
{
	static char result[4096];
	char one[2048];
	int i;
	int nfilename, isfilename;

	result[0] = '\0';
	nfilename = 0;
	for (i = 0; i < count; i++) {
		isfilename = 0;
		/* Special case for non existing filenames */
		if (strstr(data[i], "<non-existent filename>") != NULL) {
			snprintf(result, sizeof(result),
			    "filename%s sub \"<non-existent filename>\" then deny[enoent]", i ? "[1]" : "");
			break;
		}

		if (!strcmp(type[i], "uid") || !strcmp(type[i], "gid") ||
		    !strcmp(type[i], "argv"))
			continue;

		/* Special case for system calls with more than one filename */
		if (!strcmp(type[i], "filename")) {
			isfilename = 1;
			nfilename++;
		}
		
		if (strlen(result)) {
			if (strlcat(result, " and ", sizeof(result)) >= sizeof(result))
				return (-1);
		}

		/* Special treatment for filenames */
		if (isfilename) {
			char filename[2048];
			char *operator = "eq";

			if (derive) {
				operator = "match";

				snprintf(filename, sizeof(filename),
				    "%s/*", dirname(data[i]));
			} else
				strlcpy(filename, data[i], sizeof(filename));

			/* Make useful replacements */
			while (strrpl(filename, sizeof(filename),
				   home, "$HOME") != NULL)
				;
			while (strrpl(filename, sizeof(filename),
				   username, "$USER") != NULL)
				;

			snprintf(one, sizeof(one), "%s%s %s \"%s\"",
			    type[i], isfilename && nfilename == 2 ? "[1]" : "",
			    operator, filename);
		} else {
			snprintf(one, sizeof(one), "%s eq \"%s\"",
			    type[i], data[i]);
		}

		if (strlcat(result, one, sizeof(result)) >= sizeof(result))
			return (-1);;
	}

	if (!strlen(result))
		return (-1);

	/* Normal termination */
	if (i == count)
		strlcat(result, " then permit", sizeof(result));

	*presult = result;
	return (nfilename);
}

#define MAX_SYSCALLARGS	10

void
make_policy_suggestion(NSComboBox *box, char *info)
{
	char line[4096], *next, *p, *syscall;
	char *type[MAX_SYSCALLARGS], *data[MAX_SYSCALLARGS];
	int count = 0, res;

	/* Prepare parsing of info line */
	strlcpy(line, info, sizeof(line));

	next = line;
	syscall = strsep(&next, ",");

	/* See if we can make a suggestion for this system call */
	if (next == NULL)
		goto out;
	next++;
	if (!strncmp(next, "args: ", 6)) {
		count = -1;
		goto out;
	}

	while (next != NULL) {
		p = next;

		next = strstr(next, ", ");
		if (next != NULL) {
			*next = '\0';
			next += 2;
		}

		type[count] = strsep(&p, ":");
		data[count] = p + 1;

		count++;
	}

	res = make_policy(type, data, count, &p, 0);

	if (res != -1)
		[box addItemWithObjectValue: [[NSString alloc] initWithCString: p]];

	if (res > 0) {
		res = make_policy(type, data, count, &p, 1);
		if (res != -1)
			[box addItemWithObjectValue: [[NSString alloc] initWithCString: p]];
	}
			
 out:
	/* Simples policy */
	p = count == -1 ? "permit" : "true then permit";
	[box addItemWithObjectValue: [[NSString alloc] initWithCString: p]];
}

void
parameters(void)
{
	struct passwd *pw;
	uid_t uid = getuid();

	/* Find out current username. */
	if ((pw = getpwuid(uid)) == NULL)
		snprintf(username, sizeof(username), "uid %u", uid);
	else
		snprintf(username, sizeof(username), "%s", pw->pw_name);

	strlcpy(home, pw->pw_dir, sizeof(home));
}

void
socketCallBack (CFSocketRef socketref, CFSocketCallBackType type,
    CFDataRef address, const void *data, void *info)
{
    Controller *me = (Controller *) info;

    [me read_stdin];
}

@implementation Controller

- (void)read_stdin
{
	char line[4096], *p;
	char *name, *id, *polname, *filters;
	static int newsyscall = 1;
	int nfilters;

	if (freadline(line, sizeof(line), stdin) == NULL) {
		[NSApp terminate: self];
		return;
	}

	if (newsyscall) {
		p = line;
		name = strsep(&p, ",");
		if (p == NULL || *p == '\0')
			errx(1, "Bad input line");
		p++;
		strsep(&p, " ");
		if (p == NULL || *p == '\0')
			errx(1, "Bad input line");
		id = strsep(&p, "(");
		if (p == NULL || *p == '\0')
			errx(1, "Bad input line");
		strsep(&p, ":");
		if (p == NULL || *p == '\0')
			errx(1, "Bad input line");
		p++;
		polname = strsep(&p, ",");
		if (p == NULL || *p == '\0')
			errx(1, "Bad input line");
		strsep(&p, ":");
		if (p == NULL || *p == '\0')
			errx(1, "Bad input line");
		p++;
		filters = strsep(&p, ",");
		if (p == NULL || *p == '\0')
			errx(1, "Bad input line");
		nfilters = atoi(filters);
		strsep(&p, ":");
		if (p == NULL || *p == '\0')
			errx(1, "Bad input line");
		p++;

		[processname setStringValue: [[NSString alloc] initWithCString: name]];
		[processpid setStringValue:  [[NSString alloc] initWithCString: id]];
		[policyname setStringValue:  [[NSString alloc] initWithCString: polname]];

		[syscallinfo setStringValue:  [[NSString alloc] initWithCString: p]];
		[statusline setStringValue: @""];

		[entry removeAllItems];
		make_policy_suggestion(entry, p);
		[entry selectItemAtIndex: 0];

		[NSApp unhide: self];
		[window makeKeyAndOrderFront: nil];

		newsyscall = 0;

		return;
	}

	if (!strcmp(line, "OKAY")) {
		newsyscall = 1;
                
		[NSApp hide: self];
		return;
	}
	
	if (!strcmp(line, "WRONG"))
		return;

	[statusline setStringValue:  [[NSString alloc] initWithCString: line]];
}

- (IBAction)denyalways:(id)sender
{
	NSString *what;

	what = [errorcode objectValue];

	fprintf(stdout, "deny[%s]\n", [what cString]);
}

- (IBAction)denyonce:(id)sender
{
	NSString *what;

	what = [errorcode objectValue];

	fprintf(stdout, "deny-now[%s]\n", [what cString]);
}

- (IBAction)enterfilter:(id)sender
{
	NSString *what;

	what = [sender objectValue];

	fprintf(stdout, "%s\n", [what cString]);
}

- (IBAction)kill:(id)sender
{
    fprintf(stdout, "kill\n");
}

- (IBAction)permitalways:(id)sender
{
    fprintf(stdout, "permit\n");
}

- (IBAction)permitonce:(id)sender
{
    fprintf(stdout, "permit-now\n");
}

@end

@implementation Controller(ApplicationNotification)

-(void)applicationDidFinishLaunching:(NSNotification*)notification
{
	CFSocketContext context = { 0, self, NULL, NULL, NULL };
	CFRunLoopSourceRef rls;
        CFSocketRef runLoopSocket;

	setvbuf(stdout, (char *)NULL, _IOLBF, 0);

	/* Set up parameters for the user */
	parameters();

	runLoopSocket = CFSocketCreateWithNative (NULL,
	    fileno(stdin),
	    kCFSocketReadCallBack,
	    socketCallBack,
	    &context);

	if (runLoopSocket == NULL)
		goto error;

	rls = CFSocketCreateRunLoopSource (NULL, runLoopSocket, 0);
	if (rls == NULL)
		goto error;

	CFRunLoopAddSource (CFRunLoopGetCurrent(), rls, kCFRunLoopDefaultMode);
	CFRelease (rls);

	[NSApp hide: self];

	return;

 error:
	[NSApp terminate: self];
	return;
}

@end
