/*

Copyright (C) 2003 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif


#include <honeyd/hooks.h>
#include "honeycomb.h"
#include "hc_debug.h"
#include "hc_signature_hist.h"
#include "hc_string_alg.h"
#include "hc_ip.h"
#include "hc_udp.h"

static HC_Bitmap        *udp_mask_map;
static struct udp_hdr   *udp_mask;

static HC_BitmapQueue   *in_queue;


int
hc_udp_analyze(const struct ip_hdr *iphdr_old,
	       const struct ip_hdr *iphdr_new,
	       HC_Signature *sig)
{
	struct udp_hdr *udphdr_old, *udphdr_new;
	int result;

	D(("Analyzing UDP headers %p %p\n", iphdr_old, iphdr_old));

	udphdr_old = (struct udp_hdr*) ((u_char *) iphdr_old + (iphdr_old->ip_hl << 2));
	udphdr_new = (struct udp_hdr*) ((u_char *) iphdr_new + (iphdr_new->ip_hl << 2));
	
	result = hc_ip_analyze(iphdr_new, iphdr_old, sig);

	hc_sig_set_proto(sig, IP_PROTO_UDP);

	hc_blob_get_mask((u_char*) udphdr_old, UDP_HDR_LEN,
			 (u_char*) udphdr_new, UDP_HDR_LEN,
			 (u_char*) udp_mask, UDP_HDR_LEN);	
	
	/* We detect matching destination ports as we want to know what is being
	 * connected to. We also want to know when dst port is 0.
	 */
	if (udp_mask->uh_dport == USHRT_MAX || udphdr_new->uh_dport == 0) {

		D(("UDP dst port match found, or 0\n"));
		hc_sig_set_dst_port(sig,
				    ntohs(udphdr_new->uh_dport), HC_COMP_EQ,
				    0, HC_COMP_NA);
		result++;
	}
	
	/* We also register source port 0, and the source port if there's a
	 * match and it's not ephemeral. Other things would be feasible, like
	 * always register src port when there's a match, but only if the
	 * destination port is a match as well ... mhm.
	 */
	if (((udp_mask->uh_sport == USHRT_MAX) && (ntohs(udphdr_new->uh_sport) < 1024)) ||
	    (udphdr_new->uh_sport == 0)) {

		D(("UDP src port match found, or not ephemeral\n"));
		hc_sig_set_src_port(sig,
				    ntohs(udphdr_new->uh_dport), HC_COMP_EQ,
				    0, HC_COMP_NA);		
		result++;
	}

	/* Now for the fun part -- finding patterns in packet payload,
	 * but only do this if we actually have any payload.
	 */
	if (ntohs(udphdr_new->uh_ulen) > UDP_HDR_LEN) {

		LST_String *pattern;
		u_char *payload_old, *payload_new;
		u_int   payload_old_len, payload_new_len;

		D(("UDP payload analysis\n"));
		payload_old = ((u_char*) udphdr_old) + UDP_HDR_LEN;
		payload_new = ((u_char*) udphdr_new) + UDP_HDR_LEN;
		payload_old_len = ntohs(udphdr_old->uh_ulen) - UDP_HDR_LEN;
		payload_new_len = ntohs(udphdr_new->uh_ulen) - UDP_HDR_LEN;

		pattern = hc_string_alg_lcs(payload_old, payload_old_len,
					    payload_new, payload_new_len,
					    HC_UDP_PATTERN_MINLEN);

		if (pattern) {

			D(("UDP substring found: '%s'\n", lst_string_print(pattern)));
			hc_sig_set_content(sig, pattern->data,
					   lst_string_get_length(pattern));
			lst_string_free(pattern);
		}
	}
	
	return result;
}


static void
udp_match(HC_Bitmap *map, HC_Bitmap *new_map)
{
	HC_Signature sig;
	HC_Blob     *blob1, *blob2;

	blob1 = map->blobs.tqh_first;
	blob2 = new_map->blobs.tqh_first;
	
	hc_sig_init(&sig);

	if (hc_udp_analyze((struct ip_hdr *) blob1->data,
			   (struct ip_hdr *) blob2->data, &sig) > 0)
		hc_sighist_insert(&sig);

	hc_sig_clear(&sig);
}


static void
udp_hook(u_char *packet_data, u_int packet_len, void *user_data)
{
	struct ip_hdr  *iphdr;
	HC_Bitmap      *map;

	D(("UDP packet inspection ------------------------\n"));

	/* Within the queue, we need the full packet data because we
	 * also want to investigate the packet content, so keep the
	 * entire IP length:
	 */
	iphdr = (struct ip_hdr *) packet_data;
	map = hc_bitmap_new_with_data(packet_data, ntohs(iphdr->ip_len));	
	
	hc_bitmap_queue_foreach(in_queue, (HC_BitmapCB) udp_match, map);
	hc_bitmap_queue_add(in_queue, map);
}


void
hc_udp_init(void)
{
	HC_Blob *blob;

	udp_mask_map = hc_bitmap_new(UDP_HDR_LEN);
	
	blob = udp_mask_map->blobs.tqh_first;
	udp_mask = (struct udp_hdr *) blob->data;
	
	in_queue = hc_bitmap_queue_new(HC_UDP_BACKLOG);
	hooks_add_packet_hook(IP_PROTO_UDP, HD_INCOMING, udp_hook, NULL);
}
