
/*

Copyright (C) 2003 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __hc_tcp_conns_h
#define __hc_tcp_conns_h

#include <sys/queue.h>

#include "honeycomb.h"
#include "hc_bitmaps.h"


/* This structure identifies a TCP connection through IP src
 * and dst addresses and port pairs.
 */
typedef struct hc_tcp_conn_id
{
	ip_addr_t        src_addr;
	uint16_t         src_port;
	
	ip_addr_t        dst_addr;
	uint16_t         dst_port;
	
} HC_TCPConnID;



/* HC_TCP_Conn structures maintain the state for a single TCP
 * connection, reassembling exchanged messages up to a limited
 * depth.
 */
typedef struct hc_tcp_conn
{
	TAILQ_ENTRY(hc_tcp_conn) conns;

	/* Connection identifier */
	HC_TCPConnID     id;

	/* Each connection gets a unique key
	 * assigned when created:
	 */
	int              key;

	/* We remember the headers of the first
	 * packet in a connection in each direction:
	 */
	char             hdr[IP_HDR_LEN_MAX + TCP_HDR_LEN_MAX];
	
	/* Connection termination progress */
	uint32_t         fin;
	uint32_t         fin_back;
	char             fin_acked;
	char             fin_back_acked;

	/* Is connection terminated? */
	char             terminated;

	/* Has data been sent both ways? */
	char             answered;

	/* Reassembled stream data. We use one bitmap
	 * for real stream data, where each blob consists
	 * of data flowing in one direction without real data
	 * flowing the other way (other than acks).
	 *
	 * We separately keep the headers of the first
	 * chunk of new messages around, for pattern matching
	 * in those header fields.
	 */
	HC_Bitmap       *stream;
	HC_Bitmap       *stream_headers;
	int              stream_reversed;
	
} HC_TCPConn;


typedef void (*HC_TCPConnCB) (HC_TCPConn *conn, void *user_data);


/**
 * hc_tcp_conns_init - initializes TCP state management.
 * @max_num_conns: the maximum number of connections for which we keep state.
 */
void          hc_tcp_conns_init(u_int max_num_conns);


/**
 * hc_tcp_conns_find - looks up an existing connection.
 * @src_addr: IP source address.
 * @src_port: TCP source port.
 * @dst_addr: IP source address.
 * @dst_port: TCP destination port.
 *
 * The function tries to find the connection identified through the
 * input parameters and returns it.
 * 
 * Returns: found connection or %NULL if it doesn't exist (anymore).
 */
HC_TCPConn   *hc_tcp_conns_find(ip_addr_t src_addr, uint16_t src_port,
				ip_addr_t dst_addr, uint16_t dst_port);


/**
 * hc_tcp_conns_add - adds new connection to state manager.
 * @iphdr: IP header data.
 * @tcphdr: TCP input data.
 *
 * The function adds state for the connection defined through @iphdr
 * and @tcphdr to the state manager, and returns it. If the connection
 * already exists, nothing is done and the function returns %NULL.
 *
 * Returns: new connection or %NULL if already existant.
 */
HC_TCPConn   *hc_tcp_conns_add(struct ip_hdr *iphdr, struct tcp_hdr *tcphdr);


/**
 * hc_tcp_conns_drop - drops a connection's state.
 * @src_addr: IP source address.
 * @src_port: TCP source port.
 * @dst_addr: IP source address.
 * @dst_port: TCP destination port.
 * 
 * The function drops the connection defined through the input parameters
 * from the state manager.
 */
void          hc_tcp_conns_drop(ip_addr_t src_addr, uint16_t src_port,
				ip_addr_t dst_addr, uint16_t dst_port);


/**
 * hc_tcp_conns_foreach - applies a callback to each TCP connection currently known.
 * @callback: callback to call.
 * @user_data: data passed to @callback.
 *
 * The function calls @callback with each TCP connection we're currently keeping
 * state for.
 */
void          hc_tcp_conns_foreach(HC_TCPConnCB callback, void *user_data);


/**
 * hc_tcp_conn_update_state - updates the state of a connection.
 * @conn: connection to update.
 * @iphdr: packet data to use.
 *
 * The function updates the state in @conn based on the information in
 * @iphdr (and the following TCP header), including stream reassembly etc.
 */
void          hc_tcp_conn_update_state(HC_TCPConn *conn, struct ip_hdr *iphdr);


/**
 * hc_tcp_conn_get_num_messages - returns number of messages sent back and forth.
 * @conn: conn to query.
 * 
 * The function returns the number of messages currently seen in the connection's
 * lifetime. A message is a chunk of data sent without real data being sent in
 * response (i.e. at most ACK packets).
 *
 * Returns: number of messages.
 */
u_int           hc_tcp_conn_get_num_messages(HC_TCPConn *conn);


/**
 * hc_tcp_conn_get_nth_message - returns nth message's data.
 * @conn: connection to query.
 * @num: message to look up.
 *
 * Returns: data sent in message @num in connection @conn,
 * or %NULL if @num messages haven't been sent yet.
 */
HC_Blob        *hc_tcp_conn_get_nth_message(HC_TCPConn *conn, u_int num);


/**
 * hc_tcp_conn_get_nth_message - returns nth message's data.
 * @conn: connection to query.
 * @num: message header to look up.
 *
 * Returns: packet headers that started message @num in connection @conn,
 * or %NULL if @num messages haven't been sent yet.
 */
struct ip_hdr  *hc_tcp_conn_get_nth_message_header(HC_TCPConn *conn, u_int num);

#endif
