/*

Copyright (C) 2003 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __hc_signature_hist_h
#define __hc_signature_hist_h

#include "honeycomb.h"
#include "hc_signature.h"


/**
 * HC_SigHistInsertCB - signature of functions called when a signature is put into memory.
 * @sig: signature that was put intot memory.
 * @user_data: user data passed in.
 *
 * Whenever a signature is successfully placed into our memory (and not
 * discarded as a duplicate of a signature already existing), a callback
 * of this signature is called.
 */
typedef void (*HC_SigHistInsertCB) (HC_Signature *sig, void *user_data);


/**
 * hc_sighist_init - initializes signature history.
 */
void          hc_sighist_init(void);

/**
 * hc_sighist_set_max_size - sets the maximum number of signatures remembered.
 * @max_size: new maximum size of signature memory.
 *
 * The function set the size of our memory to @max_size. If this is
 * smaller than a previously set value, the oldest signatures are dropped.
 */
void          hc_sighist_set_max_size(u_int max_size);

/**
 * hc_sighist_set_insert_callback - sets callback for successful insertions.
 * @callback: callback to call.
 * @user_data: data to pass to @callback.
 *
 * This sets the new callback to be called on successfull insertions.
 */
void          hc_sighist_set_insert_callback(HC_SigHistInsertCB callback,
					     void *user_data);


/**
 * hc_sighist_insert - inserts a signature into memory.
 * @sig: signature to put into memory.
 *
 * The function first checks if we already know a signature that
 * matches @sig, and if not, inserts it and calls the insertion handler
 * defined using hc_sighist_set_insert_callback().
 */
int           hc_sighist_insert(HC_Signature *sig);

#endif
