/*

Copyright (C) 2003 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/queue.h>

#include "hc_debug.h"
#include "hc_signature_hist.h"

#define HC_SIGHIST_DEFAULT_MAX_SIZE 100


typedef struct hc_sighist_item
{
	TAILQ_ENTRY(hc_sighist_item)    items;
	HC_Signature                   *sig;
} HC_SigHistItem;

TAILQ_HEAD(hc_sighist_tqh, hc_sighist_item);

static struct hc_sighist_tqh  sig_hist;
static HC_SigHistInsertCB     sig_insert_cb;
static void                  *sig_insert_cb_data;
static u_int                  sig_max_size;
static u_int                  sig_cur_size;


static HC_SigHistItem *
sighist_item_new(HC_Signature *sig)
{
	HC_Signature   *sig_copy;
	HC_SigHistItem *item;

	if (! (sig_copy = hc_sig_copy(sig))) {
		D(("Out of memory\n"));
		return NULL;
	}

	if (! (item = calloc(1, sizeof(HC_SigHistItem)))) {
		D(("Out of memory\n"));
		return NULL;
	}
	
	item->sig = sig_copy;
	
	return item;
}


static void
sighist_item_free(HC_SigHistItem *item)
{
	if (!item)
		return;

	if (item->sig)
		free(item->sig);
	free(item);
}


static void
sighist_drop_one(void)
{
	HC_SigHistItem *item;
	
	item = sig_hist.tqh_first;
	TAILQ_REMOVE(&sig_hist, item, items);
	sighist_item_free(item);
	sig_cur_size--;
}


void          
hc_sighist_init(void)
{
	TAILQ_INIT(&sig_hist);
	sig_max_size = HC_SIGHIST_DEFAULT_MAX_SIZE;
}


void          
hc_sighist_set_max_size(u_int max_size)
{
	if (max_size >= sig_max_size) {
		sig_max_size = max_size;
		return;
	}
	
	sig_max_size = max_size;
	
	while (sig_cur_size > sig_max_size)		
		sighist_drop_one();
}


void          
hc_sighist_set_insert_callback(HC_SigHistInsertCB callback,
			       void *user_data)
{
	if (!callback)
		return;
	
	sig_insert_cb = callback;
	sig_insert_cb_data = user_data;
}


int
hc_sighist_insert(HC_Signature *sig)
{
	HC_SigHistItem *item;

	if (!sig)
		return 0;
	
	for (item = sig_hist.tqh_first; item; item = item->items.tqe_next) {
		if (hc_sig_equal(item->sig, sig)) {

			item->sig->duplicates++;
			D(("Signature duplicate -- throwing away.\n"));
			return 0;
		}
	}

	/* This creates a history item with a *copy* of the signature */
	if (! (item = sighist_item_new(sig)))
		return 0;

	TAILQ_INSERT_TAIL(&sig_hist, item, items);
	sig_cur_size++;

	if (sig_insert_cb)
		sig_insert_cb(item->sig, sig_insert_cb_data);

	if (sig_cur_size > sig_max_size)
		sighist_drop_one();

	return 1;
}
