/*
 * Copyright 2002 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Niels Provos.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/param.h>

#include "config.h"

#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/tree.h>
#include <sys/queue.h>
#include <sys/socket.h>
#include <sys/resource.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <syslog.h>
#include <dnet.h>

#undef timeout_pending
#undef timeout_initialized

#include <event.h>

#include "honeyd.h"
#include "udp.h"

struct callback cb_udp = {
	cmd_udp_read, cmd_udp_write, cmd_udp_eread, cmd_udp_connect_cb
};

void
cmd_udp_eread(int fd, short which, void *arg)
{
	struct udp_con *con = arg;
	char line[1024];
	int nread;
	struct command *cmd = &con->cmd;

	nread = read(fd, line, sizeof(line));

	if (nread <= 0) {
		if (cmd->fdwantclose) {
			/* Stdin is already closed */
			udp_free(con);
		} else {
			/* Now stdin will takes us down */
			cmd->fdwantclose = 1;
		}
		return;
	}

	if (nread == sizeof(line))
		nread--;
	line[nread] = '\0';
	
	syslog(LOG_INFO, "E%s: %s", honeyd_contoa(&con->conhdr), line);

	event_add(&cmd->peread, NULL);
}

void
udp_add_readbuf(struct udp_con *con, u_char *dat, u_int datlen)
{
	struct buffer *buf;

	if (con->cmd_pfd == -1)
		return;

	if (con->nincoming >= MAX_UDP_BUFFERS)
		return;

	buf = malloc(sizeof(struct buffer));
	if (buf == NULL)
		return;
	buf->buf = malloc(datlen);
	if (buf->buf == NULL) {
		free(buf);
		return;
	}

	memcpy(buf->buf, dat, datlen);
	buf->len = datlen;

	TAILQ_INSERT_TAIL(&con->incoming, buf, next);
	con->nincoming++;

	cmd_trigger_write(&con->cmd, 1);
}

void
cmd_udp_read(int fd, short which, void *arg)
{
	struct udp_con *con = arg;
	struct command *cmd = &con->cmd;
	u_char buf[2048];
	ssize_t len;

	len = read(fd, buf, sizeof(buf));
	if (len == -1) {
		if (errno == EINTR || errno == EAGAIN)
			goto again;
		cmd_free(&con->cmd);
		return;
	} else if (len == 0) {
		if (cmd->perrfd != -1 && !cmd->fdwantclose) {
			cmd->fdwantclose = 1;
			return;
		}
		udp_free(con);
		return;
	}

	udp_send(con, buf, len);

 again:
	cmd_trigger_read(&con->cmd, 1);
}

void
cmd_udp_write(int fd, short which, void *arg)
{
	struct udp_con *con = arg;
	struct buffer *buf;
	ssize_t len;
	
	buf = TAILQ_FIRST(&con->incoming);
	if (buf == NULL)
		return;
	len = write(fd, buf->buf, buf->len);
	if (len == -1) {
		if (errno == EINTR || errno == EAGAIN)
			goto again;
		cmd_free(&con->cmd);
		return;
	} else if (len == 0) {
		cmd_free(&con->cmd);
		return;
	}

	TAILQ_REMOVE(&con->incoming, buf, next);
	con->nincoming--;

	free(buf->buf);
	free(buf);

 again:
	cmd_trigger_write(&con->cmd, TAILQ_FIRST(&con->incoming) != NULL);
}

void
cmd_udp_connect_cb(int fd, short which, void *arg)
{
	struct udp_con *con = arg;

	/* Everything is ready */
	cmd_ready_fd(&con->cmd, &cb_udp, con);

	cmd_trigger_read(&con->cmd, 1);
	cmd_trigger_write(&con->cmd, TAILQ_FIRST(&con->incoming) != NULL);
	return;
}
