/*
 * Copyright 2002 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Niels Provos.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/param.h>

#include "config.h"

#include <sys/queue.h>
#include <sys/tree.h>

#include <err.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <dnet.h>

#include "router.h"

SPLAY_HEAD(routetree, router) routers;

int
routercompare(struct router *a, struct router *b)
{
	return (addr_cmp(&a->addr, &b->addr));
}

SPLAY_PROTOTYPE(routetree, router, node, routercompare);

SPLAY_GENERATE(routetree, router, node, routercompare);

int router_used = 0;
struct addr router_entry;

void
router_init(void)
{
	SPLAY_INIT(&routers);
}

struct router *
router_find(struct addr *addr)
{
	struct router tmp;

	tmp.addr = *addr;
	return (SPLAY_FIND(routetree, &routers, &tmp));
}

struct router *
router_new(struct addr *addr)
{
	struct router *new;

	if (router_find(addr))
		return (NULL);

	if ((new = calloc(1, sizeof(struct router))) == NULL)
		err(1, "%s: calloc", __FUNCTION__);

	TAILQ_INIT(&new->routes);

	new->addr = *addr;

	SPLAY_INSERT(routetree, &routers, new);

	return (new);
}

int
router_start(struct addr *addr)
{
	struct router *new;

	if (router_used)
		return (-1);
	router_used = 1;
	router_entry = *addr;

	if ((new = router_new(addr)) == NULL)
		return (-1);

	return (0);
}

struct router_entry *
router_entry_new(struct addr *net, struct addr *gw, enum route_type type)
{
	struct router_entry *rte;

	if ((rte = calloc(1, sizeof(struct router_entry))) == NULL)
		err(1, "%s: calloc", __FUNCTION__);
	rte->net = *net;
	rte->type = type;
	if (type == ROUTE_NET)
		rte->gw = *gw;

	return (rte);
}

int
router_add_link(struct router *r, struct addr *addr)
{
	struct router_entry *rte;

	rte = router_entry_new(addr, NULL, ROUTE_LINK);
	TAILQ_INSERT_TAIL(&r->routes, rte, next);

	return (0);
}

int
router_add_net(struct router *r, struct addr *net, struct addr *gw,
    int latency, int packetloss)
{
	struct router_entry *rte;

	rte = router_entry_new(net, gw, ROUTE_NET);
	rte->latency = latency;
	rte->packetloss = packetloss;
	TAILQ_INSERT_TAIL(&r->routes, rte, next);

	return (0);
}

int
router_withinaddr(struct addr *net, struct addr *dst)
{
	struct addr addr;

	addr = *net;
	addr.addr_bits = IP_ADDR_BITS;
#ifdef HAVE_BROKEN_DNET
	if (addr_cmp(&addr, dst) < 0)
#else
	if (addr_cmp(&addr, dst) > 0)
#endif
		return (0);

	addr_bcast(net, &addr);
	addr.addr_bits = IP_ADDR_BITS;
#ifdef HAVE_BROKEN_DNET
	if (addr_cmp(&addr, dst) > 0)
#else
	if (addr_cmp(&addr, dst) < 0)
#endif
		return (0);

	return (1);
}

struct router_entry *
router_find_nexthop(struct router *r, struct addr *dst)
{
	struct router_entry *rte;

	TAILQ_FOREACH(rte, &r->routes, next) {
		if (router_withinaddr(&rte->net, dst))
			return (rte);
	}

	return (NULL);
}
