/*
 * Copyright 2002 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Niels Provos.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/param.h>
#include <sys/types.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <sys/param.h>
#include <sys/queue.h>
#include <dirent.h>

#include <ltdl.h>
#include "plugins.h"


/* A Honeyd plugin -- it can implement a few hooks to
 * give some info about itself, and most importantly an
 * init function to register itself against any hooks
 * in the system the plugin is interested in.
 */
struct honeyd_plugin
{
	LIST_ENTRY(honeyd_plugin) lst;
	
	const char *(*name) (void);
	const char *(*description) (void);
	const char *(*author) (void);
	const char *(*version) (void);
	
	int         (*init) (void);
	
	char        *filename;
	lt_dlhandle  lt;
};


/* All plugins are simply stored in a linked list, as
 * we don't have to access them very often -- any direct
 * interactions with the plugin are set up by the plugins
 * themselves via hooks as defined in hooks.h.
 */
LIST_HEAD(plugins_lh, honeyd_plugin) plugins;


/* The following are the dummy operations we always hook
 * into newly created plugins
 */
static const char      *
plugin_dummy_name(void)
{
	return "Unnamed plugin.";
}

static const char      *
plugin_dummy_description(void)
{
	return "No description given.";
}

static const char      *
plugin_dummy_author(void)
{
	return "No author(s) specified.";
}

static const char      *
plugin_dummy_version(void)
{
	return "No version specified.";
}

static int
plugin_dummy_init(void)
{
	return 0;
}


/**
 * plugin_new - creates new plugin structure.
 * @filename: the (directory-local) filename of the plugin.
 *
 * The function allocates memory for a new plugin and 
 * initializes the new plugin.
 *
 * Returns: new plugin, or %NULL when out of memory.
 */
struct honeyd_plugin *
plugin_new(const char *filename)
{
	struct honeyd_plugin *plugin;

	if (!filename || !*filename)
		return NULL;

	plugin = calloc(1, sizeof(struct honeyd_plugin));
	if (!plugin)
		return NULL;

	plugin->filename = strdup(filename);

	/* We hook in dummy callbacks for segfault protection */
	plugin->name        = plugin_dummy_name;
	plugin->description = plugin_dummy_description;
	plugin->author      = plugin_dummy_author;
	plugin->version     = plugin_dummy_version;
	plugin->init        = plugin_dummy_init;

	return plugin;	
}



/**
 * plugin_exists - checks whether a plugin with a given name is registered.
 * @filename: name to look up.
 *
 * Returns: value > 0 when a plugin with the given filename is
 * already registered, 0 otherwise.
 */
static int
plugin_exists(const char *filename)
{
	struct honeyd_plugin *plugin;

	for (plugin = plugins.lh_first; plugin; plugin = plugin->lst.le_next) {

		if (! strcmp(filename, plugin->filename))
			return 1;
	}

	return 0;
}



/**
 * plugin_hook_up - hooks in symbols from dlopenend modules.
 * @plugin: plugin to hook up.
 *
 * The function tries to hook the callback functions that
 * define a plugin's characteristics into the @plugin structure,
 * then initializes the plugin. Note that the callbacks are
 * all initialized with dummy operations so even if we encounter
 * a bad plugin, we should still be safe.
 */
static void       
plugin_hook_up(struct honeyd_plugin *plugin)
{
	char         s[MAXPATHLEN];
	lt_ptr       sym;

	snprintf(s, MAXPATHLEN, "%s/%s", PATH_HONEYDPLUGINS, plugin->filename);
	
	if ( (plugin->lt = lt_dlopenext(s)) == NULL) {
		fprintf(stderr, "dlopenext() on %s failed: %s\n", s, lt_dlerror());
		return;
	}
	
	if ((sym = lt_dlsym(plugin->lt, "name")) != NULL)
		plugin->name = sym;
	
	if ((sym = lt_dlsym(plugin->lt, "description")) != NULL)
		plugin->description = sym;
	
	if ((sym = lt_dlsym(plugin->lt, "author")) != NULL)
		plugin->author = sym;
	
	if ((sym = lt_dlsym(plugin->lt, "version")) != NULL)
		plugin->version = sym;
	
	if ((sym = lt_dlsym(plugin->lt, "init")) != NULL)
		plugin->init = sym;

	plugin->init();
	
	syslog(LOG_INFO, "registered plugin '%s' (%s)",
	       plugin->name(), plugin->version());
}


/**
 * plugin_scan_dir - scans a directory and initializes plugins found.
 * @dir: directory to scan.
 *
 * The function iterates over all files in the directory and registers
 * any plugins whose symbols can be successfully located.
 */
static void
plugin_scan_dir(const char *dir)
{
	DIR                  *dirp;
	struct dirent        *de;
	char                 *dot;
	struct honeyd_plugin *plugin;

	/* Open every file in the directory, skip files starting
	 * with a dot, and for the others cut off any extensions
	 * so that we
	 *
	 * - use each stem only once
	 * - can easily apply lt_dlopenext().
	 */
	
	if ((dirp = opendir(dir)) == NULL) {
		return;
	}
		
	for (de = readdir(dirp); de; de = readdir(dirp)) {
		
		if (de->d_name[0] == '.')
			continue;
		
		dot = strrchr(de->d_name, '.');			
		if (dot)
			*dot = '\0';
		
		if (! plugin_exists(de->d_name)) {

			plugin = plugin_new(de->d_name);
			LIST_INSERT_HEAD(&plugins, plugin, lst);

			plugin_hook_up(plugin);
		}
	}
	
	closedir(dirp);
}


void
plugins_init(void)
{
	LIST_INIT(&plugins);

	if (lt_dlinit() != 0)
		return;

	plugin_scan_dir(PATH_HONEYDPLUGINS);
}
