/*
 * Copyright 2002 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Niels Provos.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/param.h>
#include <sys/types.h>

#include "config.h"

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#include <sys/stat.h>
#include <sys/tree.h>
#include <sys/queue.h>

#include <err.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dnet.h>
#include <ctype.h>
#include <netdb.h>
#ifdef HAVE_TIME_H
#include <time.h>
#endif

#undef timeout_pending
#undef timeout_initialized

#include <event.h>

#include "honeyd.h"

static FILE *logfp;

static char *
honeyd_logtuple(struct tuple *hdr)
{
	static char buf[128];
	char asrc[24], adst[24];
	struct addr src, dst;
	ushort sport, dport;
	
	addr_pack(&src, ADDR_TYPE_IP, IP_ADDR_BITS, &hdr->ip_src, IP_ADDR_LEN);
	addr_pack(&dst, ADDR_TYPE_IP, IP_ADDR_BITS, &hdr->ip_dst, IP_ADDR_LEN);

	if (hdr->local) {
		struct addr tmp;

		tmp = src;
		src = dst;
		dst = tmp;
		sport = hdr->dport;
		dport = hdr->sport;
	} else {
		sport = hdr->sport;
		dport = hdr->dport;
	}


	addr_ntop(&src, asrc, sizeof(asrc));
	addr_ntop(&dst, adst, sizeof(adst));

	if (hdr->type == SOCK_STREAM || hdr->type == SOCK_DGRAM)
		snprintf(buf, sizeof(buf), "%s %d %s %d",
		    asrc, sport, adst, dport);
	else if (hdr->type == SOCK_RAW)
		snprintf(buf, sizeof(buf), "%s %s: %d(%d)",
		    asrc, adst, hdr->sport, hdr->dport);
	else
		snprintf(buf, sizeof(buf), "%s %s", asrc, adst);
		

	return (buf);
}

static char *
honeyd_logtime(void)
{
	static char logtime[32];
	struct timeval tv;
	struct tm *tm;
	time_t seconds;

	if (gettimeofday(&tv, NULL) == -1)
		err(1, "%s: gettimeofday", __func__);
	seconds = tv.tv_sec;
	
	/* ctime returns 26-character string */
	tm = localtime(&seconds);
	snprintf(logtime, sizeof(logtime),
	    "%04d-%02d-%02d-%02d:%02d:%02d.%04d",
	    tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday,
	    tm->tm_hour, tm->tm_min, tm->tm_sec,
	    (int)(tv.tv_usec / 1000));

	return (logtime);
}

static char *
honeyd_logproto(int proto)
{
	static char protoname[32];

	struct protoent *pe;
	struct protoent tcp = { "tcp", NULL, IP_PROTO_TCP };
	struct protoent udp = { "udp", NULL, IP_PROTO_UDP };
	struct protoent icmp = { "icmp", NULL, IP_PROTO_ICMP };

	switch(proto) {
	case IP_PROTO_TCP:
		pe = &tcp;
		break;
	case IP_PROTO_UDP:
		pe = &udp;
		break;
	case IP_PROTO_ICMP:
		pe = &icmp;
		break;
	default:
		/* Reads a file and is very slow */
		pe = getprotobynumber(proto);
		break;
	}

	if (pe == NULL)
		snprintf(protoname, sizeof(protoname), "unkn(%d)", proto);
	else
		snprintf(protoname, sizeof(protoname), "%s(%d)",
		    pe->p_name, proto);

	return (protoname);
}

#define TESTFLAG(x,y) do { \
	if (flags & (x)) \
		tcpflags[i++] = (y); \
} while (0)

static char *
honeyd_logtcpflags(int flags)
{
	static char tcpflags[10];
	int i = 0;

	TESTFLAG(TH_FIN, 'F');
	TESTFLAG(TH_SYN, 'S');
	TESTFLAG(TH_RST, 'R');
	TESTFLAG(TH_PUSH, 'P');
	TESTFLAG(TH_ACK, 'A');
	TESTFLAG(TH_URG, 'U');
	TESTFLAG(TH_ECE, 'E');
	TESTFLAG(TH_CWR, 'C');

	tcpflags[i] = '\0';

	return (tcpflags);
}

int
honeyd_logstart(char *filename)
{
	char *logtime;

	logfp = fopen(filename, "a");
	if (logfp == NULL) {
		warn("%s: fopen(\"%s\")", __func__, filename);
		return (-1);
	}

	/* Line buffered I/O */
	setvbuf(logfp, NULL, _IOLBF, 0);

	logtime = honeyd_logtime();
	fprintf(logfp, "%s honeyd packet log started ------\n", logtime);

	return (0);
}

void
honeyd_logend(void)
{
	char *logtime;

	if (logfp == NULL)
		return;

	logtime = honeyd_logtime();
	fprintf(logfp, "%s honeyd packet log stopped ------\n", logtime);

	fclose(logfp);
}

void
honeyd_log_probe(int proto, struct tuple *hdr, int size, int flags)
{
	if (logfp == NULL)
		return;

	fprintf(logfp, "%s %s - %s: %d %s\n",
	    honeyd_logtime(),
	    honeyd_logproto(proto),
	    honeyd_logtuple(hdr),
	    size,
	    proto == IP_PROTO_TCP ? honeyd_logtcpflags(flags) : "");
}

void
honeyd_log_flownew(int proto, struct tuple *hdr)
{
	char *tuple, *logtime, *protoname;

	if (logfp == NULL)
		return;

	logtime = honeyd_logtime();
	tuple = honeyd_logtuple(hdr);
	protoname = honeyd_logproto(proto);
	fprintf(logfp, "%s %s S %s\n", logtime, protoname, tuple);
}

void
honeyd_log_flowend(int proto, struct tuple *hdr)
{
	char *tuple, *logtime, *protoname;

	if (logfp == NULL)
		return;

	logtime = honeyd_logtime();
	tuple = honeyd_logtuple(hdr);
	protoname = honeyd_logproto(proto);
	fprintf(logfp, "%s %s E %s: %d %d\n", logtime, protoname, tuple,
	    hdr->received, hdr->sent);
}
