/*
 * Copyright 2002 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Niels Provos.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/param.h>
#include <sys/types.h>

#include "config.h"
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/tree.h>
#include <sys/wait.h>
#include <sys/queue.h>

#include <err.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <dnet.h>
#include <ctype.h>

#undef timeout_pending
#undef timeout_initialized

#include <event.h>

#include "honeyd.h"
#include "personality.h"
#include "ipfrag.h"

u_char buf[65536];  /* for complete packet */

SPLAY_HEAD(fragtree, fragment) fragments;

#define DIFF(a,b) do { \
	if ((a) < (b)) return -1; \
	if ((a) > (b)) return 1; \
} while (0)

int
fragcompare(struct fragment *a, struct fragment *b)
{
	DIFF(a->ip_src, b->ip_src);
	DIFF(a->ip_dst, b->ip_dst);
	DIFF(a->ip_id, b->ip_id);

	return (0);
}

SPLAY_PROTOTYPE(fragtree, fragment, node, fragcompare);

SPLAY_GENERATE(fragtree, fragment, node, fragcompare);

TAILQ_HEAD(fragqueue, fragment) fraglru;

int nfragments;
int nfragmem;

void
ip_fragment_init(void)
{
	SPLAY_INIT(&fragments);
	TAILQ_INIT(&fraglru);

	nfragments = 0;
	nfragmem = 0;
}

struct fragment *
ip_fragment_find(ip_addr_t src, ip_addr_t dst, u_short id)
{
	struct fragment tmp, *frag;

	tmp.ip_src = src;
	tmp.ip_dst = dst;
	tmp.ip_id = id;

	frag = SPLAY_FIND(fragtree, &fragments, &tmp);

	if (frag != NULL) {
		TAILQ_REMOVE(&fraglru, frag, next);
		TAILQ_INSERT_HEAD(&fraglru, frag, next);
	}

	return (frag);
}

/* Free a fragment by removing it from all lists, etc... */
void
ip_fragent_free(struct fragent *ent)
{
	nfragmem -= ent->size;

	free(ent->data);
	free(ent);
}

void
ip_fragment_free(struct fragment *tmp)
{
	struct fragent *ent;

	timeout_del(&tmp->timeout);

	SPLAY_REMOVE(fragtree, &fragments, tmp);
	TAILQ_REMOVE(&fraglru, tmp, next);
	nfragments--;

	for (ent = TAILQ_FIRST(&tmp->fraglist); ent;
	    ent = TAILQ_FIRST(&tmp->fraglist)) {
		TAILQ_REMOVE(&tmp->fraglist, ent, next);

		ip_fragent_free(ent);
	}
	free(tmp);
}

void
ip_fragment_timeout(int fd, short which, void *arg)
{
	struct fragment *tmp = arg;
	struct addr src;

	addr_pack(&src, ADDR_TYPE_IP, IP_ADDR_BITS, &tmp->ip_src, IP_ADDR_LEN);

	syslog(LOG_DEBUG, "Expiring fragment from %s, id %d",
	    addr_ntoa(&src), ntohs(tmp->ip_id));

	ip_fragment_free(tmp);
}

void
ip_fragment_reclaim(int count)
{
	struct fragment *tmp;
	for (tmp = TAILQ_LAST(&fraglru, fragqueue); tmp && count;
	    tmp = TAILQ_LAST(&fraglru, fragqueue)) {
		ip_fragment_free(tmp);
		count--;
	}
}

struct fragment *
ip_fragment_new(ip_addr_t src, ip_addr_t dst, u_short id, enum fragpolicy pl)
{
	struct fragment *tmp = NULL;
	struct timeval tv = { IPFRAG_TIMEOUT, 0};
	int reclaim = 0;

	if (nfragmem > IPFRAG_MAX_MEM || nfragments > IPFRAG_MAX_FRAGS)
		ip_fragment_reclaim(nfragments/10);

	while (tmp == NULL && reclaim < 2) {
		tmp = calloc(1, sizeof(struct fragment));
		if (tmp == NULL) {
			reclaim++;
			ip_fragment_reclaim(nfragments/10);
		}
	}

	if (tmp == NULL)
		return (NULL);

	tmp->ip_src = src;
	tmp->ip_dst = dst;
	tmp->ip_id = id;
	tmp->fragp = pl;

	TAILQ_INIT(&tmp->fraglist);
	timeout_set(&tmp->timeout, ip_fragment_timeout, tmp);
	timeout_add(&tmp->timeout, &tv);

	SPLAY_INSERT(fragtree, &fragments, tmp);
	TAILQ_INSERT_HEAD(&fraglru, tmp, next);
	nfragments++;

	return (tmp);
}

int
ip_fragment_insert(struct fragment *fragq, struct fragent *ent, short mf)
{
	struct fragent *prev, *after;
	struct ip_hdr *ip;
	u_char *data;
	u_short overlap;
	u_short max;
	u_short off;
	u_short len;

	off = ent->off;
	len = ent->len;
	max = off + len;

	if (fragq->maxlen < max)
		fragq->maxlen = max;
	if (!mf)
		fragq->hadlastpacket = 1;

	prev = NULL;
	for (after = TAILQ_FIRST(&fragq->fraglist); after;
	    after = TAILQ_NEXT(after, next)) {
		if (off < after->off)
			break;
		prev = after;
	}

	if (prev && prev->off + prev->len > off) {
		overlap = prev->off + prev->len - off;

		if (overlap >= len) {
			if (fragq->fragp == FRAG_NEW) {
				u_char *odata = prev->data + off - prev->off;
				memcpy(odata, ent->data, len);
			}
			goto free_fragment;
		}

		if (fragq->fragp == FRAG_OLD) {
			u_char *odata = prev->data + prev->len - overlap;
			memcpy(ent->data, odata, overlap);
		}
		prev->len -= overlap;
	}

	if (after && off + len > after->off) {
		overlap = off + len - after->off;

		if (overlap > after->len) {
			if (fragq->fragp == FRAG_OLD) {
				u_char *ndata = ent->data + after->off - off;
				memcpy(ndata, after->data, after->len);
			}
			ent = after;
			goto drop_fragment;
		}
		if (fragq->fragp == FRAG_NEW) {
			u_char *ndata = ent->data + len - overlap;
			memcpy(after->data, ndata, overlap);
		}
		len -= overlap;
		ent->len = len;
	}

	if (prev)
		TAILQ_INSERT_AFTER(&fragq->fraglist, prev, ent, next);
	else
		TAILQ_INSERT_HEAD(&fragq->fraglist, ent, next);

	/* Waiting for more data */
	if (!fragq->hadlastpacket)
		return (0);

	off = 0;
	TAILQ_FOREACH(ent, &fragq->fraglist, next) {
		if (ent->off != off)
			break;
		off = ent->off + ent->len;
	}

	if (ent)
		return (0);

	/* Completely assembled */

	data = buf;
	ip = (struct ip_hdr *)data;
	for(ent = TAILQ_FIRST(&fragq->fraglist); ent;
	    ent = TAILQ_FIRST(&fragq->fraglist)) {
		TAILQ_REMOVE(&fragq->fraglist, ent, next);

		memcpy(data, ent->data, ent->len);
		data += ent->len;
		ip_fragent_free(ent);
	}

	ip->ip_len = htons(fragq->maxlen);
	ip->ip_off = 0;

	ip_fragment_free(fragq);

	return (1);

 drop_fragment:
	TAILQ_REMOVE(&fragq->fraglist, ent, next);
 free_fragment:
	ip_fragent_free(ent);
	return (0);
}

void
ip_fragment(struct template *tmpl, struct ip_hdr *ip)
{
	struct addr src;
	struct personality *person;
	struct fragment *fragq;
	struct fragent *ent;
	u_char *dat;
	short mf;
	u_short off;
	u_short len;
	u_short hlen;
	
	addr_pack(&src, ADDR_TYPE_IP, IP_ADDR_BITS, &ip->ip_src, IP_ADDR_LEN);

	if (tmpl == NULL)
		goto drop;
	person = tmpl->person;
	if (person == NULL)
		goto drop;

	if (person->fragp == FRAG_DROP)
		goto drop;

	fragq = ip_fragment_find(ip->ip_src, ip->ip_dst, ip->ip_id);

	/* Nothing here for now */
	off = ntohs(ip->ip_off);
	if (off & IP_DF)
		goto freeall;
	mf = off & IP_MF;
	off &= IP_OFFMASK;
	off <<= 3;

	dat = (u_char *)ip;
	len = ntohs(ip->ip_len);
	hlen = ip->ip_hl << 2;
	if (mf && ((len - hlen) & 0x7))
		goto freeall;

	if (off) {
		len -= hlen;
		dat += hlen;
		off += hlen;
	}

	if (off + len > IP_LEN_MAX || len == 0)
		goto freeall;

	if (fragq == NULL) {
		fragq = ip_fragment_new(ip->ip_src, ip->ip_dst, ip->ip_id,
		    person->fragp);
		if (fragq == NULL)
			goto drop;
	}

	if ((ent = calloc(1, sizeof(struct fragent))) == NULL)
		goto freeall;

	ent->off = off;
	ent->len = len;
	ent->size = len;
	if ((ent->data = malloc(len)) == NULL) {
		free(ent);
		goto freeall;
	}
	memcpy(ent->data, dat, len);
	nfragmem += len;

	syslog(LOG_DEBUG,  "Received fragment from %s, id %d: %d@%d\n",
	    addr_ntoa(&src), ntohs(ip->ip_id), len, off);

	if (ip_fragment_insert(fragq, ent, mf))
		honeyd_dispatch(tmpl, (struct ip_hdr *)buf);
	return;

 freeall:
	syslog(LOG_INFO,  "%s fragment from %s, id %d: %d@%d\n",
	    fragq ? "Freeing" : "Dropping",
	    addr_ntoa(&src), ntohs(ip->ip_id), len, off);

	if (fragq)
		ip_fragment_free(fragq);
	return;

 drop:
	syslog(LOG_INFO, "Dropping fragment from %s", addr_ntoa(&src));

	return;
}
