/*
 * Copyright 2002 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Niels Provos.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/param.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/tree.h>
#include <sys/queue.h>
#include <sys/socket.h>
#include <sys/resource.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <syslog.h>
#include <grp.h>
#include <dnet.h>

#undef timeout_pending
#undef timeout_initialized

#include <event.h>

#include "honeyd.h"
#include "personality.h"
#include "subsystem.h"
#include "tcp.h"
#include "udp.h"
#include "fdpass.h"

ssize_t atomicio(ssize_t (*)(), int, void *, size_t);

extern struct callback cb_tcp;
extern struct callback cb_udp;

void
cmd_trigger_read(struct command *cmd, int size)
{
 	if (cmd->pfd == -1 || !cmd->fdconnected)
		return;
	if (size)
		event_add(&cmd->pread, NULL);
}

void
cmd_trigger_write(struct command *cmd, int size)
{
 	if (cmd->pfd == -1 || !cmd->fdconnected)
		return;
	if (size)
		event_add(&cmd->pwrite, NULL);
}

void
cmd_free(struct command *cmd)
{
	event_del(&cmd->pread);
	event_del(&cmd->pwrite);
	close(cmd->pfd);
	cmd->pfd = -1;

	if (cmd->perrfd != -1) {
		event_del(&cmd->peread);
		close(cmd->perrfd);
		cmd->perrfd = -1;
	}
}

void
cmd_ready_fd(struct command *cmd, struct callback *cb, void *con)
{
	event_set(&cmd->pread, cmd->pfd, EV_READ, cb->cb_read, con);
	event_set(&cmd->pwrite, cmd->pfd, EV_WRITE, cb->cb_write, con);
	cmd->fdconnected = 1;

	if (cmd->perrfd != -1)
		event_set(&cmd->peread, cmd->perrfd, EV_READ, cb->cb_eread,
		    con);
}

struct addrinfo *
cmd_proxy_getinfo(char *address, int type, short port)
{
	struct addrinfo ai, *aitop;
        char strport[NI_MAXSERV];

        memset(&ai, 0, sizeof (ai));
        ai.ai_family = AF_INET;
        ai.ai_socktype = type;
        ai.ai_flags = 0;
        snprintf(strport, sizeof (strport), "%d", port);
        if (getaddrinfo(address, strport, &ai, &aitop) != 0) {
                warn("getaddrinfo");
                return (NULL);
        }

	return (aitop);
}

int
cmd_proxy_connect(struct tuple *hdr, struct command *cmd, struct addrinfo *ai,
    void *con)
{
	char ntop[NI_MAXHOST], strport[NI_MAXSERV];
	char *host = ntop, *port = strport;
	struct callback *cb;
	struct timeval tv = {10, 0};
        int fd;
        
	if (hdr->type == SOCK_STREAM)
		cb = &cb_tcp;
	else
		cb = &cb_udp;

        fd = socket(AF_INET, hdr->type, 0);
        if (fd == -1) {
                warn("socket");
                return (-1);
        }

        if (fcntl(fd, F_SETFL, O_NONBLOCK) == -1)
                warn("fcntl(O_NONBLOCK)");

        if (fcntl(fd, F_SETFD, 1) == -1)
                warn("fcntl(F_SETFD)");

	cmd->pfd = fd;
        if (connect(fd, ai->ai_addr, ai->ai_addrlen) == 0) {
		(*cb->cb_connect)(fd, EV_WRITE, con);
		return (0);
	}

	if (errno != EINPROGRESS) {
		warn("connect");
		cmd->pfd = -1;
		close(fd);
		return (-1);
	}

	event_set(&cmd->pwrite, fd, EV_WRITE, cb->cb_connect, con);
	event_add(&cmd->pwrite, &tv);

	if (getnameinfo(ai->ai_addr, ai->ai_addrlen,
		ntop, sizeof(ntop), strport, sizeof(strport),
		NI_NUMERICHOST|NI_NUMERICSERV) != 0) {

		host = "<hosterror>";
		port = "<porterror>";
	}
	syslog(LOG_INFO, "Connection established: %s -> proxy to %s:%s",
	    honeyd_contoa(hdr), host, port);

	return (0);
}

void
cmd_environment(struct template *tmpl, struct tuple *hdr)
{
	char line[256];
	struct addr addr;

	if (tmpl->person != NULL) {
		snprintf(line, sizeof(line), "%s", tmpl->person->name);
		setenv("HONEYD_PERSONALITY", line, 1);
	}

	if (hdr == NULL)
		return;
	     
	addr_pack(&addr, ADDR_TYPE_IP, IP_ADDR_BITS, &hdr->ip_src,IP_ADDR_LEN);
	snprintf(line, sizeof(line), "%s", addr_ntoa(&addr));
	setenv("HONEYD_IP_SRC", line, 1);

	addr_pack(&addr, ADDR_TYPE_IP, IP_ADDR_BITS, &hdr->ip_dst,IP_ADDR_LEN);
	snprintf(line, sizeof(line), "%s", addr_ntoa(&addr));
	setenv("HONEYD_IP_DST", line, 1);

	snprintf(line, sizeof(line), "%d", hdr->sport);
	setenv("HONEYD_SRC_PORT", line, 1);

	snprintf(line, sizeof(line), "%d", hdr->dport);
	setenv("HONEYD_DST_PORT", line, 1);
}

int
cmd_setpriv(struct template *tmpl)
{
	uid_t uid = 32767;
	gid_t gid = 32767;
	struct rlimit rl;

	if (tmpl->uid)
		uid = tmpl->uid;
	if (tmpl->gid)
		gid = tmpl->gid;

	/* Lower privileges */
#ifdef HAVE_SETGROUPS
	setgroups(1, &gid);
#endif
	setegid(gid);
	setgid(gid);
	seteuid(uid);
	setuid(uid);

	/* Raising file descriptor limits */
	rl.rlim_cur = rl.rlim_max = 24;
	if (setrlimit(RLIMIT_NOFILE, &rl) == -1)
		err(1, "setrlimit");

	return (0);
}

int
cmd_fork(struct tuple *hdr, struct command *cmd, struct template *tmpl,
    char *execcmd, char **argv, void *con)
{
	extern int honeyd_nchildren;
	int pair[2], perr[2];
	struct callback *cb;
	sigset_t sigmask;

	if (socketpair(AF_UNIX, hdr->type, 0, pair) == -1)
		return (-1);
	if (socketpair(AF_UNIX, SOCK_STREAM, 0, perr) == -1) {
		close(pair[0]);
		close(pair[1]);
		return (-1);
	}

	/* Block SIGCHLD */
	sigemptyset(&sigmask);
	sigaddset(&sigmask, SIGCHLD);
	if (sigprocmask(SIG_BLOCK, &sigmask, NULL) == -1) {
		warn("sigprocmask");
		goto fork_err;
	}

	cmd->pid = fork();
	if (cmd->pid == -1) {
		warn("fork");
		goto unmask_err;
	}

	if (cmd->pid == 0) {
		/* Child privileges */
		cmd_setpriv(tmpl);

		/* Child */
		close(pair[0]);
		if (dup2(pair[1], fileno(stdout)) == -1)
			err(1, "%s: dup2", __func__);
		if (dup2(pair[1], fileno(stdin)) == -1)
			err(1, "%s: dup2", __func__);
		close(perr[0]);
		if (dup2(perr[1], fileno(stderr)) == -1)
			err(1, "%s: dup2", __func__);

		close(pair[1]);
		close(perr[1]);

		cmd_environment(tmpl, hdr);

		if (execv(execcmd, argv) == -1)
			err(1, "%s: execv(%s)", __func__, execcmd);

		/* NOT REACHED */
	}

	close(pair[1]);
	cmd->pfd = pair[0];
	if (fcntl(cmd->pfd, F_SETFD, 1) == -1)
		warn("fcntl(F_SETFD)");
	if (fcntl(cmd->pfd, F_SETFL, O_NONBLOCK) == -1)
		warn("fcntl(F_SETFL)");

	close(perr[1]);
	cmd->perrfd = perr[0];
	if (fcntl(cmd->perrfd, F_SETFD, 1) == -1)
		warn("fcntl(F_SETFD)");
	if (fcntl(cmd->perrfd, F_SETFL, O_NONBLOCK) == -1)
		warn("fcntl(F_SETFL)");

	if (hdr->type == SOCK_STREAM)
		cb = &cb_tcp;
	else
		cb = &cb_udp;

	cmd_ready_fd(cmd, cb, con);

	event_add(&cmd->pread, NULL);
	event_add(&cmd->peread, NULL);

	honeyd_nchildren++;

	/* Install old signal handler */
	if (sigprocmask(SIG_UNBLOCK, &sigmask, NULL) == -1) {
		warn("sigprocmask");
		goto fork_err;
	}
	return (0);

	/* Error cleanup */
 unmask_err:
	/* Install old signal handler */
	if (sigprocmask(SIG_UNBLOCK, &sigmask, NULL) == -1)
		warn("sigprocmask");

 fork_err:
	close(perr[0]);
	close(perr[1]);
	close(pair[0]);
	close(pair[1]);
	cmd->pfd = -1;

	return (-1);
}

int
cmd_subsystem(struct template *tmpl, struct subsystem *sub,
    char *execcmd, char **argv)
{
	extern int honeyd_nchildren;
	struct command *cmd = &sub->cmd;
	extern struct callback subsystem_cb;
	int pair[2];
	sigset_t sigmask;

	if (socketpair(AF_UNIX, SOCK_STREAM, 0, pair) == -1)
		return (-1);

	/* Block SIGCHLD */
	sigemptyset(&sigmask);
	sigaddset(&sigmask, SIGCHLD);
	if (sigprocmask(SIG_BLOCK, &sigmask, NULL) == -1) {
		warn("sigprocmask");
		goto fork_err;
	}

	cmd->pid = fork();
	if (cmd->pid == -1) {
		warn("fork");
		goto unmask_err;
	}

	if (cmd->pid == 0) {
		/* Set privileges */
		cmd_setpriv(tmpl);

		/* Child */
		close(pair[0]);
		if (dup2(pair[1], SUBSYSTEM_MAGICFD) == -1)
			err(1, "%s: dup2", __func__);
		if (dup2(fileno(stderr), fileno(stdout)) == -1)
			err(1, "%s: dup2", __func__);
		if (dup2(fileno(stderr), fileno(stdin)) == -1)
			err(1, "%s: dup2", __func__);

		close(pair[1]);

		cmd_environment(tmpl, NULL);

		/* Setup the wrapper library */
		if (setenv("LD_PRELOAD", PATH_HONEYDLIB"/libhoneyd.so", 1) == -1)
			err(1, "%s: setenv", __func__);

		if (execv(execcmd, argv) == -1)
			err(1, "%s: execv(%s)", __func__, execcmd);

		/* NOT REACHED */
	}

	close(pair[1]);
	cmd->pfd = pair[0];
	if (fcntl(cmd->pfd, F_SETFD, 1) == -1)
		warn("fcntl(F_SETFD)");
	if (fcntl(cmd->pfd, F_SETFL, O_NONBLOCK) == -1)
		warn("fcntl(F_SETFL)");

	cmd->perrfd = -1;
	cmd_ready_fd(cmd, &subsystem_cb, sub);

	event_add(&cmd->pread, NULL);

	honeyd_nchildren++;

	/* Install old signal handler */
	if (sigprocmask(SIG_UNBLOCK, &sigmask, NULL) == -1) {
		warn("sigprocmask");
		goto fork_err;
	}
	return (0);

	/* Error cleanup */
 unmask_err:
	/* Install old signal handler */
	if (sigprocmask(SIG_UNBLOCK, &sigmask, NULL) == -1)
		warn("sigprocmask");

 fork_err:
	close(pair[0]);
	close(pair[1]);
	cmd->pfd = -1;

	return (-1);
}

int
cmd_subsystem_connect(struct tuple *hdr, struct command *cmd,
    struct port *port, void *con)
{
	struct callback *cb;
	struct subsystem *sub = port->sub;
	struct sockaddr_in si;
	struct addr src;
	int pair[2];
        
	if (hdr->type == SOCK_STREAM)
		cb = &cb_tcp;
	else
		cb = &cb_udp;

        if (socketpair(AF_LOCAL, hdr->type, 0, pair) == -1) {
                warn("socketpair");
                return (-1);
        }

        if (fcntl(pair[0], F_SETFL, O_NONBLOCK) == -1)
                warn("fcntl(O_NONBLOCK)");

        if (fcntl(pair[0], F_SETFD, 1) == -1)
                warn("fcntl(F_SETFD)");

	cmd->pfd = pair[0];

	/* Prepare sockaddr */
	addr_pack(&src, ADDR_TYPE_IP, IP_ADDR_BITS, &hdr->ip_src,IP_ADDR_LEN);
	addr_ntos(&src, (struct sockaddr *)&si);
	si.sin_port = htons(hdr->sport);

	if (send_fd(port->sub_fd, pair[1], &si, sizeof(si)) == -1)
		return (-1);

	/* After transfering the file descriptor, we may close it */
	close(pair[1]);

	/* We are connected now */
	(*cb->cb_connect)(pair[0], EV_WRITE, con);

	syslog(LOG_INFO, "Connection established: %s -> subsystem \"%s\"",
	    honeyd_contoa(hdr), sub->cmdstring);

	return (0);
}

int
cmd_subsystem_localconnect(struct tuple *hdr, struct command *cmd,
    struct port *port, void *con)
{
	struct callback *cb;
	struct subsystem *sub = port->sub;
	struct sockaddr_in si;
	struct addr src;
	int fd;
        
	if (hdr->type == SOCK_STREAM)
		cb = &cb_tcp;
	else
		cb = &cb_udp;

	if (port->sub_fd == -1) {
		char res;

		while ((fd = receive_fd(sub->cmd.pfd, NULL, NULL)) == -1) {
			if (errno != EAGAIN) {
				warnx("%s: receive_fd", __func__);
			}
		}

		/* Confirm success of failure */
		res = fd == -1 ? -1 : 0;
		atomicio(write, sub->cmd.pfd, &res, 1);
		if (fd == -1)
			return (-1);

		port->sub_fd = fd;
	}

	/* Get another fd on this special thingy */
	while ((fd = receive_fd(port->sub_fd, NULL, NULL)) == -1) {
		if (errno != EAGAIN) {
			warnx("%s: receive_fd", __func__);
			return (-1);
		}
	}

        if (fcntl(fd, F_SETFL, O_NONBLOCK) == -1)
                warn("%s: fcntl(O_NONBLOCK)", __func__);

        if (fcntl(fd, F_SETFD, 1) == -1)
                warn("%s: fcntl(F_SETFD)", __func__);

	cmd->pfd = fd;

	/* Prepare sockaddr */
	addr_pack(&src, ADDR_TYPE_IP, IP_ADDR_BITS, &hdr->ip_dst, IP_ADDR_LEN);
	addr_ntos(&src, (struct sockaddr *)&si);
	si.sin_port = htons(hdr->dport);

	/* Write the bound socket address to the other side */
	if (atomicio(write, port->sub_fd, &si, sizeof(si)) != sizeof(si))
		return (-1);

	/* Now we may close the special thingy */
	close(port->sub_fd);
	port->sub_fd = -1;

	/* We are connected now */
	(*cb->cb_connect)(fd, EV_WRITE, con);

	syslog(LOG_INFO, "Connection established: subsystem \"%s\" -> %s",
	    sub->cmdstring, honeyd_contoa(hdr));

	return (0);
}
