/*
 * Copyright 2002 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Niels Provos.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>

#include "config.h"
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifndef WIN32
#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/resource.h>
#include <unistd.h>
#endif
#include <sys/stat.h>
#include <sys/tree.h>
#include <sys/queue.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dnet.h>
#include <syslog.h>

#undef timeout_pending
#undef timeout_initialized

#include <event.h>

#include "honeyd.h"
#include "udp.h"

struct callback cb_udp = {
	cmd_udp_read, cmd_udp_write, cmd_udp_eread, cmd_udp_connect_cb
};

void
cmd_udp_eread(int fd, short which, void *arg)
{
	struct udp_con *con = arg;
	char line[1024];
	int nread;
	struct command *cmd = &con->cmd;

#ifdef WIN32
	BOOL fRet;

	fRet = PeekNamedPipe((HANDLE)fd, NULL, 0, NULL, &nread, NULL);
	if(!fRet) {
		cmd_free(&con->cmd);
		return;
	}

	if(!nread) {
		event_add(&cmd->peread, NULL);
		return;
	}

	ReadFile((HANDLE)fd, line, sizeof(line), &nread, NULL);
#else
	nread = read(fd, line, sizeof(line));
#endif

#ifdef WIN32
	if(GetLastError() != ERROR_NO_DATA && !nread) {
#else
	if (nread <= 0) {
#endif
		if (cmd->fdwantclose) {
			/* Stdin is already closed */
			udp_free(con);
		} else {
			/* Now stdin will takes us down */
			cmd->fdwantclose = 1;
		}
		return;
	}

	if (nread == sizeof(line))
		nread--;
	line[nread] = '\0';
	
	syslog(LOG_INFO, "E%s: %s", honeyd_contoa(&con->conhdr), line);

	event_add(&cmd->peread, NULL);
}

void
udp_add_readbuf(struct udp_con *con, u_char *dat, u_int datlen)
{
	struct buffer *buf;

	if (con->cmd_pfd == INVALID_SOCKET)
		return;

	if (con->nincoming >= MAX_UDP_BUFFERS)
		return;

	buf = malloc(sizeof(struct buffer));
	if (buf == NULL)
		return;
	buf->buf = malloc(datlen);
	if (buf->buf == NULL) {
		free(buf);
		return;
	}

	memcpy(buf->buf, dat, datlen);
	buf->len = datlen;

	TAILQ_INSERT_TAIL(&con->incoming, buf, next);
	con->nincoming++;

	cmd_trigger_write(&con->cmd, 1);
}

void
cmd_udp_read(int fd, short which, void *arg)
{
	struct udp_con *con = arg;
	struct command *cmd = &con->cmd;
	u_char buf[2048];
	ssize_t len;
#ifdef WIN32
	BOOL fRet;
	int iError = 0;

	if(!cmd->IsProxyCmd) {
		fRet = PeekNamedPipe((HANDLE)fd, NULL, 0, NULL, &len, NULL);
		if(!fRet) {
			cmd_free(&con->cmd);
			return;
		}

		if(!len)
			goto again;

		ReadFile((HANDLE)fd, buf, sizeof(buf), &len, NULL);
		iError = GetLastError();
	} else {
		len = recv(cmd->pfd, buf, sizeof(buf), 0);
		if(len == SOCKET_ERROR)
			iError = ERROR_NO_DATA;
	}

	if(iError != ERROR_NO_DATA && !len) {
#else
	len = read(fd, buf, sizeof(buf));

	if (len == SOCKET_ERROR) {
#endif
#ifdef WIN32
		if (errno == EINTR || errno == EAGAIN || GetLastError() == ERROR_NO_DATA)
#else
		if (errno == EINTR || errno == EAGAIN)
#endif
			goto again;
		cmd_free(&con->cmd);
		return;
	} else if (len == 0) {
		if (cmd->perrfd != INVALID_SOCKET && !cmd->fdwantclose) {
			cmd->fdwantclose = 1;
			return;
		}
		udp_free(con);
		return;
	}

	udp_send(con, buf, len);

 again:
	cmd_trigger_read(&con->cmd, 1);
}

void
cmd_udp_write(int fd, short which, void *arg)
{
	struct udp_con *con = arg;
	struct buffer *buf;
	ssize_t len;
	
	buf = TAILQ_FIRST(&con->incoming);
	if (buf == NULL)
		return;
#ifdef WIN32
	if(!WriteFile((HANDLE)fd, buf->buf, buf->len, &len, NULL)) {
#else
	len = write(fd, buf->buf, buf->len);

	if (len == SOCKET_ERROR) {
#endif
		if (errno == EINTR || errno == EAGAIN)
			goto again;
		cmd_free(&con->cmd);
		return;
	} else if (len == 0) {
		cmd_free(&con->cmd);
		return;
	}

	TAILQ_REMOVE(&con->incoming, buf, next);
	con->nincoming--;

	free(buf->buf);
	free(buf);

 again:
	cmd_trigger_write(&con->cmd, TAILQ_FIRST(&con->incoming) != NULL);
}

void
cmd_udp_connect_cb(int fd, short which, void *arg)
{
	struct udp_con *con = arg;

	/* Everything is ready */
	cmd_ready_fd(&con->cmd, &cb_udp, con);

	cmd_trigger_read(&con->cmd, 1);
	cmd_trigger_write(&con->cmd, TAILQ_FIRST(&con->incoming) != NULL);
	return;
}
