/*
 * Copyright 2002 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Niels Provos.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
%{
#include <sys/types.h>

#include "config.h"

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#include <sys/tree.h>
#include <sys/queue.h>

#include <stdio.h>
#include <stdlib.h>
#include <err.h>
#include <stdarg.h>
#include <string.h>

#include <dnet.h>

#include <event.h>

#include "honeyd.h"
#include "personality.h"
#include "router.h"

int yylex(void);
int yyparse(void);
int yyerror(char *, ...);

struct template;
int template_create(char *);
int template_add(struct template *, int, int, struct action *);
int template_subsystem(struct template *, char *);
int template_clone(char *, struct template *, int);
struct template *template_find(char *);

struct action *honeyd_protocol(struct template *, int);

void port_action_clone(struct action *, struct action *);

int lineno;
char *filename;
int errors = 0;
int curtype = -1;	/* Lex sets it to SOCK_STREAM or _DGRAM */

%}

%token	CREATE ADD PORT BIND CLONE DOT BLOCK OPEN RESET DEFAULT SET ACTION
%token	PERSONALITY RANDOM ANNOTATE NO FINSCAN FRAGMENT DROP OLD NEW COLON
%token	PROXY UPTIME DROPRATE IN UID GID ROUTE ENTRY LINK NET SLASH
%token	LATENCY MS LOSS SUBSYSTEM
%token	<string> STRING
%token	<string> CMDSTRING
%token	<number> NUMBER
%token	<number> PROTO
%type	<addr> ipaddr
%type	<addr> ipnet
%type	<ai> ipaddrplusport
%type	<action> action
%type	<tmpl> template
%type	<pers> personality
%type	<number> finscan;
%type	<fragp> fragment;
%type	<floatp> rate;
%type	<number> latency;
%type	<number> packetloss;
%union {
	char *string;
	int number;
	struct addr addr;
	struct action action;
	struct template *tmpl;
	struct personality *pers;
	struct addrinfo *ai;
	enum fragpolicy fragp;
	float floatp;
}
%%

config		: /* empty */
		| config creation
		| config addition
		| config subsystem
		| config binding
		| config set
		| config annotate
		| config route
		;

creation	: CREATE STRING
	{
		if (template_create($2) == -1)
			yyerror("Template \"%s\" exists already", $2);
		free($2);
	}
		| CREATE DEFAULT
	{
		if (template_create("default") == -1)
			yyerror("Template \"default\" exists already");
	}
;

addition	: ADD template PROTO PORT NUMBER action
	{
		if ($2 != NULL && template_add($2, $3, $5, &$6) == -1)
			yyerror("Can not add port %d to template \"%s\"",
			    $5, $2 != NULL ? $2->name : "<unknown>");
		if ($6.action)
			free($6.action);
	}
;
subsystem	: ADD template SUBSYSTEM CMDSTRING
	{
		$4[strlen($4) - 1] = '\0';
		if ($2 != NULL && template_subsystem($2, $4+1) == -1)
			yyerror("Can not add subsystem \"%s\" to template \"%s\"",
			    $4+1, $2 != NULL ? $2->name : "<unknown>");
		free($4);
	}
;
binding		: BIND ipaddr template
	{
		/* Bind to an IP address and start subsystems */
		if ($3 == NULL || template_clone(addr_ntoa(&$2), $3, 1) == -1)
			yyerror("Binding to %s failed", addr_ntoa(&$2));
	}
		| CLONE STRING template
	{
		/* Just clone.  This is not the final destination yet */
		if ($3 == NULL || template_clone($2, $3, 0) == -1)
			yyerror("Cloning to %s failed", $2);
		free($2);
	}
;
set		: SET template DEFAULT PROTO ACTION action
	{
		struct action *action;

		if ($2 == NULL) {
			yyerror("No template");
			break;
		}
		
		if ((action = honeyd_protocol($2, $4)) == NULL) {
			yyerror("Bad protocol");
			break;
		}

		port_action_clone(action, &$6);
		if ($6.action != NULL)
			free($6.action);
	}
		| SET template PERSONALITY personality
	{
		if ($2 == NULL || $4 == NULL)
			break;
		$2->person = $4;
	}
		| SET template UPTIME NUMBER
	{
		if ($2 == NULL || $4 == 0)
			break;
		$2->timestamp = $4 * 2;
	}
		| SET template DROPRATE IN rate
	{
		if ($2 == NULL)
			break;
		if ($5 > 100) {
			yyerror("Droprate too high: %f", $5);
			break;
		}

		$2->drop_inrate = $5 * 100;
	}
		| SET template UID NUMBER
	{
		if ($2 == NULL)
			break;
		if (!$4) {
			yyerror("Bad uid %d", $4);
			break;
		}
		$2->uid = $4;
	}
		| SET template UID NUMBER GID NUMBER
	{
		if ($2 == NULL)
			break;
		if (!$4 || !$6) {
			yyerror("Bad uid %d, gid %d", $4, $6);
			break;
		}
		$2->uid = $4;
		$2->gid = $6;
	}
;
annotate	: ANNOTATE personality finscan
	{
		if ($2 == NULL)
			break;
		$2->disallow_finscan = !$3;
	}
		| ANNOTATE personality fragment
	{
		if ($2 == NULL)
			break;
		$2->fragp = $3;
	}
;
route		: ROUTE ENTRY ipaddr
	{
		if (router_start(&$3) == -1)
			yyerror("Defining entry point failed");
	}
		| ROUTE ipaddr ADD NET ipnet ipaddr latency packetloss
	{
		struct router *r;

		if ((r = router_find(&$2)) == NULL) {
			yyerror("Unknown router %s", addr_ntoa(&$2));
			break;
		}
		if (router_find(&$6) == NULL)
			router_new(&$6);
		if (router_add_net(r, &$5, &$6, $7, $8) == -1)
			yyerror("Could not add route to %s", addr_ntoa(&$5));
	}
		| ROUTE ipaddr LINK ipnet
	{
		struct router *r;

		if ((r = router_find(&$2)) == NULL) {
			yyerror("Unknown router %s", addr_ntoa(&$2));
			break;
		}
		if (router_add_link(r, &$4) == -1)
			yyerror("Could not add link %s", addr_ntoa(&$4));
	}
;
finscan		: FINSCAN	{ $$ = 1; }
		| NO FINSCAN	{ $$ = 0; }
;
fragment	: FRAGMENT DROP	{ $$ = FRAG_DROP; }
		| FRAGMENT OLD	{ $$ = FRAG_OLD; }
		| FRAGMENT NEW	{ $$ = FRAG_NEW; }
;
ipaddr		: NUMBER DOT NUMBER DOT NUMBER DOT NUMBER
	{
		char src[20];
		snprintf(src, sizeof(src), "%d.%d.%d.%d",
		    $1, $3, $5, $7);
		if (addr_pton(src, &$$) < 0)
			yyerror("Illegal IP address %s", src);
	}
;
ipnet		: ipaddr SLASH NUMBER
	{
		char src[25];
		snprintf(src, sizeof(src), "%s/%d",
		    addr_ntoa(&$1), $3);
		if (addr_pton(src, &$$) < 0)
			yyerror("Illegal IP network %s", src);
	}
;
ipaddrplusport	: ipaddr COLON NUMBER
	{
		if (curtype == -1) {
			yyerror("Bad port type");
			break;
		}
		$$ = cmd_proxy_getinfo(addr_ntoa(&$1), curtype, $3);
		curtype = -1;
		if ($$ == NULL)
			yyerror("Illegal IP address port pair");
	}
;
action		: STRING
	{
		
		$$.action = $1;
		$$.status = PORT_OPEN;
	}
		| CMDSTRING
	{
		$1[strlen($1) - 1] = '\0';
		if (($$.action = strdup($1 + 1)) == NULL)
			yyerror("Out of memory");
		$$.status = PORT_OPEN;
		free($1);
	}
		| PROXY ipaddrplusport
	{
		$$.status = PORT_PROXY;
		$$.action = NULL;
		$$.aitop = $2;
	}
		| PROXY STRING COLON NUMBER
	{
		$$.status = PORT_PROXY;
		$$.action = NULL;
		$$.aitop = NULL;
		if ($2[0] != '$') {
			if (curtype == -1) {
				yyerror("Bad port type");
				break;
			}
			$$.aitop = cmd_proxy_getinfo($2, curtype, $4);
			curtype = -1;
			if ($$.aitop == NULL)
				yyerror("Illegal host name in proxy");
		} else {
			char proxy[1024];

			snprintf(proxy, sizeof(proxy), "%s:%d", $2, $4);
			$$.action = strdup(proxy);
			if ($$.action == NULL)
				yyerror("Out of memory");
		}
		free($2);
	}
		| PROXY STRING COLON STRING
	{
		char proxy[1024];
		$$.status = PORT_PROXY;
		$$.action = NULL;
		$$.aitop = NULL;

		snprintf(proxy, sizeof(proxy), "%s:%s", $2, $4);
		$$.action = strdup(proxy);
		if ($$.action == NULL)
				yyerror("Out of memory");
		free($2);
		free($4);
	}
		| BLOCK
	{
		$$.status = PORT_BLOCK;
		$$.action = NULL;
	}
		| RESET
	{
		$$.status = PORT_RESET;
		$$.action = NULL;
	}
		| OPEN
	{
		$$.status = PORT_OPEN;
		$$.action = NULL;
	}
;

template	: STRING
	{
		$$ = template_find($1);
		if ($$ == NULL)
			yyerror("Unknown template \"%s\"", $1);
		free($1);
	}
		| DEFAULT
	{
		$$ = template_find("default");
		if ($$ == NULL)
			yyerror("Unknown template \"%s\"", "default");
	}
		| ipaddr
	{
		$$ = template_find(addr_ntoa(&$1));
		if ($$ == NULL)
			yyerror("Unknown template \"%s\"", addr_ntoa(&$1));
	}
;
personality	: CMDSTRING
	{
		$1[strlen($1) - 1] = '\0';
		$$ = personality_find($1+1);
		if ($$ == NULL)
			yyerror("Unknown personality \"%s\"", $1+1);
		free($1);
	}
		| RANDOM
	{
		$$ = personality_random();
		if ($$ == NULL)
			yyerror("Random personality failed");
	}
;
rate		: NUMBER DOT NUMBER
	{
		char asc[40];

		snprintf(asc, sizeof(asc), "%d.%d", $1, $3);
		$$ = atof(asc);
	}
		| NUMBER
	{
		$$ = $1;
	}
;
latency		: /* empty */ { $$ = 0; }
		| LATENCY NUMBER MS
	{
		$$ = $2;
	}
;
packetloss	: /* empty */ { $$ = 0; }
		| LOSS rate
	{
		$$ = $2 * 100;
	}
%%

int
yyerror(char *fmt, ...)
{
	va_list ap;
	errors = 1;

	va_start(ap, fmt);
	fprintf(stderr, "%s:%d: ", filename, lineno);
	vfprintf(stderr, fmt, ap);
	fprintf(stderr, "\n");
	va_end(ap);
	return (0);
}

int
parse_configuration(FILE *input, char *name)
{
	extern FILE *yyin;

	lineno = 1;
	filename = name;
	yyin = input;
	yyparse();
	return (errors ? -1 : 0);
}
