/*
 * Copyright 2002 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Niels Provos.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef WIN32
#include "misc.h"
#include <process.h>
#include <memory.h>
#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <io.h>

struct sockinet {
  u_char  si_len;
  u_char  si_family;
  u_short si_port;
};

struct addrinfo {
  int     ai_flags;
  int     ai_family;
  int     ai_socktype;
  int     ai_protocol;
  size_t  ai_addrlen;
  char   *ai_canonname;
  struct sockaddr  *ai_addr;
  struct addrinfo  *ai_next;
};

#define NI_MAXHOST  1025
#define NI_MAXSERV    32
#define NI_NUMERICHOST  0x00000002
#define NI_NUMERICSERV  0x00000008
#else
#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/resource.h>
#include <unistd.h>
#endif
#include <sys/stat.h>
#include <sys/tree.h>
#include <sys/queue.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dnet.h>
#include <syslog.h>

#undef timeout_pending
#undef timeout_initialized

#include <event.h>

#include "honeyd.h"
#include "personality.h"
#include "subsystem.h"
#include "tcp.h"
#include "udp.h"
#include "fdpass.h"

ssize_t atomicio(ssize_t (*)(), int, void *, size_t);

extern struct callback cb_tcp;
extern struct callback cb_udp;

void
cmd_trigger_read(struct command *cmd, int size)
{
#ifdef WIN32
	/* Specify a quick timeout becuse we do no want libevent to wait INFINITE. */
	struct timeval tv = {0,1};
#endif
 	if (cmd->pfd == INVALID_SOCKET || !cmd->fdconnected)
		return;
	if (size)
#ifdef WIN32
		if(!cmd->IsProxyCmd)
			event_add(&cmd->pread, &tv);
		else
			event_add(&cmd->pread, NULL);
#else
		event_add(&cmd->pread, NULL);
#endif
}

void
cmd_trigger_write(struct command *cmd, int size)
{
#ifdef WIN32
	/* Specify a quick timeout becuse we do no want libevent to wait INFINITE. */
	struct timeval tv = {0,1};
#endif
 	if (cmd->pfd == INVALID_SOCKET || !cmd->fdconnected)
		return;
	if (size)
#ifdef WIN32
	if(!cmd->IsProxyCmd)
		event_add(&cmd->pwrite, &tv);
	else
		event_add(&cmd->pwrite, NULL);
#else
		event_add(&cmd->pwrite, NULL);
#endif
}

void
cmd_free(struct command *cmd)
{
	event_del(&cmd->pread);
	event_del(&cmd->pwrite);
#ifdef WIN32
	if(cmd->IsProxyCmd) {
		WSAEventSelect(cmd->pfd, cmd->pread.ev_fd, 0);
		WSAEventSelect(cmd->pfd, cmd->pwrite.ev_fd, 0);
		WSACloseEvent(cmd->pread.ev_fd);
		WSACloseEvent(cmd->pwrite.ev_fd);

		close(cmd->pfd);
		cmd->pfd = INVALID_SOCKET;
	} else {
		CloseHandle((HANDLE)cmd->pfd);
		cmd->pfd = INVALID_SOCKET;
		CloseHandle((HANDLE)cmd->pwritefd);
		cmd->pwritefd = INVALID_SOCKET;
	}
#else
	close(cmd->pfd);
	cmd->pfd = INVALID_SOCKET;
#endif

	if (cmd->perrfd != INVALID_SOCKET) {
		event_del(&cmd->peread);
		close(cmd->perrfd);
		cmd->perrfd = INVALID_SOCKET;
	}
}

#ifdef WIN32

#endif

void
cmd_ready_fd(struct command *cmd, struct callback *cb, void *con)
{
#ifdef WIN32
	if(cmd->IsProxyCmd) {
		if((cmd->pread.ev_fd = WSACreateEvent()) == NULL)
			return;
		if((cmd->pwrite.ev_fd = WSACreateEvent()) == NULL)
			return;

		/* 
			XXX: Creating two events which are read/write may cause issues 
						but in all my testing it works fine. 
		*/
		if(WSAEventSelect(cmd->pfd, cmd->pread.ev_fd, FD_READ|FD_WRITE|FD_CLOSE) == SOCKET_ERROR)
			errx(1, "WSAEventSelect: %d", WSAGetLastError());
		if(WSAEventSelect(cmd->pfd, cmd->pwrite.ev_fd, FD_READ|FD_WRITE|FD_CLOSE) == SOCKET_ERROR)
			errx(1, "WSAEventSelect: %d", WSAGetLastError());
		event_set(&cmd->pread, (int)cmd->pread.ev_fd, EV_READ, cb->cb_read, con);
		event_set(&cmd->pwrite, (int)cmd->pwrite.ev_fd, EV_WRITE, cb->cb_write, con);
	} else {
		event_set(&cmd->pread, cmd->pfd, EV_READ, cb->cb_read, con);
		event_set(&cmd->pwrite, cmd->pwritefd, EV_WRITE, cb->cb_write, con);
	}
#else
	event_set(&cmd->pread, cmd->pfd, EV_READ, cb->cb_read, con);
	event_set(&cmd->pwrite, cmd->pfd, EV_WRITE, cb->cb_write, con);
#endif
	cmd->fdconnected = 1;

	if (cmd->perrfd != INVALID_SOCKET)
		event_set(&cmd->peread, cmd->perrfd, EV_READ, cb->cb_eread,
		    con);
}

struct addrinfo *
cmd_proxy_getinfo(char *address, int type, short port)
{
	struct addrinfo ai, *aitop;
#ifdef WIN32
	struct hostent *host_entry;
	struct sockaddr_in *sin;
#else
        char strport[NI_MAXSERV];
#endif

        memset(&ai, 0, sizeof (ai));
        ai.ai_family = AF_INET;
        ai.ai_socktype = type;
        ai.ai_flags = 0;
#ifdef WIN32
	aitop = (struct addrinfo *)malloc(sizeof(struct addrinfo));
	if(!aitop) {
		warn("malloc");
		return (NULL);
	}
	aitop->ai_addr = (struct sockaddr *)malloc(sizeof(struct sockaddr));
	if(!aitop->ai_addr) {
		warn("malloc");
		return (NULL);
	}
	aitop->ai_addrlen = sizeof(struct sockaddr);
	sin = (struct sockaddr_in *)aitop->ai_addr;
	sin->sin_port = htons(port);

  host_entry = gethostbyname(address); 

  if(host_entry == NULL) {
		warn("gethostbyname");
		return (NULL);
  } 

  memcpy(&sin->sin_addr, host_entry->h_addr_list[0], host_entry->h_length); 

  sin->sin_family = host_entry->h_addrtype; 
  sin->sin_port = htons(port);	
#else
        snprintf(strport, sizeof (strport), "%d", port);
        if (getaddrinfo(address, strport, &ai, &aitop) != 0) {
                warn("getaddrinfo");
                return (NULL);
        }
#endif
	return (aitop);
}

int
cmd_proxy_connect(struct tuple *hdr, struct command *cmd, struct addrinfo *ai,
    void *con)
{
	char ntop[NI_MAXHOST], strport[NI_MAXSERV];
	char *host = ntop, *port = strport;
	struct callback *cb;
	struct timeval tv = {10, 0};
        int fd;
#ifdef WIN32
	struct hostent *pHost = NULL;

	cmd->IsProxyCmd = 1;
#endif        
	if (hdr->type == SOCK_STREAM)
		cb = &cb_tcp;
	else
		cb = &cb_udp;

        fd = socket(AF_INET, hdr->type, 0);
        if (fd == INVALID_SOCKET) {
                warn("socket");
                return (-1);
        }
#ifndef WIN32
        if (fcntl(fd, F_SETFL, O_NONBLOCK) == -1)
                warn("fcntl(O_NONBLOCK)");

        if (fcntl(fd, F_SETFD, 1) == -1)
                warn("fcntl(F_SETFD)");
#endif
	cmd->pfd = fd;
        if (connect(fd, ai->ai_addr, ai->ai_addrlen) == 0) {
		(*cb->cb_connect)(fd, EV_WRITE, con);
		return (0);
	}

#ifndef WIN32
	if (errno != EINPROGRESS) {
		warn("connect");
		cmd->pfd = -1;
		close(fd);
		return (-1);
	}
#endif

	event_set(&cmd->pwrite, fd, EV_WRITE, cb->cb_connect, con);
	event_add(&cmd->pwrite, &tv);

#ifdef WIN32
  pHost = gethostbyaddr(ai->ai_addr->sa_data, ai->ai_addrlen,AF_INET);
	if(pHost)
		host = pHost->h_name;
	else
		host = inet_ntoa(((struct sockaddr_in *)ai->ai_addr)->sin_addr);

	snprintf(port, sizeof(strport), "%d", ntohs(((const struct sockinet *)ai->ai_addr)->si_port));
#else
	if (getnameinfo(ai->ai_addr, ai->ai_addrlen,
		ntop, sizeof(ntop), strport, sizeof(strport),
		NI_NUMERICHOST|NI_NUMERICSERV) != 0) {

		host = "<hosterror>";
		port = "<porterror>";
	}
#endif
	syslog(LOG_INFO, "Connection established: %s -> proxy to %s:%s",
	    honeyd_contoa(hdr), host, port);

	return (0);
}

void
cmd_environment(struct template *tmpl, struct tuple *hdr)
{
	char line[256];
	struct addr addr;

	if (tmpl->person != NULL) {
		snprintf(line, sizeof(line), "%s", tmpl->person->name);
		setenv("HONEYD_PERSONALITY", line, 1);
	}

	if (hdr == NULL)
		return;
	     
	addr_pack(&addr, ADDR_TYPE_IP, IP_ADDR_BITS, &hdr->ip_src,IP_ADDR_LEN);
	snprintf(line, sizeof(line), "%s", addr_ntoa(&addr));
	setenv("HONEYD_IP_SRC", line, 1);

	addr_pack(&addr, ADDR_TYPE_IP, IP_ADDR_BITS, &hdr->ip_dst,IP_ADDR_LEN);
	snprintf(line, sizeof(line), "%s", addr_ntoa(&addr));
	setenv("HONEYD_IP_DST", line, 1);

	snprintf(line, sizeof(line), "%d", hdr->sport);
	setenv("HONEYD_SRC_PORT", line, 1);

	snprintf(line, sizeof(line), "%d", hdr->dport);
	setenv("HONEYD_DST_PORT", line, 1);
}

int
cmd_setpriv(struct template *tmpl)
{
#ifndef WIN32
	uid_t uid = 32767;
	gid_t gid = 32767;
	struct rlimit rl;

	if (tmpl->uid)
		uid = tmpl->uid;
	if (tmpl->gid)
		gid = tmpl->gid;

	/* Lower privileges */
#ifdef HAVE_SETGROUPS
	setgroups(1, &gid);
#endif
	setegid(gid);
	setgid(gid);
	seteuid(uid);
	setuid(uid);

	/* Raising file descriptor limits */
	rl.rlim_cur = rl.rlim_max = 24;
	if (setrlimit(RLIMIT_NOFILE, &rl) == -1)
		err(1, "setrlimit");
#endif
	return (0);
}

int
cmd_fork(struct tuple *hdr, struct command *cmd, struct template *tmpl,
    char *execcmd, char **argv, void *con)
{
	extern int honeyd_nchildren;
	struct callback *cb;
#ifdef WIN32
	int pid;
  SECURITY_ATTRIBUTES saAttr; 
  BOOL fSuccess;
	DWORD dwMode;
	HANDLE hChildStdinRd, hChildStdinWr, hChildStdinWrDup, 
   hChildStdoutRd, hChildStdoutWr, hChildStdoutRdDup, 
	 hChildStderrRd, hChildStderrWr, hChildStderrRdDup, 
	 hSaveStdin, hSaveStdout, hSaveStderr; 

	// Set the bInheritHandle flag so pipe handles are inherited. 

	saAttr.nLength = sizeof(SECURITY_ATTRIBUTES); 
	saAttr.bInheritHandle = TRUE; 
	saAttr.lpSecurityDescriptor = NULL; 

	// The steps for redirecting child process's STDOUT: 
	//     1. Save current STDOUT, to be restored later. 
	//     2. Create anonymous pipe to be STDOUT for child process. 
	//     3. Set STDOUT of the parent process to be write handle to 
	//        the pipe, so it is inherited by the child process. 
	//     4. Create a noninheritable duplicate of the read handle and
	//        close the inheritable read handle. 

	// Save the handle to the current STDOUT. 

	hSaveStdout = GetStdHandle(STD_OUTPUT_HANDLE); 

	// Create a pipe for the child process's STDOUT. 

	if (!CreatePipe(&hChildStdoutRd, &hChildStdoutWr, &saAttr, 0)) 
		warn("Stdout pipe creation failed\n"); 

	// Set a write handle to the pipe to be STDOUT. 

	if (!SetStdHandle(STD_OUTPUT_HANDLE, hChildStdoutWr)) 
		warn("Redirecting STDOUT failed"); 

	// Create noninheritable read handle and close the inheritable read 
	// handle. 

	fSuccess = DuplicateHandle(GetCurrentProcess(), hChildStdoutRd,
			GetCurrentProcess(), &hChildStdoutRdDup , 0,
			FALSE,
			DUPLICATE_SAME_ACCESS);
	if( !fSuccess )
			warn("DuplicateHandle failed");
	CloseHandle(hChildStdoutRd);

	// Save the handle to the current STDERR. 

	hSaveStderr = GetStdHandle(STD_ERROR_HANDLE); 

	// Create a pipe for the child process's STDERR. 

	if (!CreatePipe(&hChildStderrRd, &hChildStderrWr, &saAttr, 0)) 
		warn("Stderr pipe creation failed\n"); 

	// Set a write handle to the pipe to be STDERR. 

	if (! SetStdHandle(STD_ERROR_HANDLE, hChildStderrWr)) 
		warn("Redirecting STDERR failed"); 

	// Create noninheritable read handle and close the inheritable read 
	// handle. 

	fSuccess = DuplicateHandle(GetCurrentProcess(), hChildStderrRd,
			GetCurrentProcess(), &hChildStderrRdDup , 0,
			FALSE,
			DUPLICATE_SAME_ACCESS);
	if(!fSuccess )
			warn("DuplicateHandle failed");

	CloseHandle(hChildStderrRd);

	// The steps for redirecting child process's STDIN: 
	//     1.  Save current STDIN, to be restored later. 
	//     2.  Create anonymous pipe to be STDIN for child process. 
	//     3.  Set STDIN of the parent to be the read handle to the 
	//         pipe, so it is inherited by the child process. 
	//     4.  Create a noninheritable duplicate of the write handle, 
	//         and close the inheritable write handle. 

	// Save the handle to the current STDIN. 

	hSaveStdin = GetStdHandle(STD_INPUT_HANDLE); 

	// Create a pipe for the child process's STDIN. 

	if (! CreatePipe(&hChildStdinRd, &hChildStdinWr, &saAttr, 0)) 
		warn("Stdin pipe creation failed\n"); 

	// Set a read handle to the pipe to be STDIN. 

	if (! SetStdHandle(STD_INPUT_HANDLE, hChildStdinRd)) 
		warn("Redirecting Stdin failed"); 

	// Duplicate the write handle to the pipe so it is not inherited. 

	fSuccess = DuplicateHandle(GetCurrentProcess(), hChildStdinWr, 
		GetCurrentProcess(), &hChildStdinWrDup, 0, 
		FALSE,                  // not inherited 
		DUPLICATE_SAME_ACCESS); 
	if (! fSuccess) 
		warn("DuplicateHandle failed"); 

	CloseHandle(hChildStdinWr); 

	// Set our Pipes to Asynchronous mode.
	dwMode = PIPE_NOWAIT;
	SetNamedPipeHandleState( 
		hChildStdoutRdDup,    // pipe handle 
		&dwMode,  // new pipe mode 
		NULL,     // don't set maximum bytes 
		NULL);    // don't set maximum time 
	SetNamedPipeHandleState( 
		hChildStdinWrDup,    // pipe handle 
		&dwMode,  // new pipe mode 
		NULL,     // don't set maximum bytes 
		NULL);
	SetNamedPipeHandleState( 
		hChildStderrRdDup,    // pipe handle 
		&dwMode,  // new pipe mode 
		NULL,     // don't set maximum bytes 
		NULL);

	// Place the handles in out command struct.
	cmd->pfd = (int)hChildStdoutRdDup;
	cmd->pwritefd = (int)hChildStdinWrDup;
	cmd->perrfd = (int)hChildStderrRdDup;
	
	// Create our Environment.
	cmd_environment(tmpl, hdr);

	// Spawn process
	pid = spawnvp(P_NOWAIT, execcmd, argv);
	if(!pid) {
		warn("Could not spawn process");
		return (-1);
	}

	// To give our cmd some time to initialize. Recommended by MSDN.
	Sleep(1);

	// After process creation, restore the saved STDIN, STDOUT, and STDERR. 
	if (!SetStdHandle(STD_INPUT_HANDLE, hSaveStdin)) 
		warn("Re-redirecting Stdin failed\n"); 

	if (!SetStdHandle(STD_OUTPUT_HANDLE, hSaveStdout)) 
		warn("Re-redirecting Stdout failed\n"); 
 
	if (!SetStdHandle(STD_ERROR_HANDLE, hSaveStderr)) 
		warn("Re-redirecting Stderr failed\n"); 

	if (hdr->type == SOCK_STREAM)
		cb = &cb_tcp;
	else
		cb = &cb_udp;

	cmd_ready_fd(cmd, cb, con);

	event_add(&cmd->pread, NULL);
	event_add(&cmd->peread, NULL);

	honeyd_nchildren++;

	return (0);
#else
	int pair[2], perr[2];
	sigset_t sigmask;

	if (socketpair(AF_UNIX, hdr->type, 0, pair) == -1)
		return (-1);
	if (socketpair(AF_UNIX, SOCK_STREAM, 0, perr) == -1) {
		close(pair[0]);
		close(pair[1]);
		return (-1);
	}

	/* Block SIGCHLD */
	sigemptyset(&sigmask);
	sigaddset(&sigmask, SIGCHLD);
	if (sigprocmask(SIG_BLOCK, &sigmask, NULL) == -1) {
		warn("sigprocmask");
		goto fork_err;
	}

	cmd->pid = fork();
	if (cmd->pid == -1) {
		warn("fork");
		goto unmask_err;
	}

	if (cmd->pid == 0) {
		/* Child privileges */
		cmd_setpriv(tmpl);

		/* Child */
		close(pair[0]);
		if (dup2(pair[1], fileno(stdout)) == -1)
			err(1, "%s: dup2", __func__);
		if (dup2(pair[1], fileno(stdin)) == -1)
			err(1, "%s: dup2", __func__);
		close(perr[0]);
		if (dup2(perr[1], fileno(stderr)) == -1)
			err(1, "%s: dup2", __func__);

		close(pair[1]);
		close(perr[1]);

		cmd_environment(tmpl, hdr);

		if (execv(execcmd, argv) == -1)
			err(1, "%s: execv(%s)", __func__, execcmd);

		/* NOT REACHED */
	}

	close(pair[1]);
	cmd->pfd = pair[0];
	if (fcntl(cmd->pfd, F_SETFD, 1) == -1)
		warn("fcntl(F_SETFD)");
	if (fcntl(cmd->pfd, F_SETFL, O_NONBLOCK) == -1)
		warn("fcntl(F_SETFL)");

	close(perr[1]);
	cmd->perrfd = perr[0];
	if (fcntl(cmd->perrfd, F_SETFD, 1) == -1)
		warn("fcntl(F_SETFD)");
	if (fcntl(cmd->perrfd, F_SETFL, O_NONBLOCK) == -1)
		warn("fcntl(F_SETFL)");

	if (hdr->type == SOCK_STREAM)
		cb = &cb_tcp;
	else
		cb = &cb_udp;

	cmd_ready_fd(cmd, cb, con);

	event_add(&cmd->pread, NULL);
	event_add(&cmd->peread, NULL);

	honeyd_nchildren++;

	/* Install old signal handler */
	if (sigprocmask(SIG_UNBLOCK, &sigmask, NULL) == -1) {
		warn("sigprocmask");
		goto fork_err;
	}
	return (0);

	/* Error cleanup */
 unmask_err:
	/* Install old signal handler */
	if (sigprocmask(SIG_UNBLOCK, &sigmask, NULL) == -1)
		warn("sigprocmask");

 fork_err:
	close(perr[0]);
	close(perr[1]);
	close(pair[0]);
	close(pair[1]);
	cmd->pfd = -1;
#endif

	return (-1);
}

int
cmd_subsystem(struct template *tmpl, struct subsystem *sub,
    char *execcmd, char **argv)
{
#ifdef WIN32
	return (-1);
#else
	extern int honeyd_nchildren;
	struct command *cmd = &sub->cmd;
	extern struct callback subsystem_cb;
	int pair[2];
	sigset_t sigmask;

	if (socketpair(AF_UNIX, SOCK_STREAM, 0, pair) == -1)
		return (-1);

	/* Block SIGCHLD */
	sigemptyset(&sigmask);
	sigaddset(&sigmask, SIGCHLD);
	if (sigprocmask(SIG_BLOCK, &sigmask, NULL) == -1) {
		warn("sigprocmask");
		goto fork_err;
	}

	cmd->pid = fork();
	if (cmd->pid == -1) {
		warn("fork");
		goto unmask_err;
	}

	if (cmd->pid == 0) {
		/* Set privileges */
		cmd_setpriv(tmpl);

		/* Child */
		close(pair[0]);
		if (dup2(pair[1], SUBSYSTEM_MAGICFD) == -1)
			err(1, "%s: dup2", __func__);
		if (dup2(fileno(stderr), fileno(stdout)) == -1)
			err(1, "%s: dup2", __func__);
		if (dup2(fileno(stderr), fileno(stdin)) == -1)
			err(1, "%s: dup2", __func__);

		close(pair[1]);

		cmd_environment(tmpl, NULL);

		/* Setup the wrapper library */
		if (setenv("LD_PRELOAD", PATH_HONEYDLIB"/libhoneyd.so", 1) == -1)
			err(1, "%s: setenv", __func__);

		if (execv(execcmd, argv) == -1)
			err(1, "%s: execv(%s)", __func__, execcmd);

		/* NOT REACHED */
	}

	close(pair[1]);
	cmd->pfd = pair[0];
	if (fcntl(cmd->pfd, F_SETFD, 1) == -1)
		warn("fcntl(F_SETFD)");
	if (fcntl(cmd->pfd, F_SETFL, O_NONBLOCK) == -1)
		warn("fcntl(F_SETFL)");

	cmd->perrfd = -1;
	cmd_ready_fd(cmd, &subsystem_cb, sub);

	event_add(&cmd->pread, NULL);

	honeyd_nchildren++;

	/* Install old signal handler */
	if (sigprocmask(SIG_UNBLOCK, &sigmask, NULL) == -1) {
		warn("sigprocmask");
		goto fork_err;
	}
	return (0);

	/* Error cleanup */
 unmask_err:
	/* Install old signal handler */
	if (sigprocmask(SIG_UNBLOCK, &sigmask, NULL) == -1)
		warn("sigprocmask");

 fork_err:
	close(pair[0]);
	close(pair[1]);
	cmd->pfd = -1;

	return (-1);
#endif
}

int
cmd_subsystem_connect(struct tuple *hdr, struct command *cmd,
    struct port *port, void *con)
{
#ifdef WIN32
	return (-1);
#else
	struct callback *cb;
	struct subsystem *sub = port->sub;
	struct sockaddr_in si;
	struct addr src;
	int pair[2];
        
	if (hdr->type == SOCK_STREAM)
		cb = &cb_tcp;
	else
		cb = &cb_udp;

        if (socketpair(AF_LOCAL, hdr->type, 0, pair) == -1) {
                warn("socketpair");
                return (-1);
        }

        if (fcntl(pair[0], F_SETFL, O_NONBLOCK) == -1)
                warn("fcntl(O_NONBLOCK)");

        if (fcntl(pair[0], F_SETFD, 1) == -1)
                warn("fcntl(F_SETFD)");

	cmd->pfd = pair[0];

	/* Prepare sockaddr */
	addr_pack(&src, ADDR_TYPE_IP, IP_ADDR_BITS, &hdr->ip_src,IP_ADDR_LEN);
	addr_ntos(&src, (struct sockaddr *)&si);
	si.sin_port = htons(hdr->sport);

	if (send_fd(port->sub_fd, pair[1], &si, sizeof(si)) == -1)
		return (-1);

	/* After transfering the file descriptor, we may close it */
	close(pair[1]);

	/* We are connected now */
	(*cb->cb_connect)(pair[0], EV_WRITE, con);

	syslog(LOG_INFO, "Connection established: %s -> subsystem \"%s\"",
	    honeyd_contoa(hdr), sub->cmdstring);

	return (0);
#endif
}

int
cmd_subsystem_localconnect(struct tuple *hdr, struct command *cmd,
    struct port *port, void *con)
{
#ifdef WIN32
	return (-1);
#else
	struct callback *cb;
	struct subsystem *sub = port->sub;
	struct sockaddr_in si;
	struct addr src;
	int fd;
        
	if (hdr->type == SOCK_STREAM)
		cb = &cb_tcp;
	else
		cb = &cb_udp;

	if (port->sub_fd == -1) {
		char res;

		while ((fd = receive_fd(sub->cmd.pfd, NULL, NULL)) == -1) {
			if (errno != EAGAIN) {
				warnx("%s: receive_fd", __func__);
			}
		}

		/* Confirm success of failure */
		res = fd == -1 ? -1 : 0;
		atomicio(write, sub->cmd.pfd, &res, 1);
		if (fd == -1)
			return (-1);

		port->sub_fd = fd;
	}

	/* Get another fd on this special thingy */
	while ((fd = receive_fd(port->sub_fd, NULL, NULL)) == -1) {
		if (errno != EAGAIN) {
			warnx("%s: receive_fd", __func__);
			return (-1);
		}
	}

        if (fcntl(fd, F_SETFL, O_NONBLOCK) == -1)
                warn("%s: fcntl(O_NONBLOCK)", __func__);

        if (fcntl(fd, F_SETFD, 1) == -1)
                warn("%s: fcntl(F_SETFD)", __func__);

	cmd->pfd = fd;

	/* Prepare sockaddr */
	addr_pack(&src, ADDR_TYPE_IP, IP_ADDR_BITS, &hdr->ip_dst, IP_ADDR_LEN);
	addr_ntos(&src, (struct sockaddr *)&si);
	si.sin_port = htons(hdr->dport);

	/* Write the bound socket address to the other side */
	if (atomicio(write, port->sub_fd, &si, sizeof(si)) != sizeof(si))
		return (-1);

	/* Now we may close the special thingy */
	close(port->sub_fd);
	port->sub_fd = -1;

	/* We are connected now */
	(*cb->cb_connect)(fd, EV_WRITE, con);

	syslog(LOG_INFO, "Connection established: subsystem \"%s\" -> %s",
	    sub->cmdstring, honeyd_contoa(hdr));

	return (0);
#endif
}
