package genalg;

import java.util.Random;
import java.util.List;
import java.util.Vector;

/**
 * A Population contains a list of individuals.
 * 
 * A Population can evolve from one generation
 * to the next, applying selection, mutations and
 * cross-overs.
 * 
 * @author Wim Mees 
 * @version 0.1
 */
public class Population
{
    // class variables
    private static Random random = new Random();
    
    // instance variables
    private List listOfIndividuals = null;
    private String lowerBound = null;
    private String upperBound = null;
    private FitnessMap fitnessMap = null; 

    /**
     * Constructor for objects of class Population
     */
    public Population(int populationSize, String lowerBound, String upperBound)
    {
        // precondition assertions
        assert lowerBound!=null;
        assert upperBound!=null;
        assert populationSize>1;
        
        // initialise instance variables
        listOfIndividuals = new Vector();
        this.lowerBound = lowerBound;
        this.upperBound = upperBound;
        fitnessMap = new FitnessMap(); 

        // create an initial population
        for ( int i=0 ; i<populationSize ; i++ )
        {
            // create a new individual
            Individual individual = new Individual(lowerBound, upperBound);
            
            // add it to the list
            listOfIndividuals.add(individual);
            
            // initialise its fitness in the fitness-map
            fitnessMap.getFitness(individual.getGenes());
        }
        
        // postcondition assertions
        assert this.lowerBound!=null;
        assert this.upperBound!=null;
        assert listOfIndividuals!=null;
        assert listOfIndividuals.size()==populationSize;
        assert fitnessMap!=null;
    }
        
    /**
     * The method "evolve" computes one evolution of
     * the population to a subsequent generation.
     * 
     */
    public void evolve()
    {
        // precondition assertions
        assert listOfIndividuals!=null;
        assert listOfIndividuals.size()>0;
        
        // selection
        int nrOfRebirth = (int) Math.round(0.3 * listOfIndividuals.size());
        Maternity maternity = new Maternity(fitnessMap);
        for ( int i=0 ; i<nrOfRebirth ; i++ )
        {
            // remove a random individual
            int indexRandomIndividualThatDies = random.nextInt(listOfIndividuals.size());
            listOfIndividuals.remove(indexRandomIndividualThatDies);
            
            // birth of a new 
            Individual newBorn = new Individual(maternity.giveBirth());
            listOfIndividuals.add(newBorn);
        }
        
        // mutations
        int nrOfMutations = (int) Math.round(0.1 * listOfIndividuals.size());
        for ( int i=0 ; i<nrOfMutations ; i++ )
        {
            int indexRandomIndividual = random.nextInt(listOfIndividuals.size());
            Individual individual = (Individual) listOfIndividuals.get(indexRandomIndividual);
            individual.mutate(lowerBound, upperBound);
        }
        
        // cross-overs
        int nrOfCrossovers = (int) Math.round(0.1 * listOfIndividuals.size());
        for ( int i=0 ; i<nrOfCrossovers ; i++ )
        {
            int indexFirstRandomIndividual = random.nextInt(listOfIndividuals.size());
            int indexSecondRandomIndividual = random.nextInt(listOfIndividuals.size());
            Individual firstIndividual = (Individual) listOfIndividuals.get(indexFirstRandomIndividual);
            Individual secondIndividual = (Individual) listOfIndividuals.get(indexSecondRandomIndividual);
            firstIndividual.crossOver(secondIndividual);
        }
    }
    
    /**
     * The method "getGenomeSample" returns the genome of a 
     * random individual in the population.
     * 
     * @return  the genome as a String
     */
    String getGenomeSample()
    {
        // precondition assertions
        assert listOfIndividuals!=null;
        assert listOfIndividuals.size()>0;

        // select a random individual
        int indexRandomIndividual = random.nextInt(listOfIndividuals.size());
        Individual individual = (Individual) listOfIndividuals.get(indexRandomIndividual);
            
        // gets its genome
        String genome = individual.getGenes();

        // postcondition assertions
        assert genome!=null;
        
        return genome;
    }
    
    /**
     * The method "genomeIsFit" increases the fitness degree
     * of a given genome in the fitness map.
     * 
     * @param   genome  the genome for which to modify the fitness
     */
    void genomeIsFit(String genome)
    {
        // precondition assertions
        assert genome!=null;
        assert fitnessMap!=null;
        
        // increase fitness in the fitness-map for this genome
        fitnessMap.increaseFitness(genome);
    }
    
    /**
     * The method "genomeIsNotFit" decreases the fitness degree
     * of a given genome in the fitness map.
     * 
     * @param   genome  the genome for which to modify the fitness
     */
    void genomeIsNotFit(String genome)
    {
        // precondition assertions
        assert genome!=null;
        assert fitnessMap!=null;
        
        // increase fitness in the fitness-map for this genome
        fitnessMap.decreaseFitness(genome);
    }
    
    /**
     * The method "getPopulation" returns the genome as a String.
     * It is only used for printing debugging information.
     * 
     * @return  the population as a String
     */
    String getPopulation()
    {
        // precondition assertions
        assert listOfIndividuals!=null;

        // build a String
        String result = new String("");
        for ( int i=0 ; i<listOfIndividuals.size() ; i++ )
        {
            Individual individual = (Individual) listOfIndividuals.get(i);
            result = result + individual.getGenes() + "\r\n";
        }

        // postcondition assertions
        assert result!=null;
        
        return result;
    }
    
    /**
     * The method main here is used for testing only.
     */
    public static void main(String [] args)
    {
        Population population = new Population(20, "0.0.0", "3.9.9");
        System.out.println("population:\r\n" + population.getPopulation());
        
        for ( int i=0 ; i<10 ; i++ )
        {
            System.out.println("evolution " + i);
            population.evolve();
            System.out.println(population.getPopulation());
        }
        
        String genome = population.getGenomeSample();
        System.out.println("random genome: " + genome);
    }
}
