package genalg;

import java.util.Collection;
import java.util.Iterator;
import java.util.Random;

/**
 * The class Maternity takes a fitness-map and
 * produces children with genomes based on the 
 * fitness-map, in such a way that genomes with 
 * a higher degree of fitness are more likely to 
 * occur in the children that are produced.
 * 
 * @author Wim Mees 
 * @version 0.1
 */
public class Maternity
{
    // class variables
    private Random random = new Random();
    
    // instance variables
    private FitnessMap fitnessMap = null;
    private Collection collectionOfGenomes = null;
    private double sumOfAllFitnesses = 0.0;

    /**
     * Constructor for objects of class Maternity
     */
    public Maternity(FitnessMap fitnessMap)
    {
        // precondition assertions
        assert fitnessMap!=null;

        // initialise instance variables
        this.fitnessMap = fitnessMap;
        collectionOfGenomes = fitnessMap.getGenomes();

        sumOfAllFitnesses = 0.0;
        Iterator iterator = collectionOfGenomes.iterator();
        while (iterator.hasNext())
        {
            sumOfAllFitnesses += fitnessMap.getFitness((String) iterator.next());
        }
        
        // postcondition assertions
        assert this.fitnessMap!=null;
        assert collectionOfGenomes!=null;
    }

    /**
     * The method "giveBirth" produces a child
     * in the form of its genome. Genomes with
     * a higher degree of fitness are more likely
     * to be selected as the genome for this
     * new child.
     * 
     * @return     the genome of the child as a String
     */
    public String giveBirth()
    {
        // precondition assertions
        assert fitnessMap!=null;
        assert collectionOfGenomes!=null;
        
        // generate a random number between "0.0" and "sumOfAllFitnesses"
        double target = sumOfAllFitnesses * random.nextDouble();
        
        // find the genome in the collection where the target lies
        String child = null; 
        double sumOfFitnesses = 0.0;
        Iterator iterator = collectionOfGenomes.iterator();
        while ((iterator.hasNext()) && (child == null))
        {
            // get next genome from the collection
            String genome = (String) iterator.next();
            
            // compute interval corresponding with this genome in the collection
            double previousSumOfFitnesses = sumOfFitnesses;
            sumOfFitnesses += fitnessMap.getFitness(genome);
            
            // verify whether target lies in this interval
            if ((target>=previousSumOfFitnesses) && (target<sumOfFitnesses))
            {
                child = genome;    
            }
        }
        
        // if no child was found, the fitness scale was changed, try again
        if (child == null)
        {
            child = giveBirth();
        }    
        
        // postcondition assertions
        assert child!=null;
        
        return child;
    }
    
    /**
     * The method main here is used for testing only.
     */
    public static void main(String [] args)
    {
        FitnessMap fitnessMap = new FitnessMap();
        String [] genomes = { new String("l"), new String("m"), new String("h") };
        for ( int i=0 ; i<10 ; i++ )
        {
            fitnessMap.decreaseFitness(genomes[0]);
            fitnessMap.increaseFitness(genomes[2]);
        }
        
        System.out.println("initial situation:");
        for ( int i=0 ; i<3 ; i++ )
        {
            System.out.println("genome \"" + genomes[i] + "\" has fitness " + fitnessMap.getFitness(genomes[i]));
        }
        
        Maternity maternity = new Maternity(fitnessMap);
        System.out.println("children:");
        for ( int i=0 ; i<500 ; i++ )
        {
            System.out.print(maternity.giveBirth());
        }
        System.out.println();
    }
}
