package genalg;

import java.util.Random;

/**
 * The class Individual represents an individual in
 * a population.
 * 
 * @author Wim Mees 
 * @version 0.1
 */
public class Individual
{
    // class variables
    private static Random random = new Random();

    // instance variables
    private String genes = null;

    /**
     * Constructor for objects of class Individual
     */
    Individual(String lowerBound, String upperBound)
    {
        // precondition assertions
        assert lowerBound!=null;
        assert upperBound!=null;
    
        // create a random genome
        create(lowerBound, upperBound);
        
        // postcondition assertions
        assert genes!=null;
    }
    
    Individual(String genes)
    {
        // precondition assertions
        assert genes!=null;
        
        // initialise instance variables
        this.genes = genes;
        
        // postcondition assertions
        assert this.genes!=null;
    }
    
    Individual(Individual parentIndividual)
    {
        // precondition assertions
        assert parentIndividual!=null;
        
        // initialise instance variables
        genes = parentIndividual.genes;
        
        // postcondition assertions
        assert genes!=null;
    }
    
    /**
     * The method Create generates a random genome for this individual,
     * respecting the lowerBound and upperBound genomes that
     * are passed as parameters.
     * 
     * @param   lowerBound  the lower bound
     * @param   upperBound  the upper bound
     */
    private void create(String lowerBound, String upperBound)
    {
        // precondition assertions
        assert genes==null;  // should only be called when no genes content is present
        assert lowerBound!=null;
        assert upperBound!=null;
        assert lowerBound.length()==upperBound.length();
        
        // compute each of the genes
        byte [] content = new byte [lowerBound.length()];
        byte [] contentLowerBound = lowerBound.getBytes();
        byte [] contentUpperBound = upperBound.getBytes();
        for ( int i=0 ; i<content.length ; i++ )
        {
            int range = contentUpperBound[i] - contentLowerBound[i];
            content[i] = (byte) (contentLowerBound[i] + random.nextInt(range+1));
        }
        genes = new String(content);
        
        // postcondition assertions
        assert genes!=null;
    }

    /**
     * The method mutate mutates the genome of this individual,
     * respecting the lowerBound and upperBound genomes that
     * are passed as parameters.
     * 
     * @param   lowerBound  the lower bound
     * @param   upperBound  the upper bound
     */
    void mutate(String lowerBound, String upperBound)
    {
        // precondition assertions
        assert genes!=null;
        assert lowerBound!=null;
        assert upperBound!=null;
        assert lowerBound.length()==upperBound.length();
        assert lowerBound.length()==genes.length();
        
        // select random gene that will be mutated
        int positionOfMutation = random.nextInt(genes.length());
        
        // generate random new value for this gene, within
        // the specified bounds.
        byte [] content = genes.getBytes();
        byte [] contentLowerBound = lowerBound.getBytes();
        byte [] contentUpperBound = upperBound.getBytes();
        int range = contentUpperBound[positionOfMutation] - contentLowerBound[positionOfMutation];
        if (range>0)
        {
            content[positionOfMutation] = (byte) (contentLowerBound[positionOfMutation] + random.nextInt(range+1));
        }
        
        // replace the old genome with the mutated one
        genes = new String(content);
        
        // postcondition assertions
        assert genes!=null;
    }
    
    /**
     * The method crossOver performs a cross-over between the
     * genes of this individual and the genes of another 
     * individual that is passed as a parameter.
     * 
     * @param   individual  the other individual
     */
    void crossOver(Individual individual)
    {
        // precondition assertions
        assert individual!=null;
        assert genes!=null;
        assert individual.genes!=null;
        assert individual.genes.length()==genes.length();
        
        // select the random position where we will perform the cross-over
        int positionOfMutation = random.nextInt(genes.length());
        
        // perform the cross-over
        byte [] contentThis = genes.getBytes();
        byte [] contentIndividual = individual.genes.getBytes();
        for ( int i=positionOfMutation ; i<genes.length() ; i++ )
        {
            byte tmp = contentThis[i];
            contentThis[i] = contentIndividual[i];
            contentIndividual[i] = tmp;
        }
        
        // replace the old genomes with the mutated ones
        genes = new String(contentThis);
        individual.genes = new String(contentIndividual);
        
        // postcondition assertions
        assert genes!=null;
        assert individual.genes!=null;
    }
    
    /**
     * The method "getGenes" returns the genome as a String.
     * It is only used for printing debugging information.
     * 
     * @return  the genome as a String
     */
    String getGenes()
    {
        // precondition assertions
        assert genes!=null;
        
        return genes;
    }
    
    /**
     * The method main here is used for testing only.
     */
    public static void main(String [] args)
    {
        String lowerBound = new String("1.0");
        String upperBound = new String("3.9");
        
        Individual individual1 = new Individual(lowerBound, upperBound);
        System.out.println("individual 1 created: " + individual1.genes); 
        
        individual1.mutate(lowerBound, upperBound);
        System.out.println("individual 1 mutated: " + individual1.genes); 
        
        Individual individual2 = new Individual(lowerBound, upperBound);
        System.out.println("individual 2 created: " + individual2.genes); 
        
        individual1.crossOver(individual2);
        System.out.println("crossover occured, ");
        System.out.println("  individual 1 is now: " + individual1.genes); 
        System.out.println("  individual 2 is now: " + individual2.genes); 
    }
}
