package genalg;

import java.util.Map;
import java.util.HashMap;
import java.util.Collection;

/**
 * The FitnessMap class maps genomes to a
 * degree of fitness.
 * 
 * @author Wim Mees 
 * @version 0.1
 */
public class FitnessMap
{
    // instance variables
    private Map mapFitness = null;

    /**
     * Constructor for objects of class FitnessMap
     */
    FitnessMap()
    {
        // initialise instance variables
        mapFitness = new HashMap();
        
        // postcondition assertions
        assert mapFitness!=null;
    }

    /**
     * The method "ensureExistence" ensures that
     * a given genome is present in the map.
     * 
     * @param   genes   the genome for which to ensure existence
     */
    void ensureExistence(String genes)
    {
        // precondition assertions
        assert genes!=null;
        assert mapFitness!=null;
        
        // look up the Fitness object in the map
        // that is associated with this genome.
        if (! mapFitness.containsKey(genes))
        {
            mapFitness.put(genes, new Fitness());
        }
        
        // postcondition assertions
        assert mapFitness.containsKey(genes);
    }

    /**
     * The method "increaseFitness" increases the fitness
     * of a given genome in the fitness-map.
     * 
     * @param   genes   the genome for which to modify the fitness
     */
    void increaseFitness(String genes)
    {
        // precondition assertions
        assert genes!=null;
        assert mapFitness!=null;
        
        // ensure existence of this genome in the map
        ensureExistence(genes);
        
        // look up the Fitness object in the map
        // that is associated with this genome.
        Fitness fitness = (Fitness) mapFitness.get(genes);
        
        // adapt its fitness
        fitness.increase();
    }
    
    /**
     * The method "decreaseFitness" decreases the fitness
     * of a given genome in the fitness-map.
     * 
     * @param   genes   the genome for which to modify the fitness
     */
    void decreaseFitness(String genes)
    {
        // precondition assertions
        assert genes!=null;
        assert mapFitness!=null;
        
        // ensure existence of this genome in the map
        ensureExistence(genes);
        
        // look up the Fitness object in the map
        // that is associated with this genome.
        Fitness fitness = (Fitness) mapFitness.get(genes);
        
        // adapt its fitness
        fitness.decrease();
    }

    /**
     * The method "getFitness" looks up the fitness
     * of a given genome in the fitness-map.
     * 
     * @param   genes   the genome to look up
     * @return  the fitness value for this genome
     */
    double getFitness(String genes)
    {
        // precondition assertions
        assert genes!=null;
        assert mapFitness!=null;
        
        // ensure existence of this genome in the map
        ensureExistence(genes);
        
        // look up the Fitness object in the map
        // that is associated with this genome.
        Fitness fitness = (Fitness) mapFitness.get(genes);
        
        // retreive its fitness
        return fitness.getFitness();
    }

    /**
     * The method "getGenomes" produces a Collection
     * with all the genomes in the fitness-map.
     * 
     * @return  a collection with all the genomes
     */
    Collection getGenomes()
    {
        // precondition assertions
        assert mapFitness!=null;
        
        return mapFitness.keySet();
    }
    
    /**
     * The method main here is used for testing only.
     */
    public static void main(String [] args)
    {
        FitnessMap fitnessMap = new FitnessMap();
        String [] genomes = { new String("a0.1"), new String("b4.5"), new String("c7.8") };
        
        System.out.println("initial situation:");
        for ( int i=0 ; i<3 ; i++ )
        {
            System.out.println("genome \"" + genomes[i] + "\" has fitness " + fitnessMap.getFitness(genomes[i]));
        }
        
        System.out.println("after 10 evolutions:");
        for ( int i=0 ; i<10 ; i++ )
        {
            fitnessMap.decreaseFitness(genomes[0]);
            fitnessMap.increaseFitness(genomes[2]);
        }
        for ( int i=0 ; i<3 ; i++ )
        {
            System.out.println("genome \"" + genomes[i] + "\" has fitness " + fitnessMap.getFitness(genomes[i]));
        }
    }
}
