package fsm;

import java.util.*;
import java.io.IOException;

/**
 * The class State represents one state of the 
 * finite state machine.
 * 
 * @author Wim Mees
 * @version 0.1
 */
public class State
    extends GraphElement
{
    // instance variables
    private List outgoingTransitions = null;
    private StateOperation stateOperation = null;

    /**
     * Constructor for objects of class State
     */
    private State(String label, StateOperation stateOperation)
    {
        // precondition assertions
        assert label!=null;
        assert label.trim().length()>0;
        assert stateOperation!=null;
        
        // set label in superclass
        setLabel(label);
        
        // store link to StateOperation object
        this.stateOperation = stateOperation;
        
        // instantiate list of outgoing transitions
        outgoingTransitions = new Vector();
        
        // postcondition assertions
        assert outgoingTransitions!=null;
        assert this.stateOperation!=null;
    }

    /**
     * The method registerOutgoingTransition allows a transition
     * that links this state to another, to register itself
     * as an outgoing transition with this state, so that when
     * this state is reached, it can be evaluated to see if
     * it triggers.
     * 
     * @param   transition  the transition to add to the list
    */
    void registerOutgoingTransition(Transition transition)
    {
        // precondition assertions
        assert transition!=null;
        
        // add "transition" to the list of outgoing transitions
        outgoingTransitions.add(transition);
    }

    /**
     * The method "step" verifies whether exactly one outgoing
     * transition triggers. If this is the case, the finite
     * state machine moves to the state at the other end of
     * the triggering transition. If not, an appriopriate
     * exception is thrown.
     * 
     * @return  the state at the other side of the triggered transition 
    */
    State step() throws IOException
    {
        // precondition assertions
        assert outgoingTransitions!=null;

        // activate this state
        stateOperation.performOperation();
        
        // verify whether at most one non-default transition triggers
        // and identify it.
        Transition transitionThatTriggers = null;
        Iterator iterator = outgoingTransitions.iterator();
        while (iterator.hasNext())
        {
            Transition transition = (Transition) iterator.next();
            if ((! transition.isDefault()) && (transition.triggers()))
            {
                if (transitionThatTriggers == null)
                {
                    transitionThatTriggers = transition;
                }
                else
                {
                    // more than one transition triggers, this is not 
                    // valid behaviour.
                    System.err.println("more than one non-default transition triggers (\"" 
                        + transitionThatTriggers.getLabel() + "\", \"" + transition.getLabel() + "\"");
                    assert false;
                }
            }
        }

        // when there is no non-default transition that triggers,
        // verify whether there is exactly one default transition
        if (transitionThatTriggers == null)
        {
            iterator = outgoingTransitions.iterator();
            while (iterator.hasNext())
            {
                Transition transition = (Transition) iterator.next();
                if (transition.isDefault())
                {
                    if (transitionThatTriggers == null)
                    {
                        transitionThatTriggers = transition;
                    }
                    else
                    {
                        // more than one transition triggers, this is not 
                        // valid behaviour.
                        System.err.println("more than one default transition exists (\"" 
                            + transitionThatTriggers.getLabel() + "\", \"" + transition.getLabel() + "\"");
                        assert false;
                    }
                }
            }
        }
        
        // neither a triggered non-default transaction or a 
        // default transaction was found
        if (transitionThatTriggers == null)
        {
            // no transition triggers, this is not valid behaviour.
            System.err.println("no transition triggers");
            assert false;
        }
        
        // postcondition assertions
        assert transitionThatTriggers.getTo()!=null;
        
        // return the state at the other side of the triggered transition
        return transitionThatTriggers.getTo();
    }

    /**
     * The method "isBeginState" returns true when this
     * state is a begin state. This information is obtained
     * from the stateOperation member object. 
     * 
     * @return  true when this is a begin state 
    */
    boolean isBeginState()
    {
        return stateOperation.isBegin();
    }    
    
    /**
     * The method "isEndState" returns true when this
     * state is an end state. This information is obtained
     * from the stateOperation member object. 
     * 
     * @return  true when this is an end state 
    */
    boolean isEndState()
    {
        return stateOperation.isEnd();
    }    

    /**
     * ----------------------------------------
     * class variables and functions
     * ----------------------------------------
    */
  
    // class variables
    private static HashMap mapOfStates = new HashMap();

    public static State createState(String label, StateOperation stateOperation, String comment)
    {
        // precondition assertions
        assert label!=null;
        assert stateOperation!=null;
        assert mapOfStates!=null;
        
        // create a new State instance
        State state = new State(label, stateOperation);

        // set the comment (if any)
        if (comment!=null)
        {
            state.setComment(comment);
        }
        
        // add it to our list of states
        mapOfStates.put(label, state); 
        
        // postcondition assertions
        assert state!=null;
        
        return state;
    }

    public static State getState(String label)
    {
        // precondition assertions
        assert label!=null;
        assert mapOfStates!=null;

        // lookup state with this label in the map        
        State state = (State) mapOfStates.get(label);
        
        // postcondition assertions
        assert state!=null;
        
        return state;
    }

    static State getBeginState()
    {
        // precondition assertions
        assert mapOfStates!=null;
        
        // look for a "begin" state in our list of all states
        Iterator iterator = mapOfStates.values().iterator();
        State beginState = null;
        while (iterator.hasNext())
        {
            State state = (State) iterator.next();
            if (state.isBeginState ())
            {
                if (beginState == null)
                {
                    beginState = state;
                }
                else
                {
                    // throw exception since there should only
                    // be one begin state.
                    System.err.println("there is more than one begin state");
                    assert false;
                }
            }
        }  
        
        if (beginState == null)
        {
            // throw exception indicating that no begin state was found
            System.err.println("there is no begin state");
            assert false;
        }
        
        // postcondition assertions
        assert beginState!=null;
        
        return beginState;
    }
}
