package fsm;

/**
 * The class GraphElement is a super-class for any
 * type of element that is represented in the fsm graph,
 * e.g. states and transitions.
 * 
 * It contains a label, which must be non-null, and
 * may only be set once.
 * It also contains an optional comment.
 * 
 * The set...() methods are protected since they
 * should only be called from subclasses.
 * 
 * @author Wim Mees
 * @version 0.1
 */
public abstract class GraphElement
{
    // instance variables
    private String label = null;
    private String comment = null;

    /**
     * The method setLabel() sets a value for
     * the label of this GraphElement object.
     * 
     * Any whitespace at the beginning or the end
     * of the label is removed by setLabel().
     * 
     * The length of the label (after removing
     * heading and trailing whitespace) should
     * be non-zero.
     * (= second precondition assert)
     * 
     * The call setLabel() should only be called once. 
     * The label, once assigned, should not change. 
     * (= last precondition assert)
     * 
	 * @param  label   the new label for this object
     */
    protected void setLabel(String label)
    {
        // precondition assertions
        assert label!=null;
        assert label.trim().length()>0;
        assert this.label==null;
        
        // set label
        this.label = label.trim();
        
        // postcondition assertions
        assert this.label!=null;
    }
    
    /**
     * The method setComment() sets a value for
     * the comment of this GraphElement object.
     * 
     * It may be called several times in
     * order te replace previous comments.
     * 
	 * @param  label   the new label for this object
     */
    protected void setComment(String comment)
    {
        // precondition assertions
        assert comment!=null;
        
        // set label
        this.comment = comment;
        
        // postcondition assertions
        assert this.comment!=null;
    }
    
    /**
     * The method getLabel() returns the label
     * of this GraphElement object as a String.
     * 
     * This method should not be called before 
     * a label was set with setLabel().
     * 
     * @return  the label of this object
     */
    public String getLabel()
    {
        // precondition assertions
        assert label!=null;
        
        return label;
    }
    
    /**
     * The method getComment() returns the comment
     * associated with this GraphElement object as 
     * a String.
     * 
     * @return  the comment associated with this object
     */
    public String getComment()
    {
        // precondition assertions
        // none
        
        return comment;
    }
}
