package config;

import javax.xml.parsers.*; 
import org.xml.sax.*;  
import java.io.*;
import java.text.*;
import java.util.StringTokenizer;
import java.util.Vector;
import org.w3c.dom.*;

import fsm.*;
import sops.*;
import varex.*;
import genalg.EvolutionEngine;

/**
 * The class ConfigParser reads a configuration file in xml format
 * and parses it into a finite state machine with states and
 * transitions between states.
 * 
 * @author Wim Mees 
 * @version 0.1
 */
public class ConfigParser
{
    // counter used for generating unique (hidden) names for constants ("cstring")
    private static int constantCounter = 0;

    /**
     * Constructors for objects of class ConfigParser
     */
    public ConfigParser() throws ParserConfigurationException, SAXException, IOException
    {
        this("gmhoney.xml");
    }
    
    public ConfigParser(String filename) throws ParserConfigurationException, SAXException, IOException
    {
        // precondition assertions
        assert filename!=null;
        
        // parse the configuration file
        parseFile(filename);
    }

    /**
     * The parseFile method opens an xml file and parses it.
     * 
     * @param   filename    the name of the xml file
     */
    private void parseFile(String filename) throws ParserConfigurationException, SAXException, IOException
    {
        // precondition assertions
        assert filename!=null;

        // create a DocumentBuilderFactory and configure it        
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setValidating(true);
        factory.setNamespaceAware(true);
        
        // have the factory create an instance of the DocumentBuilder class
        DocumentBuilder builder = factory.newDocumentBuilder();
            
        // parse the content of the given xml-file and 
        // return the result as a DOM Document object.
        Document document = builder.parse(new File(filename));
        
        // get the "fsm" branch of the document tree and process it
        NodeList nodeList = document.getElementsByTagName(new String ("fsm"));

        assert nodeList.getLength()==1;  // is imposed by the DTD
        Node nodeFsm = nodeList.item(0);
        assert nodeFsm!=null;  // is imposed by the DTD
        
        parseFsm(nodeFsm);
    }
    
    private void parseFsm(Node nodeFsm)
    {
        // precondition assertions
        assert nodeFsm!=null;
        assert nodeFsm.hasChildNodes();  // is imposed by the DTD
        
        // parse each of the children
        NodeList nodeListChildren = nodeFsm.getChildNodes();
        for ( int i=0 ; i<nodeListChildren.getLength() ; i++ )
        {
            Node nodeChild = nodeListChildren.item(i);
            if (nodeChild.getNodeType()==Node.ELEMENT_NODE)
            {
                if (nodeChild.getNodeName().compareTo("state")==0)
                {
                    parseState(nodeChild);
                }
                else if (nodeChild.getNodeName().compareTo("transition")==0)
                {
                    parseTransition(nodeChild);
                }
                else if (nodeChild.getNodeName().compareTo("constant")==0)
                {
                    parseConstant(nodeChild);
                }
                else
                {
                    System.err.println("ERROR: fsm contains unknown child \"" + nodeChild.getNodeName() + "\"");
                    assert false;  // is prohibited by DTD
                }    
            }    
        }
    }
    
    private void parseState(Node nodeState)
    {
        // precondition assertions
        assert nodeState!=null;
        assert nodeState.hasChildNodes();  // is imposed by the DTD

        // information neede to create the State object
        String label = null;
        OperationNop stateOperation = null;
        String comment = null;

        // vector for storing commands
        Vector vectorCommands = new Vector();
        
        // parse each of the children of this node in order
        // to obtain the required information about the state
        NodeList nodeListChildren = nodeState.getChildNodes();
        for ( int i=0 ; i<nodeListChildren.getLength() ; i++ )
        {
            Node nodeChild = nodeListChildren.item(i);
            if (nodeChild.getNodeType()==Node.ELEMENT_NODE)
            {
                if (nodeChild.getNodeName().compareTo("label")==0)
                {
                    label = parsePcdata(nodeChild);
                }
                else if (nodeChild.getNodeName().compareTo("species")==0)
                {
                    parseStateSpecies(nodeChild);
                }
                else if (nodeChild.getNodeName().compareTo("individual")==0)
                {
                    vectorCommands.add(parseStateIndividual(nodeChild));
                }
                else if (nodeChild.getNodeName().compareTo("operation")==0)
                {
                    stateOperation = parseStateOperation(nodeChild);
                }
                else if (nodeChild.getNodeName().compareTo("constant")==0)
                {
                    System.err.println("ERROR: constants are not yet supported");
                    assert false;
                }
                else if (nodeChild.getNodeName().compareTo("comment")==0)
                {
                    comment = parsePcdata(nodeChild);
                }
                else
                {
                    System.err.println("ERROR: state contains unknown child \"" + nodeChild.getNodeName() + "\"");
                    assert false;  // is prohibited by DTD
                }    
            }    
        }
        
        // transfer the command objects to the state operation object
        assert stateOperation!=null;
        for ( int i=0 ; i<vectorCommands.size() ; i++ )
        {
            stateOperation.addCommand((CommandSpec) vectorCommands.get(i));
        }
        
        // create a new State object
        assert label!=null;
        assert stateOperation!=null;
        State state = State.createState(label, stateOperation, comment);
        
    }

    private void parseStateSpecies(Node nodeStateSpecies)
    {
        // precondition assertions
        assert nodeStateSpecies!=null;
        assert nodeStateSpecies.hasChildNodes();  // is imposed by the DTD

        // get a reference to the evolution engine
        EvolutionEngine evolutionEngine = EvolutionEngine.getEvolutionEngine();

        // children of the species node
        String label = null;
        String action_type = null;
        String lowerBound = null;
        String upperBound = null;
        
        // parse each of the children
        NodeList nodeListChildren = nodeStateSpecies.getChildNodes();
        for ( int i=0 ; i<nodeListChildren.getLength() ; i++ )
        {
            Node nodeChild = nodeListChildren.item(i);
            if (nodeChild.getNodeType()==Node.ELEMENT_NODE)
            {
                if (nodeChild.getNodeName().compareTo("label")==0)
                {
                    label = parsePcdata(nodeChild);
                }
                else if (nodeChild.getNodeName().compareTo("action")==0)
                {
                    // verify whether there are attributes
                    if (! nodeChild.hasAttributes())
                    {
                        // throw exception
                        System.err.println("ERROR: state/species attributes missing for action");
                        assert false;
                    }
                    
                    // get the attributes
                    NamedNodeMap mapOfAttributes = nodeChild.getAttributes();
                    
                    // take the type attribute
                    Node nodeAttributeType = mapOfAttributes.getNamedItem("type");
                    if (nodeAttributeType == null)
                    {
                        // throw exception
                        System.err.println("ERROR: state/species attribute \"type\" missing in put/get");
                        assert false;
                    }
                    
                    action_type = nodeAttributeType.getNodeValue();
                }
                else if (nodeChild.getNodeName().compareTo("lowerbound")==0)
                {
                    lowerBound = parsePcdata(nodeChild);
                }
                else if (nodeChild.getNodeName().compareTo("upperbound")==0)
                {
                    upperBound = parsePcdata(nodeChild);
                }
                else
                {
                    System.err.println("ERROR: state/species contains unknown child \"" + nodeChild.getNodeName() + "\"");
                    assert false;  // is prohibited by DTD
                }    
            }    
        }
        
        // parse the action_type parameter
        assert label!=null;
        assert action_type!=null;
        if (action_type.compareTo("load")==0)
        {
            assert lowerBound!=null;
            assert upperBound!=null;
            evolutionEngine.addSpecies(label, lowerBound, upperBound);
        }
        else
        {
            System.err.println("ERROR: state/species contains unknown action type \"" + action_type + "\"");
            assert false;  // is prohibited by DTD
        }    
    }

    private CommandSpec parseStateIndividual(Node nodeStateIndividual)
    {
        // precondition assertions
        assert nodeStateIndividual!=null;
        assert nodeStateIndividual.hasChildNodes();  // is imposed by the DTD

        // get a reference to the evolution engine
        EvolutionEngine evolutionEngine = EvolutionEngine.getEvolutionEngine();

        // children of the species node
        String label = null;
        String species_label = null;
        String action_type = null;
        
        // parse each of the children
        NodeList nodeListChildren = nodeStateIndividual.getChildNodes();
        for ( int i=0 ; i<nodeListChildren.getLength() ; i++ )
        {
            Node nodeChild = nodeListChildren.item(i);
            if (nodeChild.getNodeType()==Node.ELEMENT_NODE)
            {
                if (nodeChild.getNodeName().compareTo("label")==0)
                {
                    label = parsePcdata(nodeChild);
                }
                else if (nodeChild.getNodeName().compareTo("species_label")==0)
                {
                    species_label = parsePcdata(nodeChild);
                }
                else if (nodeChild.getNodeName().compareTo("action")==0)
                {
                    // verify whether there are attributes
                    if (! nodeChild.hasAttributes())
                    {
                        // throw exception
                        System.err.println("ERROR: state/individual attributes missing for action");
                        assert false;
                    }
                    
                    // get the attributes
                    NamedNodeMap mapOfAttributes = nodeChild.getAttributes();
                    
                    // take the type attribute
                    Node nodeAttributeType = mapOfAttributes.getNamedItem("type");
                    if (nodeAttributeType == null)
                    {
                        // throw exception
                        System.err.println("ERROR: state/individual attribute \"type\" missing in put/get");
                        assert false;
                    }
                    
                    action_type = nodeAttributeType.getNodeValue();
                }
                else
                {
                    System.err.println("ERROR: state/individual contains unknown child \"" + nodeChild.getNodeName() + "\"");
                    assert false;  // is prohibited by DTD
                }    
            }    
        }
        
        // parse the action_type parameter
        assert label!=null;
        assert species_label!=null;
        assert action_type!=null;
        CommandSpec command = null;
        if (action_type.compareTo("generate")==0)
        {
            // create a variable for this individual
            VariableString.create(label);
        
            // create a command object that will obtain a genome sample at
            // the moment this state is activated; we cannot already do this
            // know since the fitness map will evolve between now and when
            // this state is activated.
            command = new CommandGenalgGenerate(label, species_label);
        }
        else if (action_type.compareTo("fit")==0)
        {
            command = new CommandGenalgFit(label, species_label);
        }
        else if (action_type.compareTo("nofit")==0)
        {
            command = new CommandGenalgNoFit(label, species_label);
        }
        else
        {
            System.err.println("ERROR: state/individual contains unknown action type \"" + action_type + "\"");
            assert false;  // is prohibited by DTD
        }    
        
        // postcondition assertions
        assert command!=null;
        
        return command;
    }

    private OperationNop parseStateOperation(Node nodeStateOperation)
    {
        // precondition assertions
        assert nodeStateOperation!=null;
        assert nodeStateOperation.hasChildNodes();  // is imposed by the DTD
        
        OperationNop stateOperation = null;
        
        // parse each of the children
        NodeList nodeListChildren = nodeStateOperation.getChildNodes();
        for ( int i=0 ; i<nodeListChildren.getLength() ; i++ )
        {
            Node nodeChild = nodeListChildren.item(i);
            if (nodeChild.getNodeType()==Node.ELEMENT_NODE)
            {
                if (nodeChild.getNodeName().compareTo("put")==0)
                {
                    OperationPutGet operationPutGet = new OperationPut();
                    stateOperation = operationPutGet;
                    parseStateOperationPutGet(nodeChild, operationPutGet);
                }
                else if (nodeChild.getNodeName().compareTo("get")==0)
                {
                    OperationPutGet operationPutGet = new OperationGet();
                    stateOperation = operationPutGet;
                    parseStateOperationPutGet(nodeChild, operationPutGet);
                }
                else if (nodeChild.getNodeName().compareTo("nop")==0)
                {
                    stateOperation = new OperationNop();
                }
                else if (nodeChild.getNodeName().compareTo("begin")==0)
                {
                    stateOperation = new OperationBegin();
                }
                else if (nodeChild.getNodeName().compareTo("end")==0)
                {
                    stateOperation = new OperationEnd();
                }
                else
                {
                    System.err.println("ERROR: state/operation contains unknown child \"" + nodeChild.getNodeName() + "\"");
                    assert false;  // is prohibited by DTD
                }    
            }    
        }
        
        return stateOperation;
    }
    
    private void parseStateOperationPutGet(Node nodeStateOperationPutGet, OperationPutGet stateOperationPutGet)
    {
        // precondition assertions
        assert nodeStateOperationPutGet!=null;
        assert nodeStateOperationPutGet.hasChildNodes();  // is imposed by the DTD
        assert stateOperationPutGet!=null;
        
        // parse each of the children
        NodeList nodeListChildren = nodeStateOperationPutGet.getChildNodes();
        for ( int i=0 ; i<nodeListChildren.getLength() ; i++ )
        {
            Node nodeChild = nodeListChildren.item(i);
            if (nodeChild.getNodeType()==Node.ELEMENT_NODE)
            {
                if (nodeChild.getNodeName().compareTo("field")==0)
                {
                    // verify whether there are attributes
                    if (! nodeChild.hasAttributes())
                    {
                        // throw exception
                        System.err.println("ERROR: state/operation attributes missing in put/get");
                        assert false;
                    }
                    
                    // get the attributes
                    NamedNodeMap mapOfAttributes = nodeChild.getAttributes();
                    
                    // take the type attribute
                    Node nodeAttributeType = mapOfAttributes.getNamedItem("type");
                    if (nodeAttributeType == null)
                    {
                        // throw exception
                        System.err.println("ERROR: state/operation attribute \"type\" missing in put/get");
                        assert false;
                    }
                    
                    if (nodeAttributeType.getNodeValue().compareTo("cstring")==0)
                    {
                        // generate a name for a constant string
                        NumberFormat numberFormat = new DecimalFormat("000");
                        String variableName = "cstring_" + numberFormat.format(++constantCounter);
                    
                        // create a variable with this name and fill in its value
                        String variableValue = parsePcdata(nodeChild);
                        VariableSpec variable = VariableString.create(variableName);
                        variable.set(variableValue);
                    
                        // add the variable as a field to the operation
                        stateOperationPutGet.addVariable(variable);
                    }
                    else if (nodeAttributeType.getNodeValue().compareTo("string")==0)
                    {
                        // get the name of the variable
                        String variableName = parsePcdata(nodeChild);
                        
                        // look for an existing variable with this name,
                        // or create a new one if no variable exists
                        VariableSpec variable = null;
                        if (Variable.containsVariable(variableName))
                        {
                            variable = Variable.getVariable(variableName);
                        }
                        else
                        {
                            variable = VariableString.create(variableName);
                        }
                        
                        // add the variable as a field to the operation
                        stateOperationPutGet.addVariable(variable);
                    }
                    else if (nodeAttributeType.getNodeValue().compareTo("int")==0)
                    {
                        // get the name of the variable
                        String variableName = parsePcdata(nodeChild);
                        
                        // look for an existing variable with this name,
                        // or create a new one if no variable exists
                        VariableSpec variable = null;
                        if (Variable.containsVariable(variableName))
                        {
                            variable = Variable.getVariable(variableName);
                        }
                        else
                        {
                            variable = VariableInt.create(variableName);
                        }
                    
                        // add the variable as a field to the operation
                        stateOperationPutGet.addVariable(variable);
                    }
                    else if (nodeAttributeType.getNodeValue().compareTo("float")==0)
                    {
                        // get the name of the variable
                        String variableName = parsePcdata(nodeChild);
                        
                        // look for an existing variable with this name,
                        // or create a new one if no variable exists
                        VariableSpec variable = null;
                        if (Variable.containsVariable(variableName))
                        {
                            variable = Variable.getVariable(variableName);
                        }
                        else
                        {
                            variable = VariableFloat.create(variableName);
                        }
                        
                        // add the variable as a field to the operation
                        stateOperationPutGet.addVariable(variable);
                    }
                    else if (nodeAttributeType.getNodeValue().compareTo("macro")==0)
                    {
                        // get the macro command
                        String macroCommand = parsePcdata(nodeChild);
                        
                        // create a macro object
                        VariableSpec macro = new Macro(macroCommand);
                        
                        // add the macro object as a field to the operation
                        stateOperationPutGet.addVariable(macro);
                    }
                    else if (nodeAttributeType.getNodeValue().compareTo("individual")==0)
                    {
                        // get the name of the individual
                        String variableName = parsePcdata(nodeChild);
                        
                        // look for an existing variable with this name,
                        // which should exist, being created by an <individual>
                        // node in the xml file beforehand.
                        VariableSpec variable = null;
                        if (Variable.containsVariable(variableName))
                        {
                            variable = Variable.getVariable(variableName);
                        }
                        else
                        {
                            System.err.println("ERROR: unknown individual \"" + variableName + "\"");
                            assert false;
                        }
                        
                        // add the variable as a field to the operation
                        stateOperationPutGet.addVariable(variable);
                    }
                    else
                    {
                        System.err.println("ERROR: state/operation unknown attribute \"" + nodeAttributeType.getNodeValue() + "\" in put/get operation");
                        assert false;  // is prohibited by DTD
                    }    
                }
                else
                {
                    System.err.println("ERROR: node is not \"Node.ELEMENT_NODE\"");
                    assert false;  // is prohibited by DTD
                }
            }    
        }
    }
    
    private void parseTransition(Node nodeTransition)
    {
        // precondition assertions
        assert nodeTransition!=null;
        assert nodeTransition.hasChildNodes();  // is imposed by the DTD
        
        String label = null;
        State stateFrom = null;
        State stateTo = null;
        TransitionCondition condition = null;
        String comment = null;
        
        // parse each of the children
        NodeList nodeListChildren = nodeTransition.getChildNodes();
        for ( int i=0 ; i<nodeListChildren.getLength() ; i++ )
        {
            Node nodeChild = nodeListChildren.item(i);
            if (nodeChild.getNodeType()==Node.ELEMENT_NODE)
            {
                if (nodeChild.getNodeName().compareTo("label")==0)
                {
                    assert label==null;  // is imposed by DTD
                    label = parsePcdata(nodeChild);
                }
                else if (nodeChild.getNodeName().compareTo("from")==0)
                {
                    assert stateFrom==null;  // is imposed by DTD
                    String labelStateFrom = parsePcdata(nodeChild);
                    stateFrom = State.getState(labelStateFrom);
                }
                else if (nodeChild.getNodeName().compareTo("to")==0)
                {
                    assert stateTo==null;  // is imposed by DTD
                    String labelStateTo = parsePcdata(nodeChild);
                    stateTo = State.getState(labelStateTo);
                }
                else if (nodeChild.getNodeName().compareTo("condition")==0)
                {
                    assert condition==null;  // is imposed by DTD
                    condition = parseTransitionCondition(nodeChild);
                }
                else if (nodeChild.getNodeName().compareTo("comment")==0)
                {
                    assert comment==null;  // is imposed by DTD
                    comment = parsePcdata(nodeChild);
                }
                else
                {
                    assert false;  // is prohibited by DTD
                }    
            }    
        }
        
        // create a new State object
        assert label!=null;
        assert stateFrom!=null;
        assert stateTo!=null;
        new Transition(label, stateFrom, stateTo, condition, comment);
    }

    private TransitionCondition parseTransitionCondition(Node nodeCondition)
    {
        // precondition assertions
        assert nodeCondition!=null;
        assert nodeCondition.hasChildNodes();  // is imposed by the DTD

        // get condition as "text"
        String conditionString = parsePcdata(nodeCondition);

        // parse the condition text into an expression
        ExpressionSpec expression = parseExpression(conditionString);
        
        // embed this expression in a TransitionCondition object
        TransitionCondition condition = new TransitionConditionExpression(expression);
        
        // postcondition assertions
        assert condition!=null;
        
        return condition;
    }
    
    private ExpressionSpec parseExpression(String expressionString)
    {
        // precondition assertions
        assert expressionString!=null;
    
        ExpressionSpec expression = null;

        // parse the string
        StringTokenizer expressionStringTokenizer = new StringTokenizer(expressionString, "()<>=&|", true);
        while (expressionStringTokenizer.hasMoreTokens())
        {
            String token = expressionStringTokenizer.nextToken().trim();
            if (token.compareTo("(")==0)
            {
                // read up to next ")"
                int openCount = 1;
                String enclosedExpressionString = new String();
                while (openCount >0)
                {
                    String nextToken = expressionStringTokenizer.nextToken().trim();
                    if (nextToken.compareTo("(")==0)
                    {
                        openCount++;
                    }
                    else if (nextToken.compareTo(")")==0)
                    {
                        openCount--;
                    }
                    else
                    {
                        enclosedExpressionString = enclosedExpressionString + nextToken;
                    }
                }
                
                // convert this to an expression
                ExpressionSpec expressionLeft = parseExpression(enclosedExpressionString);
                
                // check if a boolean binary operator follows, 
                // in turn followed by another "()" enclosed expression
                if (expressionStringTokenizer.hasMoreTokens())
                {
                    String nextToken = expressionStringTokenizer.nextToken().trim();
                    if (nextToken.compareTo("^")==0)
                    {
                        // binary boolean AND
                        expression = new ExpressionBinaryBooleanAnd(expressionLeft, parseExpression(expressionStringTokenizer.nextToken("")));
                    }
                    else if (nextToken.compareTo("|")==0)
                    {
                        // binary boolean OR
                        expression = new ExpressionBinaryBooleanOr(expressionLeft, parseExpression(expressionStringTokenizer.nextToken("")));
                    }
                    else
                    {
                        System.err.println("ERROR: illegal operator \"" + nextToken + "\" in expression \"" + expressionString + "\"");
                        assert false;
                    }
                }
                else
                {
                    expression = expressionLeft;
                }
                
            }
            else if (token.compareTo("!")==0)
            {
                // binary "NOT" operator
                expression = new ExpressionUnaryBoolean(parseExpression(expressionStringTokenizer.nextToken("")));
            }
            else if (Variable.containsVariable(token))
            {
                // this is an expression, consisting of a variable, followed by a binary operator 
                // and another variable or a constant; therefore, parse it that way
                Variable variableLeft = Variable.getVariable(token);
                String operatorString = expressionStringTokenizer.nextToken().trim();
                String variableRightString = expressionStringTokenizer.nextToken().trim();
                
                // is the right member a variable or a constant ?
                Variable variableRight = null;
                if (Variable.containsVariable(variableRightString))
                {
                    variableRight = Variable.getVariable(variableRightString);
                }
                else
                {
                    // TBD: floating point or integer constant as a function of left variable
                    
                    // generate a name for a constant string
                    NumberFormat numberFormat = new DecimalFormat("000");
                    String variableRightName = "cstring_" + numberFormat.format(++constantCounter);
                    
                    // create a variable with this name and fill in its value
                    variableRight = VariableString.create(variableRightName);
                    variableRight.set(variableRightString);
                }
                
                // interpret the operator string
                if (operatorString.compareTo("=")==0)
                {
                    expression = new ExpressionBinaryVariableEqual(variableLeft, variableRight);
                }
                else if (operatorString.compareTo("<")==0)
                {
                    expression = new ExpressionBinaryVariableLessThan(variableLeft, variableRight);
                }
                else if (operatorString.compareTo(">")==0)
                {
                    expression = new ExpressionBinaryVariableGreaterThan(variableLeft, variableRight);
                }
                else
                {
                    System.err.println("ERROR: illegal operator \"" + operatorString + "\" in expression \"" + expressionString + "\"");
                    assert false;
                }
            }
            else
            {
                System.err.println("ERROR: illegal begin \"" + token + "\" in expression \"" + expressionString + "\"");
                assert false;
            }
        }
    
        // postcondition assertions
        assert expression!=null;
        
        return expression;
    }
    
    private void parseConstant(Node nodeConstant)
    {
        // precondition assertions
        assert nodeConstant!=null;
        assert nodeConstant.hasChildNodes();  // is imposed by the DTD
        
        System.out.println("constant: tbd");
    }

    private String parsePcdata(Node nodePcdata)
    {
        // precondition assertions
        assert nodePcdata!=null;
        assert nodePcdata.hasChildNodes();  // is imposed by the DTD

        String value = null;
        
        // parse each of the children
        NodeList nodeListChildren = nodePcdata.getChildNodes();
        for ( int i=0 ; i<nodeListChildren.getLength() ; i++ )
        {
            Node nodeChild = nodeListChildren.item(i);
            if (nodeChild.getNodeType()==Node.TEXT_NODE)
            {
                if (nodeChild.getNodeName().compareTo("#text")==0)
                {
                    value = nodeChild.getNodeValue();
                }
                else
                {
                    System.out.println("error name: " + nodeChild.getNodeName());
                    assert false;  // is prohibited by DTD
                }
            }
            else
            {
                System.out.println("error type: " + nodeChild.getNodeType());
                assert false;  // is prohibited by DTD
            }    
        }
        
        // postcondition assertions
        assert value!=null;
        return value;
    }
    
    public static void main (String [] args) throws ParserConfigurationException, SAXException, IOException
    {
        new ConfigParser();
    }
}
