/**
 *	Copyright (c) 2003, Andrew B. Smith and Jason M. Fox
 *	All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 *	modification, are permitted provided that the following conditions are met:
 *
 *	* Redistributions of source code must retain the above copyright notice, 
 *	  this list of conditions and the following disclaimer.
 *	* Redistributions in binary form must reproduce the above copyright notice, 
 *	  this list of conditions and the following disclaimer in the documentation 
 *	  and/or other materials provided with the distribution.
 *	* Neither the name of the CERIAS nor the names of its contributors 
 *	  may be used to endorse or promote products derived from this software without 
 *	  specific prior written permission.
 *
 *	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 *	ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 *	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 *	IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 *	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 *	BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *	DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY 
 *	OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 *	OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 *	OF THE POSSIBILITY OF SUCH DAMAGE.
 */

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.util.Vector;


/**
 * The RandomNetGui class is in charge of making the beautiful gui that helps
 * users generate random honeyd configuration files
 */
public class RandomNetGui implements ActionListener {

    /**
     * Pointer to an instance of the <code>MiddleClass</code> that the
     * gui should interact with
     */
    private MiddleClass _middleClass;
	
    /**
     * the main Frame of the gui
     */
    private JFrame _frame; 
	
    /**
     * JList that holds the router fingerprint objects
     */
    private JList _routerFingerprintList;
    
    /**
     * JList that holds the host fingerprint objects
     */
    private JList _hostFingerprintList;
	
    /**
     * JList that holds the service objects
     */
    private JList _serviceList;  
	
    /**
     * Swing component that holds the file to output to
     */
    private JTextField _outputText;
	
    /**
     * Swing component that holds cidr string
     */
    private JTextField _addressText;
	
    /**
     * Swing component to hold the RandomNet Config filename
     */
    private JTextField _rnconfigText; 
	
    /**
     * Swing component that holds number of routers
     */
    private JTextField _routerField;
	
    /**
     * Swing component that holds number of layers
     */
    private JTextField _layerField;
	  
    /**
     * Allows the user to select file to save random net config file to
     */
    private JFileChooser _saveAsFileChooser; 
	
    /**
     * Swing component for complexity
     */
    private JRadioButton _complexityButton1; 
	
    /**
     * Swing component for complexity
     */
    private JRadioButton _complexityButton2;
	
    /**
     * Swing component for complexity
     */
    private JRadioButton _complexityButton3;

    /**
     * this is the functions that creates the gui.  It creates most of the
     * components and layout managers, and lays out the components in the gui
     *
     * @return the gui components
     */
    public Component createComponents() {
      
        // load fingerprints vector
        Fingerprint fingerprints[] = _middleClass.getAvailableFingerprints();
        Vector availableFingerprintsVector = new Vector(fingerprints.length); 

        for (int i = 0; i < fingerprints.length; i++) {
            availableFingerprintsVector.add(i, fingerprints[i]);
        }
		
        // load services vector
        Service services[] = _middleClass.getAvailableServices();
        Vector availableServicesVector = new Vector(services.length); 

        for (int i = 0; i < services.length; i++) {
            availableServicesVector.add(i, services[i].getServiceName());
        }
        // start creating gui controls
        _routerFingerprintList = new JList(availableFingerprintsVector);
        JScrollPane routerFingerprintScrollPane = new JScrollPane(_routerFingerprintList);
        Vector routerFingerprintIndices = _middleClass.getRouterFingerprintIndices();
        int routerFingerprintArray[] = new int[routerFingerprintIndices.size()];

        for (int i = 0; i < routerFingerprintIndices.size(); i++) {
            routerFingerprintArray[i] = Integer.parseInt((String) routerFingerprintIndices.get(i));
        }
        _routerFingerprintList.setSelectedIndices(routerFingerprintArray);
        _hostFingerprintList = new JList(availableFingerprintsVector);
        JScrollPane hostFingerprintScrollPane = new JScrollPane(_hostFingerprintList);
        Vector hostFingerprintIndices = _middleClass.getHostFingerprintIndices();
        int hostFingerprintArray[] = new int[hostFingerprintIndices.size()];

        for (int i = 0; i < hostFingerprintIndices.size(); i++) {
            hostFingerprintArray[i] = Integer.parseInt((String) hostFingerprintIndices.get(i));
        }
        _hostFingerprintList.setSelectedIndices(hostFingerprintArray);           		// fingerprintScrollPane.setPreferredSize(new Dimension(200,250));
        _serviceList = new JList(availableServicesVector);
        JScrollPane serviceScrollPane = new JScrollPane(_serviceList);
        Vector serviceIndices = _middleClass.getServiceIndices();
        int serviceArray[] = new int[serviceIndices.size()];

        for (int i = 0; i < serviceIndices.size(); i++) {
            serviceArray[i] = Integer.parseInt((String) serviceIndices.get(i));
        }
        _serviceList.setSelectedIndices(serviceArray);
        // serviceScrollPane.setPreferredSize(new Dimension(200,250));
        JLabel fingerprintLabel = new JLabel("Available Router Fingerprints:");
        JLabel hostFingerprintLabel = new JLabel("Available Host Fingerprints:");
        JLabel serviceLabel = new JLabel("Available Service Scripts:");
		
        JLabel outputLabel = new JLabel("Output filename:");

        _outputText = new JTextField(30);
        _outputText.setText(_middleClass.getHoneydConfigFilename());
		
        JLabel addressLabel = new JLabel("CIDR Address Space:");

        _addressText = new JTextField(30);
        _addressText.setText(_middleClass.getCIDRString());
		
        JLabel rnconfigLabel = new JLabel("RandomNet Configuration File:");

        _rnconfigText = new JTextField(30);
        _rnconfigText.setText(_middleClass.getRandomNetConfigFilename());
		
        JTabbedPane tabbedPane = new JTabbedPane();
        JPanel basicPanel = new JPanel();
        JPanel advancedPanel = new JPanel();

        tabbedPane.addTab("Basic Options", basicPanel);
        tabbedPane.addTab("Advanced Options", advancedPanel);
		
        JButton cancelButton = new JButton("Exit");

        cancelButton.setActionCommand("exit");
        cancelButton.addActionListener(this);
        JButton generateButton = new JButton("Generate");

        generateButton.setActionCommand("generate");
        generateButton.addActionListener(this);
        generateButton.setDefaultCapable(true);
        // JButton saveAsButton = new JButton("Save As...");
        // saveAsButton.setActionCommand("saveas");
        // saveAsButton.addActionListener(this);
        JPanel buttonPane = new JPanel();

        buttonPane.add(cancelButton);
        // buttonPane.add(saveAsButton);
        buttonPane.add(generateButton);
        // basic panel
        _complexityButton1 = new JRadioButton("One");
        _complexityButton2 = new JRadioButton("Two");
        _complexityButton3 = new JRadioButton("Three");
        int complexityLevel = _middleClass.getComplexityLevel();

        if (complexityLevel == 0) {// grey out complexity setting
        } else if (complexityLevel == 1) {
            _complexityButton1.setSelected(true);
        } else if (complexityLevel == 2) {
            _complexityButton2.setSelected(true);
        } else if (complexityLevel == 3) {
            _complexityButton3.setSelected(true);
        }
		
        ButtonGroup complexityGroup = new ButtonGroup();

        complexityGroup.add(_complexityButton1);
        complexityGroup.add(_complexityButton2);
        complexityGroup.add(_complexityButton3);
        basicPanel.add(new JLabel("Complexity Level Setting:"));
        basicPanel.add(_complexityButton1);
        basicPanel.add(_complexityButton2);
        basicPanel.add(_complexityButton3);
		
        // advanced panel
        GridBagLayout advancedLayout = new GridBagLayout();
        GridBagConstraints ac = new GridBagConstraints();

        advancedPanel.setLayout(advancedLayout);
        JLabel routerLabel = new JLabel("Number of routers:");

        ac.gridx = 0;
        ac.gridy = 0;
        ac.fill = GridBagConstraints.HORIZONTAL;
        advancedLayout.setConstraints(routerLabel, ac);
        advancedPanel.add(routerLabel);
        _routerField = new JTextField(3);
        _routerField.setText(Integer.toString(_middleClass.getNumberOfRouters()));
        ac.gridx = 1;
        ac.gridy = 0;
        advancedLayout.setConstraints(_routerField, ac);
        advancedPanel.add(_routerField);
        JLabel layerLabel = new JLabel("Number of layers:");

        ac.gridx = 0;
        ac.gridy = 1;
        ac.fill = GridBagConstraints.HORIZONTAL;
        advancedLayout.setConstraints(layerLabel, ac);
        advancedPanel.add(layerLabel);
        _layerField = new JTextField(3);
        _layerField.setText(Integer.toString(_middleClass.getNumberOfLayers()));
        ac.gridx = 1;
        ac.gridy = 1;
        advancedLayout.setConstraints(_layerField, ac);
        advancedPanel.add(_layerField);
		
        /*
         * An easy way to put space between a top-level container
         * and its contents is to put the contents in a JPanel
         * that has an "empty" border.
         */
        JPanel pane = new JPanel();

        pane.setPreferredSize(new Dimension(700, 800));
        pane.setBorder(BorderFactory.createEmptyBorder(10, // top
                10, // left
                10, // bottom
                10) // right
                );
										
        // create gridbag and fill in 
        GridBagLayout gridbag = new GridBagLayout();
        GridBagConstraints c = new GridBagConstraints();

        pane.setLayout(gridbag);
        c.fill = GridBagConstraints.BOTH;
        c.gridx = 0;
        c.gridy = 0;
        // c.ipadx = 10;
        // c.ipady = 10;
        gridbag.setConstraints(fingerprintLabel, c);
        pane.add(fingerprintLabel);
        c.gridx = 1;
        c.gridy = 0;
        gridbag.setConstraints(serviceLabel, c);
        pane.add(serviceLabel);
        c.gridx = 0;
        c.gridy = 1;
        c.weightx = 0.5;
        c.weighty = 1.0;
        gridbag.setConstraints(routerFingerprintScrollPane, c);
        pane.add(routerFingerprintScrollPane);
        c.gridx = 0;
        c.gridy = 2;
        c.weightx = 0.5;
        c.weighty = 0.0;
        gridbag.setConstraints(hostFingerprintLabel, c);
        pane.add(hostFingerprintLabel);
        c.gridx = 0;
        c.gridy = 3;
        c.weightx = 0.5;
        c.weighty = 1.0;
        gridbag.setConstraints(hostFingerprintScrollPane, c);
        pane.add(hostFingerprintScrollPane);
        c.gridx = 1;
        c.gridy = 1;
        c.weightx = 0.5;
        c.weighty = 0.0;
        c.gridheight = 3;
        gridbag.setConstraints(serviceScrollPane, c);
        pane.add(serviceScrollPane);
        c.gridx = 0;
        c.gridy = 4;
        c.weighty = 0.0;
        c.gridheight = 1;
        c.anchor = GridBagConstraints.EAST;
        gridbag.setConstraints(outputLabel, c);
        pane.add(outputLabel);
        c.gridx = 1;
        c.gridy = 4;
        c.weighty = 0.0;
        c.fill = GridBagConstraints.HORIZONTAL;
        gridbag.setConstraints(_outputText, c);
        pane.add(_outputText);
        c.gridx = 0;
        c.gridy = 5;
        c.weighty = 0.0;
        c.anchor = GridBagConstraints.EAST;
        gridbag.setConstraints(addressLabel, c);
        pane.add(addressLabel);
        c.gridx = 1;
        c.gridy = 5;
        c.weighty = 0.0;
        c.fill = GridBagConstraints.HORIZONTAL;
        gridbag.setConstraints(_addressText, c);
        pane.add(_addressText);
        c.gridx = 0;
        c.gridy = 6;
        c.weighty = 0.0;
        c.anchor = GridBagConstraints.EAST;
        gridbag.setConstraints(rnconfigLabel, c);
        pane.add(rnconfigLabel);
        c.gridx = 1;
        c.gridy = 6;
        c.weighty = 0.0;
        c.fill = GridBagConstraints.HORIZONTAL;
        gridbag.setConstraints(_rnconfigText, c);
        pane.add(_rnconfigText);
        c.gridx = 0;
        c.gridy = 7;
        c.fill = GridBagConstraints.BOTH;
        c.gridwidth = 2;
        gridbag.setConstraints(tabbedPane, c);
        pane.add(tabbedPane);
        c.gridwidth = 1;
        c.gridx = 1;
        c.gridy = 8;
        c.fill = GridBagConstraints.NONE;
        gridbag.setConstraints(buttonPane, c);
        pane.add(buttonPane);
		
        return pane;
    }
	
    /**
     * The RandomNetGui sets up the gui
     *
     * @param	middleClass
     */
    public RandomNetGui(MiddleClass middleClass) {
        // initialize private data
        _middleClass = middleClass;
	
        // set look and feel
        try {
            UIManager.setLookAndFeel(UIManager.getCrossPlatformLookAndFeelClassName());
        } catch (Exception e) {}
        // Create the top-level container
        _frame = new JFrame("RandomNet GUI");
        Component contents = createComponents();

        _frame.getContentPane().add(contents);

        // Finish setting up the frame, and show it.
        _frame.addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {
                System.exit(0);
            }
        });
        _frame.pack();
        _frame.setVisible(true);
		
    }
	
    /**
     * when the generate button is pressed, this function will parse the
     * the information from the gui and then generate the new randomnet
     * configuration file.
     */
    public void actionPerformed(ActionEvent e) {
	
        if ("exit".equals(e.getActionCommand())) {
            // cancel button was pressed, lets get out of here!
            System.exit(0);
			
        } else if ("generate".equals(e.getActionCommand())) {
		
		      // inform the middle class which fingerprints and services 
            // have been selected
            int[]routerSelection = _routerFingerprintList.getSelectedIndices();
            int[]hostSelection = _hostFingerprintList.getSelectedIndices();
            int[]serviceSelection = _serviceList.getSelectedIndices();

            if (routerSelection.length == 0 || hostSelection.length == 0
                    || serviceSelection.length == 0) {
                JOptionPane.showMessageDialog(_frame,
                        "You must select at least one item" + " from each list");
                return;	
            } else {
                _middleClass.selectRouterFingerprints(routerSelection);
                _middleClass.selectHostFingerprints(hostSelection);
                _middleClass.selectServices(serviceSelection);							// inform middle class about output, config and CIDR String
                _middleClass.setHoneydConfigFilename(_outputText.getText());
                _middleClass.setCIDR(_addressText.getText());
                _middleClass.setRandomNetConfigFilename(_rnconfigText.getText());

                // inform middle class about complexity levels or router/layer info
                if (_complexityButton1.isSelected()) {
                    _middleClass.setComplexity(1);
                } else if (_complexityButton2.isSelected()) {
                    _middleClass.setComplexity(2);
                } else if (_complexityButton3.isSelected()) {
                    _middleClass.setComplexity(3);
                }
                _middleClass.setNumberOfRouters(Integer.parseInt(_routerField.getText()));
                _middleClass.setNumberOfLayers(Integer.parseInt(_layerField.getText()));			
                // inform the middle class to generate new random net configuration file
                _middleClass.saveConfigFile();
                _middleClass.generate();
			   }
        } else if ("saveas".equals(e.getActionCommand())) {
            // open file chooser dialog
            _saveAsFileChooser = new JFileChooser();
            int returnVal = _saveAsFileChooser.showOpenDialog(_frame);
		
        }
    } 
	
}
