/**
 *	Copyright (c) 2003, Andrew B. Smith and Jason M. Fox
 *	All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 *	modification, are permitted provided that the following conditions are met:
 *
 *	* Redistributions of source code must retain the above copyright notice, 
 *	  this list of conditions and the following disclaimer.
 *	* Redistributions in binary form must reproduce the above copyright notice, 
 *	  this list of conditions and the following disclaimer in the documentation 
 *	  and/or other materials provided with the distribution.
 *	* Neither the name of the CERIAS nor the names of its contributors 
 *	  may be used to endorse or promote products derived from this software without 
 *	  specific prior written permission.
 *
 *	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 *	ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 *	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 *	IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 *	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 *	BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *	DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY 
 *	OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 *	OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 *	OF THE POSSIBILITY OF SUCH DAMAGE.
 */
 
import java.io.FileWriter;
import java.io.IOException;

import java.util.Date;
import java.util.Random;
import java.util.Vector;


/**
 * Queries the MiddleClass and generates the config.random honeyd
 * configuration file.
 *
 * As listed in the honeyd man pages, the syntax for a valid configuration
 * file is as follows:<br>
 *<code>
 *  config  = creation | addition | binding | set | annotate | route [config]<br>
 *  creation= "create" template-name | "create" "default"<br>
 *  addition= "add" template-name proto "port" port-number action<br>
 *  binding = "bind" ip-address template-name |<br>
 *            "clone" template-name template-name<br>
 *  set     = "set" template-name "default" proto "action" action |<br>
 *            "set" template-name "personality" personality-name |<br>
 *            "set" template-name "personality" "random"<br>
 *            "set" template-name "subsystem" cmd-string<br>
 *            "set" template-name "uptime" seconds<br>
 *            "set" template-name "droprate" "in" percent<br>
 *            "set" template-name "uid" number ["gid" number]<br>
 *            "set" ip-address "uptime" seconds<br>
 *  annotate= "annotate" personality-name [no] finscan |<br>
 *            "annotate" personality-name "fragment" \<br>
 *                  ("drop" | "old" | "new")<br>
 *  route   = "route" "entry" ipaddr |<br>
 *            "route" ipaddr "link" ipnetwork |<br>
 *            "route" ipaddr "add" "net" ipnetwork ipaddr \<br>
 *                  ["latency" number"ms"] ["loss" percent]<br>
 *  proto   = "tcp" | "udp" | "icmp"<br>
 *  action  = "block" | "open" | "reset" | cmd-string | \<br>
 *                  "proxy" ipaddr":"port<br>
 *</code>
 *
 * @author Jason M. Fox
 */
public class RandomNetGeneration implements Runnable {

    public static class AddressNode {

        public int layerNumber;
        public long addressBits;
        public int cidrMask;
        public boolean isRouter;    // 0 for host, 1 for router
        public int numberOfDomains;
    
        private static void debugAddressNodes(Vector nodes) {
            for (int i = 0; i < nodes.size(); i++) {
                AddressNode node = (AddressNode) nodes.elementAt(i);

                System.out.println("NODE #" + i + "\n" + "\tlayerNumber     = "
                        + node.layerNumber + "\n" + "\taddressBits     = "
                        + node.addressBits + " ("
                        + getAddressString(node.addressBits) + ")\n"
                        + "\tcidrMask        = " + node.cidrMask + "\n"
                        + "\tisRouter        = " + node.isRouter + "\n"
                        + "\tnumberOfDomains = " + node.numberOfDomains);
            }
        }
    }

    /**
     * vector of nodes to represent the router hierarchy of
     * the randomly generated network
     */
    private Vector _addressNodes;
    
    /**
     * Represents the MiddleClass entity to query during
     * the honeyd configuration file generation process.
     */
    private MiddleClass _middleClass;
 
    /**
     * File writing handle for the configuration file.
     */ 
    private FileWriter _file;
   
    /**
     * Random number generator is used by different methods in this class
     * including calculateNumberOfRouters()
     */
    private Random _randomGenerator;
	
    /**
     * contains the addresses used for routers.  This is used so that
     * when randomly generating hosts, we do not accidentally overwrite
     * a router address
     */
    private Vector _routerAddresses; 	
   
	 /**
	  * save host addresses for end of file
	  */
	 private Vector _hostBindings;
	 
    /**
     * The number of routers is used in various methods of this class.  
     * It is initially set by the calculateNumberOfRouters() method
     */
    private int _numberOfRouters;	
    private int _numRoutersLeft;

    /**
     * The number of layers in the routing hierarchy.  It is
     * initially set by the calculateNumberOfLayers() method.
     */
    private int _numberOfLayers;
	
    /**
     * The number of hosts to be generated.
     */
    private int _numberOfHosts; 

    /**
     * the number of templates that were generated
     */
    private int _numRouterTemplates;
    private int _numHostTemplates; 
	
    /**
     * Basic, public constructor for the <code>RandomNetGeneration</code>.
     *
     * @param middleClass instance of <code>MiddleClass</code> to
     * query during the honeyd configuration file generation process.
     * @throws IOException instantiation of the <code>FileWriter</code> class
     * within this constructor may throw an exception therefore pass the
     * exception to the caller for handling
     */ 
    public RandomNetGeneration(MiddleClass middleClass) throws IOException {
        _middleClass = middleClass;
        // create the actual configuration file
        _file = new FileWriter(_middleClass.getHoneydConfigFilename());
        // initialize the Random Number Generator using time as seed
        _randomGenerator = new Random(System.currentTimeMillis());

        _addressNodes = new Vector();
        _routerAddresses = new Vector();
		  _hostBindings = new Vector();
    }

    /**
     * The main entry point for <code>RandomNetGeneration</code>.  Invoking
     * the <code>run</code> method results in generating a honeyd configuration
     * file based upon the parameters specified in the <code>MiddleClass</code>
     * instance.
     */
    public void run() {

        try {

            // add header comments to config file
            _file.write("########## '" + _middleClass.getHoneydConfigFilename()
                    + "'");
            _file.write(" generated by RandomNet\n");
            _file.write("########## " + (new Date()).toString() + "\n\n");
            // generate the contents of the configuration file
            generateRoutes();
            writeRoutes();
            generateTemplates();
            generateBindings();

            // flush the stream and close file
            _file.flush();
            _file.close();

        } catch (IOException ioe) {
            ioe.printStackTrace();
            System.err.println(ioe.getMessage());
        }
    }

    /**
     * Generate and write the route information to the config file
     * as specified by the <code>MiddleClass</code> instance.
     */
    private void generateRoutes() {
        // get number of routers and layer settings
        _numberOfRouters = calculateNumberOfRouters();
        _numberOfLayers = calculateNumberOfLayers();
        _numRoutersLeft = _numberOfRouters - 1;
      
        // generate sensical networks using the specified criteria
        // create the entry router (top of the router hierarchy)
        AddressNode root = new AddressNode();

        root.layerNumber = 0;
        root.addressBits = getAddressBits(_middleClass.getCIDRString());
        root.cidrMask = getCidrMask(_middleClass.getCIDRString());
        root.isRouter = true;
        root.numberOfDomains = generateNumberOfSubdomains();

        _addressNodes.add(root);
        makeHierarchy(root, root.layerNumber);

        //AddressNode.debugAddressNodes(_addressNodes);
    }

    private void makeHierarchy(AddressNode curNode, int currentLayer) {

        // base case
        if (currentLayer == _numberOfLayers) {
            return;
        }
     
        // exhausted router quota 
        if (_numRoutersLeft <= 0) {
            return;
        }

        // create sub-domain address spaces
        AddressNode[] subDomains = getSubdomains(curNode);

        // by default, the first sub-domain represents a host domain
        if (subDomains.length > 0) {
            // subDomains[0].isRouter = false;
				if (subDomains[0] != null)
               _addressNodes.add(subDomains[0]);
        }

        for (int i = 1; i < subDomains.length; i++) {
            // subDomains[i].isRouter = true;
				if (subDomains[i] != null) {
               subDomains[i].numberOfDomains = generateNumberOfSubdomains();
               _addressNodes.add(subDomains[i]);
               makeHierarchy(subDomains[i], currentLayer + 1);
				}
        }
    }
 
    /**
     * Generate a number of subdomains for a router.  The count
     * will always include a host domain.  And if there are
     * still routers left, then the count will also include
     * at least one router.
     *
     * As a side-effect, the instance variables _numRoutersLeft
     * will be modified appropriately.
     */ 
    private int generateNumberOfSubdomains() {

        /*
         System.out.println("_numRoutersLeft="+_numRoutersLeft+
         " _numberOfLayers="+_numberOfLayers+
         " Math.abs(a-b)="+Math.abs(_numRoutersLeft-_numberOfLayers));
         */
        int seed = Math.abs(_numRoutersLeft - _numberOfLayers);
        int ret = _randomGenerator.nextInt(seed == 0 ? 1 : seed);

        // need at least one host domain
        ret = ret == 0 ? 1 : ret; 

        // also need at least one router if still routers available
        if (_numRoutersLeft > 0 && ret == 1) {
            ret++;
        }

        // update count of remaining routers
        _numRoutersLeft = _numRoutersLeft - ret + 1;
        return ret;
      
    }

    /**
     * this is the "second pass".  We go through the Vector of addressNodes
     * and write information to the honeyd config file
     */
    private void writeRoutes() {
   
        try {
            // first entry in the vector should be "highest level" address
            // space, and we need this for the entry point
            _file.write("\n# Router/Routes Setup\n");
            AddressNode tempNode = (AddressNode) _addressNodes.get(0);

            _file.write("route entry "
                    + getAddressString(tempNode.addressBits | 1) + "\n");
			
            // add link and "add net" information for each router
            for (int i = 0; i < _addressNodes.size(); i++) {
                tempNode = (AddressNode) _addressNodes.get(i);
                if (tempNode.isRouter) {
                    _routerAddresses.add(new Long(tempNode.addressBits | 1));
            
                    // for each sub-domain, add the "add net" statement
                    AddressNode[] subNodes = getSubdomains(tempNode);

                    for (int s = 0; s < subNodes.length; s++) {
                        if (subNodes[s] != null) {
                            if (subNodes[s].isRouter) {
                                _file.write("route "
                                        + getAddressString(tempNode.addressBits
                                                | 1));
                                _file.write(" add net "
                                        + getAddressString(subNodes[s].addressBits));
                                _file.write("/" + subNodes[s].cidrMask + " ");
                                _file.write(getAddressString(subNodes[s].addressBits
                                        | 1));
                                _file.write(" latency "
                                        + (_randomGenerator.nextInt(100) + 1));
                                _file.write("ms loss 0."
                                        + (_randomGenerator.nextInt(2) + 1)
                                        + "\n");
                            } else {
                                // link statement 
                                _file.write("route "
                                        + getAddressString(tempNode.addressBits
                                                | 1));
                                _file.write(" link "
                                        + getAddressString(subNodes[s].addressBits));
                                _file.write("/" + subNodes[s].cidrMask + "\n");
										  addHostBindings(subNodes[s].addressBits,
										      subNodes[s].cidrMask);

                            }
                        }
                    }
                }
            }
            _file.write("\n");
            _file.flush();
         
        } catch (IOException ioe) {
            ioe.printStackTrace();
            System.err.println(ioe.getMessage());
        }
    } 

	 
	 private void addHostBindings(long bits, int mask) {
	 
		// see how many bits we have for host address space
		int bitsToPlayWith = 32 - mask - 1;

		// don't add if error
		if (bitsToPlayWith > 32 | bitsToPlayWith < 2) {
			return;
		}
		
		int complexity = _middleClass.getComplexityLevel();
		if (complexity <= 0 | complexity > 3) 
		   complexity = 3;
		int newBits = 2; // jump router	
		for (int i=0; i<(_randomGenerator.nextInt(complexity * 20)+1); i++) {
		   newBits += (long)_randomGenerator.nextInt(50)+1;
			if (newBits >= java.lang.Math.pow(2, bitsToPlayWith)) 
			   return;
			_hostBindings.add(new Long(bits + newBits));
		}
		
	 }
	 
	 
    /**
     * Generate and write the annotation information to the config file
     * as specified by by the <code>MiddleClass</code> instance.
     */
    private void generateAnnotations() {
        // randomly annotate personalities with fragment and finscan settings
     
        try {
            int hosts = _middleClass.getNumberOfHostSelectedFingerprints();

            for (int i = 0; i < hosts; i++) {
                switch (_randomGenerator.nextInt(3)) {
                case 0:
                    _file.write("annotate \""
                            + _middleClass.getHostFingerprintName(i)
                            + "\" no finscan\n");
                    break;

                case 1:
                    _file.write("annotate \""
                            + _middleClass.getHostFingerprintName(i)
                            + "\" finscan\n");
                    break;

                case 2:
                default:
                    break;
                }

                switch (_randomGenerator.nextInt(4)) {
                case 0:
                    _file.write("annotate \""
                            + _middleClass.getHostFingerprintName(i)
                            + "\" fragment drop\n");
                    break;

                case 1:
                    _file.write("annotate \""
                            + _middleClass.getHostFingerprintName(i)
                            + "\" fragment old\n");
                    break;

                case 2:
                    _file.write("annotate \""
                            + _middleClass.getHostFingerprintName(i)
                            + "\" fragment new\n");
                    break;

                case 3:
                default:
                    break;
                }
            }

            int routers = _middleClass.getNumberOfRouterSelectedFingerprints();

            for (int i = 0; i < routers; i++) {
                switch (_randomGenerator.nextInt(3)) {
                case 0:
                    _file.write("annotate \""
                            + _middleClass.getRouterFingerprintName(i)
                            + "\" no finscan\n");
                    break;

                case 1:
                    _file.write("annotate \""
                            + _middleClass.getRouterFingerprintName(i)
                            + "\" finscan\n");
                    break;

                case 2:
                default:
                    break;
                }

                switch (_randomGenerator.nextInt(4)) {
                case 0:
                    _file.write("annotate \""
                            + _middleClass.getRouterFingerprintName(i)
                            + "\" fragment drop\n");
                    break;

                case 1:
                    _file.write("annotate \""
                            + _middleClass.getRouterFingerprintName(i)
                            + "\" fragment old\n");
                    break;

                case 2:
                    _file.write("annotate \""
                            + _middleClass.getRouterFingerprintName(i)
                            + "\" fragment new\n");
                    break;

                case 3:
                default:
                    break;
                }
            }
        } catch (IOException ioe) {
            ioe.printStackTrace();
            System.err.println(ioe.getMessage());
        }
    }

    /**
     * Generate and write the template information to the config file
     * as specified by by the <code>MiddleClass</code> instance.
     */
    private void generateTemplates() {
        generateAnnotations();

        // randomly create templates with varying services and personalities
        // based upon settings specified in _middleClass
        // for now: generate one template for each host and router to be created
        // We separate numHosts from numRouters in case we want to add functionality
        // to use certain personalites for routers, and others for hosts
        // It should be an easy extension to have routerFingerprints and 
        // hostFingerprints separated in the _middleClass
        _numberOfHosts = calculateNumberOfHosts();
        int complexity = _middleClass.getComplexityLevel();

        if (complexity < 1 || complexity > 3) {
            complexity = 3;
        }
        // arbitrarily(sp?) assign number of templates to generate	  
        _numRouterTemplates = (_randomGenerator.nextInt(_numberOfRouters) + 1)
                * complexity;
        _numHostTemplates = (_randomGenerator.nextInt(20) + 1)
                * complexity;
	  
        try {
            // lets take care of the routers first
            _file.write("\n# Router templates\n");
            for (int i = 0; i < _numRouterTemplates; i++) {
                // create the name of this template and add to file
                String templateName = "router_template" + i;

                _file.write("create " + templateName + "\n");
                writeRandomRouterPersonality(i);
                writeRandomTCPAction(templateName);
                writeRandomUDPAction(templateName);
                writeRandomICMPAction(templateName);
                writeRandomServices(templateName);
                writeRandomUptime(templateName);
                writeRandomDroprate(templateName);
                writeRandomUid(templateName);
                _file.write("\n");
                _file.flush();
            }
            _file.write("\n# Host templates\n");
				
				// we need a default template to take care of all 
				// unassigned hosts
				_file.write("create default\n");
				_file.write("set default personality ");
            int randomInt = _randomGenerator.nextInt(
			      _middleClass.getNumberOfHostSelectedFingerprints());
           _file.write("\"" + _middleClass.getHostFingerprintName(randomInt)
                + "\"\n");
            writeRandomTCPAction("default");
            writeRandomUDPAction("default");
				writeRandomICMPAction("default");
				writeRandomServices("default");
				writeRandomUptime("default");
				writeRandomDroprate("default");
				writeRandomUid("default");
				_file.write("\n");
				_file.flush();
				
            // now take care of the host templates
            for (int i = 0; i < _numHostTemplates; i++) {
                // create the name of this template and add to file
                String templateName = "host_template" + i;

                _file.write("create " + templateName + "\n");
                writeRandomHostPersonality(i);
                writeRandomTCPAction(templateName);
                writeRandomUDPAction(templateName);
                writeRandomICMPAction(templateName);
                writeRandomServices(templateName);
                writeRandomUptime(templateName);
                writeRandomDroprate(templateName);
                writeRandomUid(templateName);
                _file.write("\n");
                _file.flush();
            }
        } catch (IOException ioe) {
            ioe.printStackTrace();
            System.err.println(ioe.getMessage());  	
        }
	  
    }

    /**
     * Generate and write the binding information to the config file.
     */
    private void generateBindings() {
        // once routes and templates have been generated, bind particular
        // templates to particular hosts in the network
		
        try {
            _file.write("\n# Router bindings\n");
            for (int i = 0; i < _routerAddresses.size(); i++) {
                _file.write("bind "
                        + getAddressString(((Long) _routerAddresses.get(i)).longValue())
                        + " router_template"
                        + _randomGenerator.nextInt(_numRouterTemplates) + "\n");
            }
            _file.write("\n# Host bindings\n");
			
            // get cidr string
            /*long addressBits = getAddressBits(_middleClass.getCIDRString());
            int cidrMask = getCidrMask(_middleClass.getCIDRString());
			
            // see how many bits we have for host address space
            int bitsToPlayWith = 32 - cidrMask - 1;

            if (bitsToPlayWith > 32 | bitsToPlayWith < 2) {
                System.err.println("Invalid CIDR String");
            }
          
            // if the 2 ^ num_bits is smaller than _numberOfHosts, we need to change
            // the number of hosts
            int possibleNumHosts = (int) java.lang.Math.pow((double) 2,
                    (double) bitsToPlayWith);

            if (possibleNumHosts < _numberOfHosts) {
                _numberOfHosts = possibleNumHosts;
            }
           
            // generate bindings
            // this is done in this fashion to avoid overlaps, only problem is that
            // they are going to be pretty evenly distributed in the address space
            long hostBits = 1;

            for (int i = 0; i < _numberOfHosts; i++) {
                hostBits += _randomGenerator.nextInt(30) + 1; // 30 is arbitrary
                if (hostBits
                        >= java.lang.Math.pow((double) 2,
                                (double) bitsToPlayWith)) {
                    break;   // we are done, screw the _numberOfHosts count
                }
                // or in the bits
                long address = getAddressBits(_middleClass.getCIDRString())
                        | hostBits;

                // check to see if this address has been allocated as a router address
                for (int r = 0; r < _routerAddresses.size(); r++) {
                    if (address == ((Long) _routerAddresses.get(r)).longValue()) {
                        address++; // this will make sure last bit isn't 1
                    }
                }
                _file.write("bind " + getAddressString(address)
                        + " host_template"
                        + _randomGenerator.nextInt(_numHostTemplates) + "\n");
			   
            }*/
				
				for (int i=0; i<_hostBindings.size(); i++) {
				   Long l = (Long)_hostBindings.get(i);
				   long addr = l.longValue();
					_file.write("bind " + getAddressString(addr) 
					   + " host_template" + _randomGenerator.nextInt(_numHostTemplates)
						+ "\n");
				}
            _file.flush(); 
		
        } catch (IOException ioe) {
            ioe.printStackTrace();
            System.err.println(ioe.getMessage());
		
        }
    }

    /**
     * Calculate number of hosts by using either complexity level or a 
     * hueristic based on the number of routers and layers.  This function
     * should also consider the size of the address space
     *
     * @return number of hosts to generate templates for
     */
    private int calculateNumberOfHosts() {
	  
        // this is a very BASIC heuristic right now, we use complexity level 3 if
        // the number of routers and layers are given
        int complexity = _middleClass.getComplexityLevel();

        if (complexity <= 0 || complexity > 3) {
            // To save execution, set complexity to 1
            System.err.println("Invalid complexity setting.  Using complexity=1");
            complexity = 3;
        }
		
        return ((_randomGenerator.nextInt(1000) + 1) * complexity);
	
    }	
   
    /**
     * Calculate the number of routers.  This will be based on complexity level, 
     * or the number of routers if defined by the _middleClass
     *
     * @return number of routers 
     */
    private int calculateNumberOfRouters() {
   
        // first, see if the number or routers was specified by the _middleClass
        // if so, use that number
        // if not, use complexity level to generate number
        int numRouters = _middleClass.getNumberOfRouters();

        if (numRouters != 0) {
            return numRouters;
        }
		 
        int complexity = _middleClass.getComplexityLevel();

        if (complexity <= 0 || complexity > 3) {
            // To save execution, set complexity to 1
            System.err.println("Invalid complexity setting.  Using complexity=1");
            complexity = 1;
        }		 
	  
        // simple heuristic for now.  We take a random number between 0 and 4,
        // add one and multiply by the complexity level.  This can be changed if
        // necessary.	  
        int randomInt = _randomGenerator.nextInt(4) + 1;

        randomInt++; // to make a value between 1 and 5
        return (randomInt * complexity);
	  
    }
   
    /**
     * Calculate the number of heirarchical layers to be used.  If set by the 
     * _middleClass, this will be used.  Otherwise, the complexity level will
     * be used to generate number or layers using a simple heuristic
     *
     * @return number of layers
     */
    private int calculateNumberOfLayers() {
   
        // first, see if the number of layers was specified by the _middleClass
        // if so, use that number
        // if not, use complexity level to generate number
        int numLayers = _middleClass.getNumberOfLayers();

        if (numLayers != 0) {
            return numLayers;
        }
		 
        int complexity = _middleClass.getComplexityLevel();

        if (complexity <= 0 || complexity > 3) {
            // To save execution, set complexity to 1
            System.err.println("Invalid complexity setting.  Using complexity=1");
            complexity = 1;
        }		 
	  
        // simple heuristic for now.  We take a random number between 0 and 2,
        // add one and multiply by the complexity level.  This can be changed if
        // necessary.	  
        int randomInt = _randomGenerator.nextInt(2 + 1);

        randomInt++; // to make a value between 1 and 3
        return (randomInt * complexity);
    }	

    /**
     * @return array of AddressNode's with the layerNumber,
     * addressBits, and cidrMask fields properly set.  Returns
     * an empty array if the address space is exhausted.
     */
    private AddressNode[] getSubdomains(AddressNode node) {
        AddressNode[] ret = new AddressNode[node.numberOfDomains];
        
        // calculate the new cidr mask based upon number of domains
        int newCidrMask = node.cidrMask
                + (int) Math.ceil(Math.log(node.numberOfDomains) / Math.log(2));
            
        // need to check if node.numberOfDomains is a power of two
        int powOfTwo = 2;

        for (int i = 0; i < 31; i++) {
            if (powOfTwo == node.numberOfDomains) {
                newCidrMask++;
                break;
            }
            powOfTwo <<= 1;
        }
        
        // need to do some bounds checking to determine if exhaust
        // the address space
        if (newCidrMask >= 32) {
            //return ret;        
            return new AddressNode[0];
        }
            
        for (int i = 0; i < node.numberOfDomains; i++) {
            ret[i] = new AddressNode();
            ret[i].layerNumber = node.layerNumber + 1;
            ret[i].addressBits = node.addressBits
                    | ((i + 1) << (32 - newCidrMask));
            ret[i].cidrMask = newCidrMask;
            ret[i].isRouter = i == 0 ? false : true;
        }
        
        return ret;
    }
    
    /**
     * @return the string representation of a ip address
     */
    private static String getAddressString(long l) {

        long[]temp = new long[4];

        /*
         temp[0] = l & 0xFF000000;
         temp[1] = l & 0x00FF0000;
         temp[2] = l & 0x0000FF00;
         temp[3] = l & 0x000000FF;
         */
        temp[0] = (l >> 24) & 0xff;
        temp[1] = (l >> 16) & 0xff;
        temp[2] = (l >> 8) & 0xff;
        temp[3] = l & 0xff;
        String s = Long.toString(temp[0]) + "." + Long.toString(temp[1]) + "."
                + Long.toString(temp[2]) + "." + Long.toString(temp[3]);

        return s;
            
    } 

    /**
     * @param addr cidr address in the slash notation
     * (e.g. 10.1.0.0/16)
     * @return 32-bit representation of the cidr address in string
     */
    private long getAddressBits(String addr) {
        byte[] tmp = new byte[4];
        int beginIndex = 0; 
        int fromIndex = 0;

        // parse past the dots
        for (int i = 0; i < tmp.length - 1; i++) {
            fromIndex = addr.indexOf('.', ++fromIndex);
            tmp[i] = (Byte.valueOf(addr.substring(beginIndex, fromIndex))).byteValue();
            beginIndex = fromIndex + 1;
        }

        // parse till the slash
        fromIndex = addr.indexOf('/', fromIndex);
        tmp[3] = (Byte.valueOf(addr.substring(beginIndex, fromIndex))).byteValue();

        // create a long and return it
        return (long) ((tmp[0] << 24) | (tmp[1] << 16) | (tmp[2] << 8) | tmp[3]);
    }

    /**
     * @param addr cidr address in slash notation
     * (e.g. 10.1.0.0/16)
     * @return the CIDR mask specified by the cidr address
     */
    private int getCidrMask(String addr) {

        // parse till the slash
        return (Integer.valueOf(addr.substring(addr.indexOf('/') + 1))).intValue();
    }

    /**
     * Outputs a random router template personality
     *
     * @param   i  index into routerSelectedFingerprints
     */
    private void writeRandomRouterPersonality(int index) throws IOException {
        _file.write("set router_template" + index + " personality ");
        int randomInt = _randomGenerator.nextInt(_middleClass.getNumberOfRouterSelectedFingerprints());

        _file.write("\"" + _middleClass.getRouterFingerprintName(randomInt)
                + "\"\n");
    }
	
    /**
     * Outputs a random host template personality
     *
     * @param   i  index into hostSelectedFingerprints
     */
    private void writeRandomHostPersonality(int index) throws IOException {
        _file.write("set host_template" + index + " personality ");
        int randomInt = _randomGenerator.nextInt(_middleClass.getNumberOfHostSelectedFingerprints());

        _file.write("\"" + _middleClass.getHostFingerprintName(randomInt)
                + "\"\n");
    }
	
    /**
     * Outputs a random host template tcp action
     */
    private void writeRandomTCPAction(String templateName) throws IOException {
        // form is like "set routertwo default tcp action reset"
        // we should only set this 50% of the time
        if (1 == _randomGenerator.nextInt(2)) {
            // 0=block, 1=open, 2=reset
            int test = _randomGenerator.nextInt(3);

            if (test == 0) {
                _file.write("set " + templateName
                        + " default tcp action block\n");
            } else if (test == 1) {
                _file.write("set " + templateName + " default tcp action open\n");
            } else if (test == 2) {
                _file.write("set " + templateName
                        + " default tcp action reset\n");
            }
        }
    }
	
    /**
     * Outputs a random host template udp action
     */
    private void writeRandomUDPAction(String templateName) throws IOException {
        // form is like "set routertwo default udp action reset"
        // we should only set this 50% of the time
        if (1 == _randomGenerator.nextInt(2)) {
            // 0=block, 1=open, 2=reset
            int test = _randomGenerator.nextInt(1000) + 1;

            if (test == 1) {
                _file.write("set " + templateName
                        + " default udp action block\n");
            } else if (test > 2) {
                _file.write("set " + templateName + " default udp action open\n");
            } else if (test == 2) {
                _file.write("set " + templateName
                        + " default udp action reset\n");
            }
        }
    }
	
    /**
     * Outputs a random host template icmp action
     */
    private void writeRandomICMPAction(String templateName) throws IOException {
        // form is like "set routertwo default icmp action reset"
        // we should only set this 50% of the time
        if (1 == _randomGenerator.nextInt(2)) {
            // 0=block, 1=open, 2=reset
            int test = _randomGenerator.nextInt(1000) + 1;

            if (test == 1) {
                _file.write("set " + templateName
                        + " default icmp action block\n");
            } else if (test > 2) {
                _file.write("set " + templateName
                        + " default icmp action open\n");
            } else if (test == 2) {
                _file.write("set " + templateName
                        + " default icmp action reset\n");
            }
        }
    }
	
    /**
     * Outputs a random host template services
     */
    private void writeRandomServices(String templateName) throws IOException {
        Vector services = _middleClass.getSelectedServices();

        for (int i = 0; i < services.size(); i++) {
            switch (_randomGenerator.nextInt(2)) {
            case 0:
                Service service = (Service) services.elementAt(i);

                _file.write("add " + templateName + " " + service.getProtocol()
                        + " port " + service.getPort() + " \""
                        + service.getServiceName() + "\"\n");
                break;

            case 1:
            default:
                break;
            }
        }
    }
	
    /**
     * Outputs a random host template uptime
     */
    private void writeRandomUptime(String templateName) throws IOException {
        if (1 >= _randomGenerator.nextInt(10)) { // 20 percent of time
            int uptime = _randomGenerator.nextInt(500000); // again, arbitrary

            _file.write("set " + templateName + " uptime "
                    + Integer.toString(uptime + 1) + "\n");
        }
    }
	
    /**
     * Outputs a random host template droprate
     */
    private void writeRandomDroprate(String templateName) throws IOException {
        if (2 >= _randomGenerator.nextInt(10)) { // 30 precent of time
            // random between 0 and 1
            int firstDigit = _randomGenerator.nextInt(10);
            int secondDigit = _randomGenerator.nextInt(10);

            _file.write("set " + templateName + " droprate in 0."
                    + Integer.toString(firstDigit)
                    + Integer.toString(secondDigit) + "\n");
        }
    }
	
    /**
     * Outputs a random host template uid
     */
    private void writeRandomUid(String templateName) throws IOException {
        if (0 == _randomGenerator.nextInt(10)) { // 10 precent of time
            // random between 0 and 1
            int firstDigit = _randomGenerator.nextInt(10);
            int secondDigit = _randomGenerator.nextInt(10);

            _file.write("set " + templateName + " uid "
                    + (Integer.toString(_randomGenerator.nextInt(300) + 1)));
            if (0 == _randomGenerator.nextInt(2)) {
                _file.write(" gid "
                        + (Integer.toString(_randomGenerator.nextInt(200) + 1)));
            }
            _file.write("\n");
        }
    }

    /**
     * Simple test driver for <code>RandomNetGeneration</code>.
     *
     * @param args argument array
     */
    public static void main(String[] args) {}
}
