/**
 *	Copyright (c) 2003, Andrew B. Smith and Jason M. Fox
 *	All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 *	modification, are permitted provided that the following conditions are met:
 *
 *	* Redistributions of source code must retain the above copyright notice, 
 *	  this list of conditions and the following disclaimer.
 *	* Redistributions in binary form must reproduce the above copyright notice, 
 *	  this list of conditions and the following disclaimer in the documentation 
 *	  and/or other materials provided with the distribution.
 *	* Neither the name of the CERIAS nor the names of its contributors 
 *	  may be used to endorse or promote products derived from this software without 
 *	  specific prior written permission.
 *
 *	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 *	ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 *	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 *	IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 *	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 *	BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *	DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY 
 *	OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 *	OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 *	OF THE POSSIBILITY OF SUCH DAMAGE.
 */

import java.io.*;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.Date;


/**
 * The MiddleClass is in charge of representing data collected in the rn.conf or 
 * the gui to the generation class.
 * <p>
 * Special notes:
 *		(1) If a complexity level is specified, the numberOfRouter, numberOfLayers, 
 * 			layers private date will be useless.  Basically, the complexity level
 *			setting	is for users who do not want more fine grained control 
 *			over the random network. 
 */
public class MiddleClass {

    /**
     * Specified in the rn.conf or set by the RandomNetGui, this represents
     * the number of routers to appear in the honeyd configuration file.
     */
    private int _numberOfRouters;
	
    /**
     * Specified in the rn.conf or set by the <code>RandomNetGui</code>, this 
     * represents the number of heirarchical layers to appear in the honeyd 
     * configuration file
     */
    private int _numberOfLayers;
	
    /**
     * The complexity level is a beginner setting.  Basically, it will choose the 
     * the number of layers and number of routers for the user
     */
    private int _complexityLevel;
	
    /**
     * Holds all of the available fingerprints as specified in the nmap.prints file
     */
    private Fingerprint _availableFingerprints[];	
	
    /**
     * The user specified selected fingerprints for routers
     */
    private Vector _routerSelectedFingerprints;
	
    /**
     * The user specified selected fingerprints for hosts
     */
    private Vector _hostSelectedFingerprints;

    /**
     * The number of available fingerprints as listed in the nmap.prints file
     */
    private int _numberOfAvailableFingerprints;
	
    /**
     * holds all of the available "fake" services that reside in the honeyd scripts
     * directory
     */
    private Service _availableServices[];	
	
    /**
     * "Fake" services to use on systems in the RandomNet 
     */
    private Vector _selectedServices;

    /**
     * The honeyd config file in which to put the random net generated
     */
    private String _honeydConfigFilename;
	
    /**
     * The RandomNet configuration filename
     */
    private String _randomNetConfigFilename; 

    /**
     * The location where the honeyd directory resides
     */
    private String _honeydDirectory;
	
    /**
     * The location of the nmap.prints file to read fingerprints from
     */
    private String _nmapFingerprintsFilename;
	
    /**
     * The CIDR address string for which to allocate routers and hosts into
     */
    private String _cidrString; 
	
    /**
     * holds the indices that we will later place in the routerSelectedFingerprints
     */
    private Vector _routerFingerprintIndices;
	
    /**
     * holds the indices that we will later place in the hostSelectedFingerprints
     */
    private Vector _hostFingerprintIndices;
	
    /**
     * holds the indices that we will later place in the selectedServices
     */
    private Vector _serviceIndices; 
	  
    /**
     * The basic class constructor which loads the configuration information
     * and starts a gui if necessary.
     *
     * @param randomNetConfigFilename	location of RandomNet config file
     * @param honeydConfigFilename		file that RandomNet will output honeyd config 
     *									file to	 
     * @param guiEnabled				true if gui mode should be turned on
     */
    public MiddleClass(String randomNetConfigFilename, String honeydConfigFilename, 
            boolean guiEnabled) {
        // store output file and random net config filename
        _honeydConfigFilename = honeydConfigFilename;
        _randomNetConfigFilename = randomNetConfigFilename;
        // create storage vectors
        _routerSelectedFingerprints = new Vector();
        _hostSelectedFingerprints = new Vector();
        _selectedServices = new Vector();
        _hostFingerprintIndices = new Vector();
        _routerFingerprintIndices = new Vector();
        _serviceIndices = new Vector();
        // loads rn.conf file for location of honeydDirecotry and nmap.prints file
        loadRandomNetConfig(_randomNetConfigFilename, guiEnabled);
        // fill in the fingerprints structure
        fillAvailableFingerprints(_nmapFingerprintsFilename);
        fillSelectedFingerprints();
        // fill in the services structure
        fillAvailableServices(_honeydDirectory);
        fillSelectedServices();
		
        // create gui if enabled
        if (guiEnabled) {
            RandomNetGui gui = new RandomNetGui(this);
        } else {
		      generate();
		  }
		
    } 
	
    /**
     * loads the RandomNet configuration file information
     *
     * @param randomNetConfigFilename	filename of the RandomNet config file
     * @param okToFail					it is ok to fail to find a config file
     *									if gui-mode is enabled
     */
    private void loadRandomNetConfig(String randomNetConfigFilename, 
            boolean okToFail) {	
		
        // open Randomnet config file
        BufferedReader randomNetConfigReader = null;

        try {
            randomNetConfigReader = new BufferedReader(new FileReader(new File(randomNetConfigFilename)));
        } catch (FileNotFoundException e) {
            System.out.println("RandomNet config file not found at: "
                    + randomNetConfigFilename);
            System.exit(0);
        }
        // parse honeyd directory and nmap.prints location out of config file
        _honeydDirectory = null;
        _nmapFingerprintsFilename = null;
        _cidrString = "";
        _honeydConfigFilename = "./config.random";
        _complexityLevel = 0;
        _numberOfRouters = 0;
        _numberOfLayers = 0;
        try {
            String configString = randomNetConfigReader.readLine();

            while (configString != null) {
                StringTokenizer configTokenizer = new StringTokenizer(configString,
                        "=");

                if (configTokenizer.hasMoreTokens()) {
                    String tempString = configTokenizer.nextToken();

                    if (tempString.equals("HONEYDDIR")) {
                        _honeydDirectory = configTokenizer.nextToken();
                    } else if (tempString.equals("NMAPPRINTS")) {
                        _nmapFingerprintsFilename = configTokenizer.nextToken();
                    } else if (tempString.equals("cidr_address")) {
                        _cidrString = configTokenizer.nextToken();
                    } else if (tempString.equals("HONEYDCONFIG")) {
                        _honeydConfigFilename = configTokenizer.nextToken();
                    } else if (tempString.equals("num_routers")) {
                        _numberOfRouters = Integer.parseInt(configTokenizer.nextToken());
                    } else if (tempString.equals("num_layers")) {
                        _numberOfLayers = Integer.parseInt(configTokenizer.nextToken());
                    } else if (tempString.equals("complexity_level")) {
                        _complexityLevel = Integer.parseInt(configTokenizer.nextToken());
                        if (_complexityLevel > 3 || _complexityLevel < 1) {
                            _complexityLevel = 0;
                        }	
                    } else if (tempString.equals("router_fingerprint")) {
                        StringTokenizer tok = new StringTokenizer(configTokenizer.nextToken(),
                                "#");

                        if (tok.hasMoreTokens()) {
                            _routerFingerprintIndices.add((tok.nextToken()).trim());
                        }	
                    } else if (tempString.equals("host_fingerprint")) {
                        StringTokenizer tok = new StringTokenizer(configTokenizer.nextToken(),
                                "#");

                        if (tok.hasMoreTokens()) {
                            _hostFingerprintIndices.add((tok.nextToken()).trim());
                        }	
                    } else if (tempString.equals("use_service")) {
                        StringTokenizer tok = new StringTokenizer(configTokenizer.nextToken(),
                                "#");

                        if (tok.hasMoreTokens()) {
                            _serviceIndices.add((tok.nextToken()).trim());
                        }	
                    }
                }
                configString = randomNetConfigReader.readLine();
            }
            randomNetConfigReader.close();
        } catch (Exception e) {
            System.out.println("Error(1): " + e.toString());
            System.exit(0);
        }
		
    } 
	
    /**
     * loads the _availableFingerprints structure from the nmap.prints file
     *
     * @param nmapFingerprints	filename of nmap.prints
     */
    private void fillAvailableFingerprints(String nmapFingerprints) {
		
        /*
         * Note: come back and re-write to use vectors so we don't go through
         * file twice...
         */
		
        BufferedReader fingerprintsIn = null;	 // wrapper around nmap.prints
	
        // First, we look for the nmap.prints file
        try {
            fingerprintsIn = new BufferedReader(new FileReader(new File(nmapFingerprints)));
        } catch (FileNotFoundException e) {
            System.out.println("nmap.prints file not found at location: "
                    + nmapFingerprints);
            System.exit(0);
        }
        // Count the number of fingerprints in the nmap.prints file
        System.out.print("Reading " + nmapFingerprints + " ...");
        try {
            String oneLiner = fingerprintsIn.readLine();

            while (oneLiner != null) {
                StringTokenizer token = new StringTokenizer(oneLiner);

                if (token.hasMoreTokens()) {
                    String tempString = token.nextToken();

                    if (tempString.equals("Fingerprint")) {
                        _numberOfAvailableFingerprints++;
                    }
                }
                oneLiner = fingerprintsIn.readLine();
            }
			
        } catch (IOException e) {
            System.out.println("Error(2): " + e.toString());
            System.exit(0);
        }
        System.out.println(" done.");
        // Create and fill in the _availableFingerprints data structure
        _availableFingerprints = new Fingerprint[_numberOfAvailableFingerprints];
        try {
            // HACK -> I don't know how to reset the buffered reader, so I close it and
            // open it again
            fingerprintsIn.close();
            fingerprintsIn = new BufferedReader(new FileReader(new File(nmapFingerprints)));		
        } catch (Exception e) {
            System.out.println("Error(3): " + e.toString());
            System.exit(0);
        }
        for (int i = 0; i < _numberOfAvailableFingerprints; i++) {
            String oneLiner;
            StringTokenizer token;

            // jump to next fingerprint name
            while (true) {
                try {
                    oneLiner = fingerprintsIn.readLine();
                    token = new StringTokenizer(oneLiner);
                    if (token.hasMoreTokens()) {
                        if ((token.nextToken()).equals("Fingerprint")) {
                            break;
                        }
                    }
                } catch (IOException e) {
                    System.out.println("Error(4): " + e.toString());
                    System.exit(0);
                }
            }
            // next token should be fingerprint name
            String fingerprintName = "";

            while (token.hasMoreTokens()) {
                fingerprintName += token.nextToken() + " ";
            }
            _availableFingerprints[i] = new Fingerprint(fingerprintName, i);
        }
		
    } 
	
    /**
     * fills in the selectedFingerprints structure
     */
    private void fillSelectedFingerprints() {
		
        // copy fingerprints from index array into selected fingerprints
        if (!(_routerFingerprintIndices.isEmpty())) {
            for (int i = 0; i < _routerFingerprintIndices.size(); i++) {
                int index = Integer.parseInt((String) _routerFingerprintIndices.get(i));

                _routerSelectedFingerprints.add(_availableFingerprints[index]);
            }	
        }
        if (!(_hostFingerprintIndices.isEmpty())) {
            for (int i = 0; i < _hostFingerprintIndices.size(); i++) {
                int index = Integer.parseInt((String) _hostFingerprintIndices.get(i));

                _hostSelectedFingerprints.add(_availableFingerprints[index]);
            }	
        }
    } 
	
    /**
     * fills in the _availableServices structure
     *
     * @param honeydDirectory directory of honeyd source and executables
     */
    private void fillAvailableServices(String honeydDirectory) {
	
        // open the scripts directory
        String scriptsDirectoryString = honeydDirectory + "scripts";
        File scriptsDirectory = null;

        try {
            scriptsDirectory = new File(scriptsDirectoryString);
            if (!(scriptsDirectory.isDirectory())) {
                System.out.println("Error opening scripts directory: "
                        + scriptsDirectoryString);
                System.exit(0);		
            }
        } catch (Exception e) {
            System.out.println("Could not find scripts directory at location: "
                    + scriptsDirectoryString);
            System.exit(0);
        }
		
        // parse the available services
        String directoryListing[] = scriptsDirectory.list();

        _availableServices = new Service[directoryListing.length];
        for (int i = 0; i < directoryListing.length; i++) {

            // FIX ME
            // bind the port number and protocol to service in the gui
            int port;
            String proto;

            if (directoryListing[i].equals("router-telnet.pl")) {
                port = 23;
                proto = "tcp";
            } else if (directoryListing[i].equals("test.sh")) {
                port = 22;
                proto = "tcp";
            } else if (directoryListing[i].equals("web.sh")) {
                port = 80;
                proto = "tcp";
            } else {
                port = 1024;
                proto = "tcp";
            }

            _availableServices[i] = new Service(directoryListing[i], 100, port,
                    proto);
        }
	
    }
	
    /**
     * fills in the _selectedServices vector by taking the _availableServices
     * array and copying out those indices in the serviceIndices vector
     * Note: serviceIndices are strings not true indices
     */
    public void fillSelectedServices() {
	
        // if there are no selected, return
        if (_serviceIndices.isEmpty()) {
            return;
        }
		
        // bad search algorithm.  But, the number of services are so small
        // that this shouldn't be a problem.		
        for (int i = 0; i < _availableServices.length; i++) {
            for (int j = 0; j < _serviceIndices.size(); j++) {
                if (_availableServices[i].getServiceName().equals((String) _serviceIndices.get(j))) {
                    _selectedServices.add(_availableServices[i]);
                    // replace service name with index number in availableServices
                    _serviceIndices.setElementAt(Integer.toString(i), j);
                }
            }
        }	
		
    } 
	
    /******************************************************************************
     ********************** PUBLIC METHODS ****************************************
     ******************************************************************************
     */
	
    /**
     * outputs a new RandomNet configuration file
     */
    public void saveConfigFile() {
		
        // open the file for writing
        FileWriter newConfigFile = null;

        try {
            newConfigFile = new FileWriter(_randomNetConfigFilename);
            // add header
            newConfigFile.write("########## generated by RandomNetGui\n");
            newConfigFile.write("########## " + (new Date()).toString() + "\n\n");

            // add path information
            newConfigFile.write("# Path Information\n");
            newConfigFile.write("HONEYDDIR=" + _honeydDirectory + "\n");
            newConfigFile.write("NMAPPRINTS=" + _nmapFingerprintsFilename + "\n");
            newConfigFile.write("HONEYDCONFIG=" + _honeydConfigFilename + "\n\n");

            // add basic configuration information
            newConfigFile.write("# Basic Configuration Information\n");
            if (_complexityLevel != 0) {
                newConfigFile.write("complexity_level=" + _complexityLevel
                        + "\n");
            }
            if (_numberOfRouters != 0) {
                newConfigFile.write("num_routers=" + _numberOfRouters + "\n");
            }
            if (_numberOfLayers != 0) {
                newConfigFile.write("num_layers=" + _numberOfLayers + "\n");
            }
            newConfigFile.write("cidr_address=" + _cidrString + "\n\n");	
            // add fingerprints to use
            newConfigFile.write("# Router Fingerprint information\n");
            for (int i = 0; i < _routerSelectedFingerprints.size(); i++) {
                Fingerprint tempFingerprint = (Fingerprint)
                        _routerSelectedFingerprints.get(i);

                newConfigFile.write("router_fingerprint="
                        + (tempFingerprint.getId()));
                newConfigFile.write(" # " + _routerSelectedFingerprints.get(i)
                        + "\n");			
            }
            newConfigFile.write("\n");
            newConfigFile.write("# Host Fingerprint information\n");
            for (int i = 0; i < _hostSelectedFingerprints.size(); i++) {
                Fingerprint tempFingerprint = (Fingerprint)
                        _hostSelectedFingerprints.get(i);

                newConfigFile.write("host_fingerprint="
                        + (tempFingerprint.getId()));
                newConfigFile.write(" # " + _hostSelectedFingerprints.get(i)
                        + "\n");			
            }
            newConfigFile.write("\n");							// add services to use
            newConfigFile.write("# Service information\n");
            for (int i = 0; i < _selectedServices.size(); i++) {
                Service tempService = (Service)
                        _selectedServices.get(i);

                newConfigFile.write("use_service="
                        + (tempService.getServiceName()));
                newConfigFile.write("\n");
            }
            newConfigFile.write("\n");		
            // flush and close the file
            newConfigFile.flush();
            newConfigFile.close();

        } catch (IOException ioe) {
            ioe.printStackTrace();
            System.err.println(ioe.getMessage());
            System.exit(0);
        }	
		
    }  
	
    /**
     * calls the RandomNetGeneration class to generate the honeyd config file
     */
    public void generate() {
	
        // it is assumed at this point that all of the private member data
        // has been filled in correctly. So, we simply need to create a
        // RandomNetGeneration class and do the generation
        try {
		      System.out.print("Generating honeyd config file ... ");
            RandomNetGeneration rng = new RandomNetGeneration(this);

            rng.run();
				System.out.println("done");
        } catch (IOException ioe) {
            ioe.printStackTrace();
            System.err.println(ioe.getMessage());
        }
		
    } 
	 
    /**
     * copies fingerprints from the _availableFingerprints array to the
     * _routerSelectedFingerprints array
     *
     * @param	indices		the indices to copy
     */
    public void selectRouterFingerprints(int indices[]) {
        // delete the old
        _routerSelectedFingerprints.removeAllElements();
		
        // copy each finger print from available to selected array
        for (int i = 0; i < indices.length; i++) {
            _routerSelectedFingerprints.add(_availableFingerprints[(indices[i])]);
        }
		
    }  
	
    /**
     * copies fingerprints from the _availableFingerprints array to the
     * _hostSelectedFingerprints array
     *
     * @param	indices		the indices to copy
     */
    public void selectHostFingerprints(int indices[]) {
        // delete the old
        _hostSelectedFingerprints.removeAllElements();
		
        // copy each finger print from available to selected array
        for (int i = 0; i < indices.length; i++) {
            _hostSelectedFingerprints.add(_availableFingerprints[(indices[i])]);
        }
		
    } 	
	
    /**
     * copies services from the _availableServices array to the
     * _selectedServices array
     *
     * @param	indices		the indices to copy
     */
    public void selectServices(int indices[]) {
        // delete the old
        _selectedServices.removeAllElements();
		
        // copy each finger print from available to selected array
        for (int i = 0; i < indices.length; i++) {
            _selectedServices.add(_availableServices[(indices[i])]);
        }
		
    }
	
    /**
     * sets the honeyd output file
     *
     * @param output filename
     */
    public void setHoneydConfigFilename(String filename) {
        _honeydConfigFilename = filename;
    } 
	
    /**
     * sets the random net config filename
     *
     * @param RandomNet config filename
     */
    public void setRandomNetConfigFilename(String filename) {
        _randomNetConfigFilename = filename;
    } 
	
    /**
     * sets the CIDR String
     *
     * @param CIDR String
     */
    public void setCIDR(String cidr) {
        _cidrString = cidr;
    } 
	
    /**
     * sets the complexity level
     *
     * @param the complexity level to use for generation
     */
    public void setComplexity(int complexityLevel) {
        _complexityLevel = complexityLevel;
    } 
	
    /**
     * sets the number of Routers
     * @param the number of routers
     */
    public void setNumberOfRouters(int num) {
        _numberOfRouters = num;
    }
	
    /**
     * sets the number of layers
     * @param the number of layers to use
     */
    public void setNumberOfLayers(int num) {
        _numberOfLayers = num;
    } 

    /**
     * @return CIDR address string
     */
    public String getCIDRString() {
        return _cidrString;
    } 
	
    /**
     * this function returns the availableFingerprints
     *
     * @return the array of availableFingerprints
     */
    public Fingerprint[] getAvailableFingerprints() {
        return _availableFingerprints;
    } 
	
    /**
     * @return the fingerprint indices that should be selected
     */
    public Vector getRouterFingerprintIndices() {
        return _routerFingerprintIndices;
    }
    
    /**
     * @return the fingerprint indices that should be selected for host
     */
    public Vector getHostFingerprintIndices() {
        return _hostFingerprintIndices;
    }
	
    /**
     * @return the number of router selected fingerprints
     */
    public int getNumberOfRouterSelectedFingerprints() {
        return _routerSelectedFingerprints.size();
    } 
	
    /**
     * @return the number of host selected fingerprints
     */
    public int getNumberOfHostSelectedFingerprints() {
        return _hostSelectedFingerprints.size();
    } 
	 
    /**
     * @param num the index into the _routerSelectedFingerprints vector
     * @return the name of the selected fingerprint in question
     */
    public String getRouterFingerprintName(int num) {
        return ((Fingerprint) _routerSelectedFingerprints.get(num)).getFingerprintName();	
    }
	
    /**
     * @param num the index into the _hostSelectedFingerprints vector
     * @return the name of the selected fingerprint in question
     */
    public String getHostFingerprintName(int num) {
        return ((Fingerprint) _hostSelectedFingerprints.get(num)).getFingerprintName();
    }
	 
    /**
     * this function returns the availableServices
     *
     * @return the array of availableServices
     */
    public Service[] getAvailableServices() {
        return _availableServices;
    }  
	
    /**
     * @return the service indices that should be selected
     */
    public Vector getServiceIndices() {
        return _serviceIndices;
    }
	  
    /**
     * @return honeyd configuration file name
     */
    public String getHoneydConfigFilename() {
        return _honeydConfigFilename;
    }
	
    /**
     * @return RandomNet Configuration filename
     */
    public String getRandomNetConfigFilename() {
        return _randomNetConfigFilename;
    }
	
    /**
     * @return complexity level setting
     */
    public int getComplexityLevel() {
        return _complexityLevel;
    }
	
    /**
     * @return number of routers
     *
     */
    public int getNumberOfRouters() {
        return _numberOfRouters;
    } 
	
    /**
     * @return number of layers
     */
    public int getNumberOfLayers() {
        return _numberOfLayers;
    }

    public Vector getSelectedServices() {
        return _selectedServices;
    }
}
