/*
 *  idmapd.c
 *
 *  Userland daemon for idmap.
 *
 *  Copyright (c) 2002 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Marius Aamodt Eriksen <marius@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * XXX race condition in checking .../idmap ; use a short timeout to remedy
 */

#define _GNU_SOURCE
#include <sys/time.h>
#include <sys/types.h>
#include <sys/poll.h>
#include <sys/socket.h>

#include <nfs_idmap.h>

#include <err.h>
#include <errno.h>
#include <event.h>
#include <fcntl.h>
#include <dirent.h>
#include <unistd.h>
#include <netdb.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <grp.h>
#include <limits.h>
#undef _GNU_SOURCE 

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "cfg.h"
#include "queue.h"

#ifndef PIPEFS_DIR
#define PIPEFS_DIR  "/rpc_pipefs/nfs"
#endif

#ifndef NFSD_DIR
#define NFSD_DIR  "/proc/net/rpc"
#endif

#ifndef NFS4NOBODY_USER
#define NFS4NOBODY_USER "nobody"
#endif

#ifndef NFS4NOBODY_GROUP
#define NFS4NOBODY_GROUP "nobody"
#endif

/* From Niels */
#define CONF_SAVE(w, f) do {			\
	char *p = f;				\
	if (p != NULL)				\
		(w) = p;			\
} while (0)

struct idmap_client {
	int                        ic_fd;
	int                        ic_dirfd;
	char                       ic_clid[30];
	char                       ic_path[PATH_MAX]; 
	int                        ic_scanned;
	struct event               ic_event;
	char                      *ic_id;
	TAILQ_ENTRY(idmap_client)  ic_next;
};

TAILQ_HEAD(idmap_clientq, idmap_client);

static void dirscan_cb(int, short, void *);
static void clntscan_cb(int, short, void *);
static int  openpipe(struct idmap_client *);
static void idmap_cb(int, short, void *);
static int  validate_ascii(char *, u_int32_t);

static void reply_idtoname(struct idmap_msg *);
static void reply_nametoid(struct idmap_msg *);

static int nfsd_open(char *);

size_t  strlcat(char *, const char *, size_t);
size_t  strlcpy(char *, const char *, size_t);
ssize_t atomicio(ssize_t (*)(), int, void *, size_t);
int     daemon(int, int);

static int verbose = 0;
static char domain[512];
static char pipefsdir[PATH_MAX];
static char *nobodyuser, *nobodygroup;
static uid_t nobodyuid;
static gid_t nobodygid;
static struct idmap_client nfsd_ic[2];

/* Used by cfg.c */
char *conf_path;

int
main(int argc, char **argv)
{
	int fd, opt, fg = 0, nfsdret;
	struct idmap_clientq icq;
	struct event rootdirev, clntdirev;
	struct passwd *pw;
	struct group *gr;
	struct stat sb;
	char *xpipefsdir = NULL;

	conf_path = SYSCONFDIR "/idmapd.conf";
	nobodyuser = NFS4NOBODY_USER;
	nobodygroup = NFS4NOBODY_GROUP;
	strlcpy(pipefsdir, PIPEFS_DIR, sizeof(pipefsdir));

#define GETOPTSTR "vfd:p:U:G:c:"
        while ((opt = getopt(argc, argv, GETOPTSTR)) != -1)
		if (opt == 'c')
			conf_path = optarg;

        optind = 1;

	if (stat(conf_path, &sb) == -1 && (errno == ENOENT || errno == EACCES)) {
		warn("Skipping configuration file \"%s\"", conf_path);
	} else {
		conf_init();
		verbose = conf_get_num("General", "Verbosity", 0);
		CONF_SAVE(xpipefsdir, conf_get_str("General", "Pipefs-Directory"));
		if (xpipefsdir != NULL)
			strlcpy(pipefsdir, xpipefsdir, sizeof(pipefsdir));
		CONF_SAVE(nobodyuser, conf_get_str("Mapping", "Nobody-User"));
		CONF_SAVE(nobodygroup, conf_get_str("Mapping", "Nobody-Group"));
	}

        while ((opt = getopt(argc, argv, GETOPTSTR)) != -1)
                switch (opt) {
		case 'v':
			verbose++;
			break;
		case 'f':
			fg = 1;
			break;
		case 'd':
			strlcpy(domain, optarg, sizeof(domain));
			break;
		case 'p':
			strlcpy(pipefsdir, optarg, sizeof(pipefsdir));
			break;
		case 'U':
			nobodyuser = optarg;
			break;
		case 'G':
			nobodygroup = optarg;
			break;
		default:
			break;
		}

	if (domain[0] == '\0') {
		struct hostent *he;
		char hname[64], *c;

		if (gethostname(hname, sizeof(hname)) == -1)
			errx(1, "Error getting hostname");

		if ((he = gethostbyname(hname)) == NULL)
			errx(1, "Error resolving hostname: %s", hname);

		if ((c = strchr(he->h_name, '.')) == NULL || *++c == '\0')
			errx(1, "Error resolving domain, "
			    "please use the -d switch");

		strlcpy(domain, c, sizeof(domain));
	}

	if ((pw = getpwnam(nobodyuser)) == NULL)
		errx(1, "Could not find user \"%s\"", nobodyuser);
	nobodyuid = pw->pw_uid;

	if ((gr = getgrnam(nobodygroup)) == NULL)
		errx(1, "Could not find group \"%s\"", nobodygroup);
	nobodygid = gr->gr_gid;

	if (strlen(domain) == 0)
		errx(1, "Invalid domain; please specify with -d switch");

	if (verbose > 2)
		warnx("Using domain \"%s\"", domain);

	if (!fg)
		daemon(0, 0);

	event_init();

	nfsdret = nfsd_open(NFSD_DIR);

	if ((fd = open(pipefsdir, O_RDONLY)) != -1) {
		if (fcntl(fd, F_SETSIG, SIGUSR1) == -1)
			err(1, "fcntl(%s)", pipefsdir);
		if (fcntl(fd, F_NOTIFY,
			DN_CREATE | DN_DELETE | DN_MODIFY | DN_MULTISHOT) == -1)
			err(1, "fcntl(%s)", pipefsdir);

		TAILQ_INIT(&icq);

		/* These events are persistent */
		signal_set(&rootdirev, SIGUSR1, dirscan_cb, &icq);
		signal_add(&rootdirev, NULL);
		signal_set(&clntdirev, SIGUSR2, clntscan_cb, &icq);
		signal_add(&clntdirev, NULL);
		
		/* Fetch current state */
		dirscan_cb(0, 0, &icq);

		/*
		 * XXX there is a race condition here; perhaps add
		 * rootdirev with a timeout of 0 ...
		 */
	}

	if (nfsdret != 0 && fd == 0)                                            
		errx(1, "Neither NFS client nor NFSd found");                   

	event_dispatch();
	return (0);
}

static void
dirscan_cb(int fd, short which, void *data)
{
	int nent, i;
	struct dirent **ents;
	struct idmap_client *ic;
	char path[PATH_MAX];
	struct idmap_clientq *icq = data;

	nent = scandir(pipefsdir, &ents, NULL, alphasort);
	if (nent == -1) {
		warn("scandir(%s)", pipefsdir);
		return;
	}

	for (i = 0;  i < nent; i++) {
		if (ents[i]->d_reclen > 4 &&
		    strncmp(ents[i]->d_name, "clnt", 4) == 0) {
			TAILQ_FOREACH(ic, icq, ic_next)
				if (strcmp(ents[i]->d_name + 4, ic->ic_clid) == 0) 
					break;
			if (ic == NULL) {
				if ((ic = calloc(1, sizeof(*ic))) == NULL)
					return;
				strlcpy(ic->ic_clid, ents[i]->d_name + 4,
				    sizeof(ic->ic_clid));
				path[0] = '\0';
				snprintf(path, sizeof(path), "%s/%s",
				    pipefsdir, ents[i]->d_name);

				if ((ic->ic_dirfd = open(path, O_RDONLY, 0)) == -1) {
					warn("open(%s)", path);
					free(ic);
					return;
				}

				strlcat(path, "/idmap", sizeof(path));
				strlcpy(ic->ic_path, path, sizeof(ic->ic_path));

				if (verbose > 0)
					warnx("New client: %s", ic->ic_clid);

				if (openpipe(ic) == -1) {
					close(ic->ic_dirfd);
					free(ic);
					return;
				}

				ic->ic_id = "Client";

				TAILQ_INSERT_TAIL(icq, ic, ic_next);
			}
			ic->ic_scanned = 1;
		}
	}

	TAILQ_FOREACH(ic, icq, ic_next) {
		if (!ic->ic_scanned) {
			close(ic->ic_fd);
			close(ic->ic_dirfd);
			TAILQ_REMOVE(icq, ic, ic_next);
			if (verbose > 0) {
				warnx("Stale client: %s", ic->ic_clid);
				warnx("\t-> closed %s", ic->ic_path);
			}
			event_del(&ic->ic_event);
			free(ic);
		}

		ic->ic_scanned = 0;
	}
	return;
}

static void
clntscan_cb(int fd, short which, void *data)
{
	struct idmap_clientq *icq = data;
	struct idmap_client *ic;

	TAILQ_FOREACH(ic, icq, ic_next) 
		if (ic->ic_fd == -1)
			if (openpipe(ic) == -1) {
				close(ic->ic_dirfd);
				TAILQ_REMOVE(icq, ic, ic_next);
				free(ic);
			}
}

static void
idmap_cb(int fd, short which, void *data)
{
	struct idmap_client *ic = data;
	struct idmap_msg im;

	if (which != EV_READ)
		goto out;

	if (atomicio(read, ic->ic_fd, &im, sizeof(im)) != sizeof(im)) {
		if (verbose > 0)
			warn("read(%s)", ic->ic_path);
		goto out;
	}

	switch (im.im_conv) {
	case IDMAP_CONV_IDTONAME:
		reply_idtoname(&im);
		if (verbose > 1)
			warnx("%s %s: (%s) id \"%d\" -> name \"%s\"",
			    ic->ic_id, ic->ic_clid,
			    im.im_type == IDMAP_TYPE_USER ? "user" : "group",
			    im.im_id, im.im_name);
		break;
	case IDMAP_CONV_NAMETOID:
		if (validate_ascii(im.im_name, sizeof(im.im_name)) == -1) {
			im.im_status |= IDMAP_STATUS_INVALIDMSG;
			goto out_msg;
		}
		reply_nametoid(&im);
		if (verbose > 1)
			warnx("%s %s: (%s) name \"%s\" -> id \"%d\"",
			    ic->ic_id, ic->ic_clid,
			    im.im_type == IDMAP_TYPE_USER ? "user" : "group",
			    im.im_name, im.im_id);
		break;
	default:
		warnx("Invalid conversion type (%d) in message", im.im_conv);
		im.im_status |= IDMAP_STATUS_INVALIDMSG;
		break;
	}

 out_msg:
	/* XXX */
	if (atomicio(write, ic->ic_fd, &im, sizeof(im)) != sizeof(im))
		warn("write(%s)", ic->ic_path);
 out:
	event_add(&ic->ic_event, NULL);
}

static int
nfsd_open(char *path)
{
	struct idmap_client *ic;

	ic = &nfsd_ic[0];

	strlcpy(ic->ic_path, path, sizeof(ic->ic_path));
	strlcat(ic->ic_path, "/nfs4.nametoid/channel", sizeof(ic->ic_path));

	if ((ic->ic_fd = open(ic->ic_path, O_RDWR, 0)) == -1)
		return (-1);

	event_set(&ic->ic_event, ic->ic_fd, EV_READ, idmap_cb, ic);
	event_add(&ic->ic_event, NULL);

	if (verbose > 0)
		warnx("Opened %s", ic->ic_path);

	ic = &nfsd_ic[1];

	strlcpy(ic->ic_path, path, sizeof(ic->ic_path));
	strlcat(ic->ic_path, "/nfs4.idtoname/channel", sizeof(ic->ic_path));

	if ((ic->ic_fd = open(ic->ic_path, O_RDWR, 0)) == -1)
		return (-1);

	event_set(&ic->ic_event, ic->ic_fd, EV_READ, idmap_cb, ic);
	event_add(&ic->ic_event, NULL);

	ic->ic_id = "Server";
	strlcpy(ic->ic_clid, domain, sizeof(ic->ic_clid));

	if (verbose > 0)
		warnx("Opened %s", ic->ic_path);

	return (0);
}

static int
openpipe(struct idmap_client *ic)
{
	if ((ic->ic_fd = open(ic->ic_path, O_RDWR, 0)) == -1) {
		switch (errno) {
		case ENOENT:
			fcntl(ic->ic_dirfd, F_SETSIG, SIGUSR2);
			fcntl(ic->ic_dirfd, F_NOTIFY,
			    DN_CREATE | DN_DELETE | DN_MULTISHOT);
			break;
		default:
			warn("open(%s)", ic->ic_path);
			return (-1);
		}
	} else {
		event_set(&ic->ic_event, ic->ic_fd, EV_READ, idmap_cb, ic);
		event_add(&ic->ic_event, NULL);
		fcntl(ic->ic_dirfd, F_SETSIG, 0);
		fcntl(ic->ic_dirfd, F_NOTIFY, 0);
		if (verbose > 0)
			warnx("Opened %s", ic->ic_path);
	}

	return (0);
}

static void
reply_idtoname(struct idmap_msg *im)
{
        struct passwd *pw;
        struct group *gr;
        char *name = NULL;

        switch (im->im_type) {
        case IDMAP_TYPE_USER:
                if ((pw = getpwuid(im->im_id)) != NULL)
                        name = pw->pw_name;
		else
			name = nobodyuser;
                break;
        case IDMAP_TYPE_GROUP:
                if ((gr = getgrgid(im->im_id)) != NULL)
                        name = gr->gr_name;
		else
			name = nobodygroup;
                break;
        }

	im->im_status |= IDMAP_STATUS_SUCCESS;

	/* XXX check for truncation; i.e. strlen of name */
	strlcpy(im->im_name, name, sizeof(im->im_name));
	strlcat(im->im_name, "@", sizeof(im->im_name));
	strlcat(im->im_name, domain, sizeof(im->im_name));	
}

static void
reply_nametoid(struct idmap_msg *im)
{
        struct passwd *pw = NULL;
        struct group *gr = NULL;
        u_int32_t id = 0;
	char *c;

	/* Strip @DOMAIN info */
	if ((c = strchr(im->im_name, '@')) != NULL)
		*c = '\0';

        switch (im->im_type) {
        case IDMAP_TYPE_USER:
                if ((pw = getpwnam(im->im_name)) != NULL)
                        id = pw->pw_uid;
		else
			id = nobodyuid;
                break;
        case IDMAP_TYPE_GROUP:
                if ((gr = getgrnam(im->im_name)) != NULL)
                        id = gr->gr_gid;
		else
			id = nobodygid;
                break;
        }

	if (c != NULL)
		*c = '@';

	im->im_status |= IDMAP_STATUS_SUCCESS;

	im->im_id = id;
}

static int
validate_ascii(char *string, u_int32_t len)
{
	int i;

	for (i = 0; i < len; i++) {
		if (string[i] == '\0')
			break;

		if (string[i] & 0x80)
			return (-1);
	}

	if (string[i] != '\0')
		return (-1);

	return (i + 1);
}
