/*
 * Copyright (c) 1995, 1996, 1997, 1998 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* copyright (c) 2000 
   the regents of the university of michigan 
   all rights reserved 
   
   permission is granted to use, copy, create derivative works and  
   redistribute this software and such derivative works for any purpose,  
   so long as the name of the university of michigan is not used in any  
   advertising or publicity pertaining to the use or distribution of this  
   software without specific, written prior authorization.  if the above  
   copyright notice or any other identification of the university of  
   michigan is included in any copy of any portion of this software, then  
   the disclaimer below must also be included. 
   
   this software is provided as is, without representation from the  
   university of michigan as to its fitness for any purpose, and without  
   warranty by the university of michigan of any kind, either express or  
   implied, including without limitation the implied warranties of  
   merchantability and fitness for a particular purpose.  the regents of  
   the university of michigan shall not be liable for any damages,  
   including special, indirect, incidental, or consequential damages, with  
   respect to any claim arising out or in connection with the use of the  
   software, even if it has been or is hereafter advised of the  
   possibility of such damages. */


#include "xfs_locl.h"

RCSID("$Id: xfs_syscalls.c,v 1.15 1998/12/15 04:59:39 assar Exp $");

/*
 * XFS system calls.
 */

#include <xfs_syscalls.h>
#include <xfs_message.h>
#include <xfs_dev.h>
#include <xfs_node.h>
#include <xfs_deb.h>
#include <xfs_vfsops.h>

/* Misc syscalls */
#include <kafs.h>

/*
 * Def pag:
 *  33536 <= g0 <= 34560
 *  32512 <= g1 <= 48896
 */

#define XFS_PAG1_LLIM 33536
#define XFS_PAG1_ULIM 34560
#define XFS_PAG2_LLIM 32512
#define XFS_PAG2_ULIM 48896

static gid_t pag_part_one = XFS_PAG1_LLIM;
static gid_t pag_part_two = XFS_PAG2_LLIM;

int
xfs_is_pag(struct cred *cred)
{
    /* The first group is the gid of the user ? */

    if (cred->cr_ngroups >= 3 &&
	cred->cr_groups[1] >= XFS_PAG1_LLIM &&
	cred->cr_groups[1] <= XFS_PAG1_ULIM &&
	cred->cr_groups[2] >= XFS_PAG2_LLIM &&
	cred->cr_groups[2] <= XFS_PAG2_ULIM)
	return 1;
    else
	return 0;
}

pag_t
xfs_get_pag(struct cred * cred)
{
    if (xfs_is_pag(cred)) {

	return (((cred->cr_groups[1] << 16) & 0xFFFF0000) |
		((cred->cr_groups[2] & 0x0000FFFF)));

    } else
	return cred->cr_uid;	       /* XXX */
}

static int
xfs_setpag_call(struct proc *proc)
{
    struct cred *cred = proc->p_cred;
    int i;

    if (!xfs_is_pag(cred)) {

	/* Check if it fits */
	if (cred->cr_ngroups + 2 >= ngroups_max)
	    return set_errno(E2BIG); /* XXX Hmmm, better error ? */

	proc->p_cred = crcopy(cred);
	cred = proc->p_cred;

	/* Copy the groups */
	for (i = cred->cr_ngroups; i >= 0; i--) {
	    cred->cr_groups[i + 2] = cred->cr_groups[i];
	}
	cred->cr_ngroups += 2;

    } else
	cred = crcopy(cred);

    cred->cr_groups[1] = pag_part_one;
    cred->cr_groups[2] = pag_part_two++;

    if (pag_part_two > XFS_PAG2_ULIM) {
	pag_part_one++;
	pag_part_two = XFS_PAG2_LLIM;
    }
    return 0;
}

/*
 *
 */

static int
fhget_call (struct vnode *vp,
	    struct ViceIoctl *vice_ioctl)
{
    int error;
    struct xfs_fh_args fh_args;

    XFSDEB(XDEBSYS, ("xfs_fhget: vp = %x, vice_ioctl = %x\n",
		     (int)vp, (int)vice_ioctl));

    if (vp == NULL)
	return set_errno(EBADF);

    fh_args.fsid = vp->v_vfsp->vfs_fsid;

    fh_args.fid.fid_len = MAXFIDSZ;

    error = VOP_FID(vp, &fh_args.fid);
    VN_RELE(vp);
    if (error) {
	XFSDEB(XDEBSYS, ("fhget: vop_fid failed: %d\n", error));
	return set_errno(error);
    }

    if (vice_ioctl->out_size < sizeof(fh_args)) {
	XFSDEB(XDEBSYS, ("fhget: too small argument\n"));
	return set_errno(EINVAL);
    }

    error = copyout ((caddr_t)&fh_args,
		     (caddr_t)vice_ioctl->out,
		     sizeof(fh_args));

    return set_errno(error);
}

/*
 *
 */

static int
fhopen_call (struct vnode *vp,
	     struct ViceIoctl *vice_ioctl,
	     int flags)
{
    int error;
    struct xfs_fh_args fh_args;

    XFSDEB(XDEBSYS, ("xfs_fhopen: vp = %x\n", (int)vp));

    if (vp != NULL) {
	VN_RELE(vp);
	return set_errno(EINVAL);
    }

    if (vice_ioctl->in_size != sizeof(fh_args))
	return set_errno(EINVAL);

    error = copyin ((caddr_t)vice_ioctl->in,
		    (caddr_t)&fh_args, sizeof(fh_args));
    if (error)
	return set_errno(error);
    return xfs_fhopen (fh_args.fsid, fh_args.fid, flags);
}

/*
 * Send the pioctl to arlad
 */

static int
remote_pioctl (int a_opcode,
	       struct vnode *vp,
	       struct ViceIoctl *vice_ioctl)
{
    struct xfs_message_pioctl msg;
    struct xfs_message_wakeup_data *msg2;
    int error;

    if (vice_ioctl->in_size > 2048) {
	printf("xfs_pioctl_call: got a humongous in packet: opcode: %d",
	       a_opcode);
	return set_errno(EINVAL);
    }
    if (vice_ioctl->in_size != 0) {
	error = copyin((caddr_t)vice_ioctl->in,
		       (caddr_t)&msg.msg,
		       vice_ioctl->in_size);

	if (error)
	    return set_errno(error);
    }

    if (vp != NULL) {
	struct xfs_node *xn;

	xn = VNODE_TO_XNODE(vp);
	msg.handle = xn->handle;
	VN_RELE(vp);
    }

    msg.header.opcode = XFS_MSG_PIOCTL;
    msg.opcode = a_opcode;

    msg.insize = vice_ioctl->in_size;
    msg.cred.uid = curproc->p_cred->cr_uid;
    msg.cred.pag = xfs_get_pag(curproc->p_cred);

    error = xfs_message_rpc(0, &msg.header, sizeof(msg)); /* XXX */
    msg2 = (struct xfs_message_wakeup_data *) & msg;

    if (error == 0)
	error = msg2->error;
    else
	error = EINVAL; /* return EINVAL to not confuse applications */
    
    if (error == 0 && vice_ioctl->out_size)
	error = copyout((caddr_t)msg2->msg,
			(caddr_t)vice_ioctl->out, 
			min(msg2->len, vice_ioctl->out_size));
    return set_errno(error);
}

/*
 * Read/set the xfs debug level according to `vice_ioctl'
 */

static int
xfs_debug(struct ViceIoctl *vice_ioctl)
{
    int32_t flags;
    int error;

    if (!suser(CRED()))
	return EPERM;

    if (vice_ioctl->in_size != 0) {
	if (vice_ioctl->in_size < sizeof(int32_t))
	    return EINVAL;
	
	error = copyin (vice_ioctl->in,
			&flags,
			sizeof(flags));
	if (error)
	    return error;
	
	xfsdeb = flags;
    }
    
    if (vice_ioctl->out_size != 0) {
	if (vice_ioctl->out_size < sizeof(int32_t))
	    return EINVAL;
	
	error = copyout (&xfsdeb,
			 vice_ioctl->out,
			 sizeof(int32_t));
	if (error)
	    return error;
    }
    return 0;
}

/*
 * Handle `pioctl'
 */

static int
xfs_pioctl_call(char *a_pathP,
		int a_opcode,
		struct ViceIoctl *a_paramsP,
		int a_followSymlinks)
{
    int error;
    struct ViceIoctl vice_ioctl;
    struct vnode *vp = NULL;
	
    XFSDEB(XDEBSYS, ("xfs_pioctl\n"));

    /* Copy in the data structure for us */

    error = copyin((caddr_t)a_paramsP,
		   (caddr_t)&vice_ioctl,
		   sizeof(vice_ioctl));

    if (error) 
	return set_errno(error);

    if (a_pathP != NULL) {
	char path[MAXPATHLEN];

	error = copyinstr (a_pathP, path, sizeof(path), NULL);
	if (error)
	    return set_errno(error);

	XFSDEB(XDEBSYS, ("xfs_syscall: looking up: %s\n", path));

	error = lookupname(path,
			   UIO_SYSSPACE,
			   a_followSymlinks ? FOLLOW : NO_FOLLOW,
			   NULL,
			   &vp);
	if (error)
	    return set_errno(EINVAL);
	XFSDEB(XDEBSYS, ("xfs_syscall: lookup -> %d, vp = %x\n",
			 error, (int)vp));
    }

    switch (a_opcode) {
    case VIOC_FHGET :
	XFSDEB(XDEBSYS, ("calling fhget(%x, %x)\n",
			 (int)vp, (int)&vice_ioctl));
	return fhget_call (vp, &vice_ioctl);
    case VIOC_FHOPEN :
	return fhopen_call (vp, &vice_ioctl, a_followSymlinks);
    case VIOC_XFSDEBUG :
	return xfs_debug (&vice_ioctl);
    default :
	return remote_pioctl (a_opcode, vp, &vice_ioctl);
    }
}

static int
xfs_syscall(int operation,
	    char *a_pathP,
	    int a_opcode,
	    struct ViceIoctl *a_paramsP,
	    int a_followSymlinks)
{
    int ret;

    XFSDEB(XDEBSYS, ("xfs_syscall\n"));

    switch (operation) {
    case AFSCALL_PIOCTL:
	ret = xfs_pioctl_call(a_pathP, a_opcode, a_paramsP,
			      a_followSymlinks);
	break;
    case AFSCALL_SETPAG:
	ret = xfs_setpag_call(curproc);
	break;
    default:
	uprintf("Unimplemeted call: %d\n", operation);
	ret = set_errno(EINVAL);
    }
    return ret;
}

static struct sysent xfs_sysent = {
    5,
    SE_ARGC | SE_LOADABLE,
    xfs_syscall
};

struct modlsys xfs_modlsys = {
    &mod_syscallops,
    "xfs syscall",
    &xfs_sysent
};
