/*
 * Copyright (c) 1995, 1996, 1997, 1998 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* copyright (c) 2000 
   the regents of the university of michigan 
   all rights reserved 
   
   permission is granted to use, copy, create derivative works and  
   redistribute this software and such derivative works for any purpose,  
   so long as the name of the university of michigan is not used in any  
   advertising or publicity pertaining to the use or distribution of this  
   software without specific, written prior authorization.  if the above  
   copyright notice or any other identification of the university of  
   michigan is included in any copy of any portion of this software, then  
   the disclaimer below must also be included. 
   
   this software is provided as is, without representation from the  
   university of michigan as to its fitness for any purpose, and without  
   warranty by the university of michigan of any kind, either express or  
   implied, including without limitation the implied warranties of  
   merchantability and fitness for a particular purpose.  the regents of  
   the university of michigan shall not be liable for any damages,  
   including special, indirect, incidental, or consequential damages, with  
   respect to any claim arising out or in connection with the use of the  
   software, even if it has been or is hereafter advised of the  
   possibility of such damages. */


#include "xfs_locl.h"
#include <xfs_message.h>
#include <xfs_msg_locl.h>
#include <xfs_dev.h>
#include <xfs_fs.h>
#include <xfs_deb.h>
#include <nxfs.h>

RCSID("$Id: xfs_dev.c,v 1.12 1998/12/15 05:51:40 assar Exp $");

/*
 * Queues of xfs_links hold outbound messages and processes sleeping
 * for replies. The last field is used to return error to sleepers and
 * to keep record of memory to be deallocated when messages have been
 * delivered or dropped.
 */
struct xfs_link {
  struct xfs_link *prev, *next;
  struct xfs_message_header *message;
  kmutex_t mutex;
  kcondvar_t cv;
  u_int error_or_size;		/* error on sleepq and size on messageq */
};  

struct xfs_channel {
  dev_info_t *dip;
  struct xfs_link messageq;	/* Messages not yet read */
  struct xfs_link sleepq;	/* Waiting for reply message */
  u_int nsequence;
  struct pollhead pollhead;
  struct proc *selecting_proc;
  struct xfs_message_header *message_buffer;
  int status;
#define CHANNEL_OPENED	0x1
};

static void *xfs_dev_state;

static void
xfs_initq(struct xfs_link *q)
{
  q->next = q;
  q->prev = q;
}

/* Is this queue empty? */
#define xfs_emptyq(q) ((q)->next == (q))

/* Is this link on any queue? Link *must* be inited! */
#define xfs_onq(link) ((link)->next != 0 || (link)->prev != 0)

/* Append q with p */
static void
xfs_appendq(struct xfs_link *q, struct xfs_link *p)     
{
  p->next = q;
  p->prev = q->prev;
  p->prev->next = p;
  q->prev = p;
}

static void
xfs_outq(struct xfs_link *p)     
{
  p->next->prev = p->prev;
  p->prev->next = p->next;
  p->next = p->prev = 0;
}

/*
 * Only allow one open.
 */
int
xfs_devopen(dev_t *devp, int flags, int otyp, cred_t *credp)
{
  struct xfs_channel *chan;

  XFSDEB(XDEBDEV, ("xfs_devopen dev = %ld, flags = %d, otyp = %d\n",
		   *devp, flags, otyp));

  if (otyp != OTYP_CHR)
      return EINVAL;

  chan = (struct xfs_channel *)ddi_get_soft_state(xfs_dev_state, getminor(*devp));
  if (chan == NULL)
      return ENXIO;

  /* Only allow one reader/writer */
  if (chan->status & CHANNEL_OPENED)
    return EBUSY;
  else
    chan->status |= CHANNEL_OPENED;

  chan->message_buffer = xfs_alloc(MAX_XMSG_SIZE);

  ASSERT(chan->message_buffer != NULL);

  return 0;
}

/*
 * Wakeup all sleepers and cleanup.
 */
int
xfs_devclose(dev_t dev, int flags, int otyp, cred_t *credp)
{
  struct xfs_channel *chan;
  struct xfs_link *first;

  XFSDEB(XDEBDEV, ("xfs_devclose dev = %ld, flags = %d, otyp = %d\n",
		   dev, flags, otyp));

  chan = (struct xfs_channel *)ddi_get_soft_state(xfs_dev_state, getminor(dev));
  if (chan == NULL)
      return ENXIO;

  /* Sanity check, paranoia? */
  if (!(chan->status & CHANNEL_OPENED))
    panic("xfs_devclose never opened?");

  chan->status &= ~CHANNEL_OPENED;

  /* No one is going to read those messages so empty queue! */
  while (!xfs_emptyq(&chan->messageq)) {
      XFSDEB(XDEBDEV, ("before outq(messageq)\n"));
      first = chan->messageq.next;
      xfs_outq(first);
      if (first->error_or_size != 0)
	xfs_free(first, first->error_or_size);
      XFSDEB(XDEBDEV, ("after outq(messageq)\n"));
  }

  /* Wakeup those waiting for replies that will never arrive. */
  while (!xfs_emptyq(&chan->sleepq)) {
      XFSDEB(XDEBDEV, ("before outq(sleepq)\n"));
      first = chan->sleepq.next;
      mutex_enter(&first->mutex);
      xfs_outq(first);
      first->error_or_size = ENODEV;
      cv_signal(&first->cv);
      mutex_exit(&first->mutex);
      XFSDEB(XDEBDEV, ("after outq(sleepq)\n"));
  }
  
  if (chan->message_buffer) {
      xfs_free(chan->message_buffer, MAX_XMSG_SIZE);
      chan->message_buffer = 0;
  }
      
  /* Free all xfs_nodes. */
  free_all_xfs_nodes(&xfs[getminor(dev)]);
  return 0;
}

/*
 * Move messages from kernel to user space.
 */
int
xfs_devread(dev_t dev, struct uio *uiop, cred_t *credp)
{
  struct xfs_channel *chan;
  struct xfs_link *first;
  int error = 0;

  XFSDEB(XDEBDEV, ("xfs_devread dev = %ld\n", dev));

  chan = (struct xfs_channel *)ddi_get_soft_state(xfs_dev_state,
						  getminor(dev));
  if (chan == NULL)
      return ENXIO;

  while (!xfs_emptyq (&chan->messageq)) {
      /* Remove message */
      first = chan->messageq.next;

      ASSERT(first->message->size != 0);

      error = uiomove((caddr_t) first->message, first->message->size,
		      UIO_READ, uiop);
      if (error)
	  break;
      xfs_outq(first);

      if (first->error_or_size != 0)
	  xfs_free(first, first->error_or_size);
  }
  return error;
}

/*
 * Move messages from user space to kernel space,
 * wakeup sleepers, insert new data in VFS.
 */
int
xfs_devwrite(dev_t dev, struct uio *uiop, cred_t *credp)
{
  struct xfs_channel *chan;
  char *p;
  int error;
  u_int cnt;
  struct xfs_message_header *msg_buf;

  XFSDEB(XDEBDEV, ("xfs_devwrite dev = %ld\n", dev));

  chan = (struct xfs_channel *)ddi_get_soft_state(xfs_dev_state,
						  getminor(dev));
  if (chan == NULL)
      return ENXIO;

  cnt = uiop->uio_resid;
  error = uiomove((caddr_t) chan->message_buffer, MAX_XMSG_SIZE,
		  UIO_WRITE, uiop);
  if (error != 0)
    return error;
  
  cnt -= uiop->uio_resid;

  /*
   * This thread handles the received message.
   */

  ASSERT(chan->message_buffer != NULL);

  for (p = (char *)chan->message_buffer;
       cnt > 0;
       p += msg_buf->size, cnt -= msg_buf->size) {
      msg_buf = (struct xfs_message_header *)p;

      ASSERT(msg_buf);
      ASSERT(msg_buf->size);

      error = xfs_message_receive (getminor(dev),
				   msg_buf,
				   msg_buf->size);
  }
  XFSDEB(XDEBDEV, ("xfs_devwrite error = %d\n", error));
  return error;
}

/*
 * Not used.
 */
int
xfs_devioctl(dev_t dev, int cmd, int arg, int mode, cred_t *credp,
	     int *rvalp)
{
  XFSDEB(XDEBDEV, ("xfs_devioctl dev = %ld, cmd = %d\n", dev, cmd));
  return EINVAL;
}

/*
 * Are there any messages on this filesystem?
 */
int
xfs_chpoll(dev_t dev, short events, int anyyet,
	   short *reventsp, struct pollhead **phpp)
{
  struct xfs_channel *chan;

  XFSDEB(XDEBDEV, ("xfs_devselect dev = %ld, events = %d, anyyet = %d\n",
		   dev, events, anyyet));

  chan = (struct xfs_channel *)ddi_get_soft_state(xfs_dev_state, getminor(dev));
  if (chan == NULL)
      return ENXIO;

  if (!(events & POLLRDNORM))
      return 0;

  if (!xfs_emptyq(&chan->messageq)) {
      *reventsp |= POLLRDNORM;
  } else {
      *reventsp = 0;
      if (!anyyet)
	  *phpp = &chan->pollhead;
  }
  return 0;
}

/*
 * Send a message to user space.
 */
int
xfs_message_send(int fd, struct xfs_message_header *message, u_int size)
{
  struct xfs_channel *chan;
  struct {
    struct xfs_link this_message;
    struct xfs_message_header msg;
  } *t;

  ASSERT(message != NULL);

  chan = (struct xfs_channel *)ddi_get_soft_state(xfs_dev_state, fd);
  if (chan == NULL)
      return ENXIO;

  XFSDEB(XDEBMSG, ("xfs_message_send opcode = %d\n", message->opcode));

  if (!(chan->status & CHANNEL_OPENED))	/* No receiver? */
    return ENODEV;
  
  /* Prepare message and copy it later */
  ASSERT(size != 0);

  message->size = size;
  message->sequence_num = chan->nsequence++;

  t = xfs_alloc(sizeof(t->this_message) + size);
  ASSERT(t != NULL);

  t->this_message.error_or_size = sizeof(t->this_message) + size;
  bcopy((caddr_t)message, (caddr_t)&t->msg, size);

  t->this_message.message = &t->msg;
  xfs_appendq(&chan->messageq, &t->this_message);
  pollwakeup(&chan->pollhead, POLLRDNORM);
  return 0;
}

/*
 * Send a message to user space and wait for reply.
 */
int
xfs_message_rpc(int fd, struct xfs_message_header *message, u_int size)
{
  struct xfs_channel *chan;
  struct xfs_link *this_message;
  struct xfs_link *this_process;
  struct xfs_message_header *msg;
  int ret;

  ASSERT(message != NULL);

  chan = (struct xfs_channel *)ddi_get_soft_state(xfs_dev_state, fd);
  if (chan == NULL)
      return ENXIO;

  XFSDEB(XDEBMSG, ("xfs_message_rpc opcode = %d\n", message->opcode));

  if (!(chan->status & CHANNEL_OPENED))	/* No receiver? */
    return ENODEV;
  
  if (size < sizeof(struct xfs_message_wakeup)) {
      printf("XFS PANIC Error: Message to small to receive wakeup, opcode = %d\n", message->opcode);
      return ENOMEM;
  }
  this_message = xfs_alloc(sizeof(struct xfs_link));
  this_process = xfs_alloc(sizeof(struct xfs_link));
  bzero (this_message, sizeof(struct xfs_link));
  bzero (this_process, sizeof(struct xfs_link));
  msg = xfs_alloc(size);
  bcopy((caddr_t)message, (caddr_t)msg, size);

  mutex_init(&this_process->mutex, "this_process", MUTEX_DRIVER, NULL);
  cv_init(&this_process->cv, "this_process", CV_DRIVER, NULL);

  ASSERT(size != 0);

  msg->size = size;
  msg->sequence_num = chan->nsequence++;
  this_message->error_or_size = 0;
  this_message->message = msg;
  this_process->message = msg;
  xfs_appendq(&chan->messageq, this_message);
  xfs_appendq(&chan->sleepq, this_process);
  pollwakeup(&chan->pollhead, POLLRDNORM);
  this_process->error_or_size = 0;
  XFSDEB(XDEBMSG, ("messageq = %x, next = %x"
		   "first: %d:%u\n",
		   (int)&chan->messageq, (int)&chan->messageq.next,
		   chan->messageq.next->message->opcode,
		   chan->messageq.next->message->size));
  XFSDEB(XDEBMSG, ("xfs_message_rpc before sleep\n"));
  mutex_enter(&this_process->mutex);
  if(cv_wait_sig(&this_process->cv, &this_process->mutex) == 0) {
      XFSDEB(XDEBMSG, ("caught signal\n"));
      this_process->error_or_size = EINTR;
  }
  XFSDEB(XDEBMSG, ("xfs_message_rpc after sleep\n"));
  /*
   * Caught signal, got reply message or device was closed.
   * Need to clean up both messageq and sleepq.
   */
  if (xfs_onq(this_message)) {
      xfs_outq(this_message);
  }
  if (xfs_onq(this_process)) {
      xfs_outq(this_process);
  }
  ret = this_process->error_or_size;

  cv_destroy (&this_process->cv);
  mutex_exit(&this_process->mutex);
  mutex_destroy(&this_process->mutex);

  bcopy((caddr_t)msg, (caddr_t)message, size);

  xfs_free(this_message, sizeof(*this_message));
  xfs_free(this_process, sizeof(*this_process));
  xfs_free(msg, size);

  return ret;
}

/*
 * For each message type there is a message handler
 * that implements its action, xfs_message_receive
 * invokes the correct function.
 */
int
xfs_message_receive(int fd, struct xfs_message_header *message, u_int size)
{
  ASSERT(message != NULL);

  XFSDEB(XDEBMSG, ("xfs_message_receive opcode = %d\n", message->opcode));

  /* Dispatch and coerce message type */
  switch (message->opcode) {
  case XFS_MSG_WAKEUP:
    return xfs_message_wakeup(fd, (struct xfs_message_wakeup *) message, message->size);
  case XFS_MSG_WAKEUP_DATA:
    return xfs_message_wakeup_data(fd,
				   (struct xfs_message_wakeup_data *) message,
				   message->size);
  case XFS_MSG_INSTALLROOT:
    return xfs_message_installroot(fd, (struct xfs_message_installroot *) message, message->size);
  case XFS_MSG_INSTALLNODE:
    return xfs_message_installnode(fd, (struct xfs_message_installnode *) message, message->size);
  case XFS_MSG_INSTALLATTR:
    return xfs_message_installattr(fd, (struct xfs_message_installattr *) message, message->size);
  case XFS_MSG_INSTALLDATA:
    return xfs_message_installdata(fd, (struct xfs_message_installdata *) message, message->size);
  case XFS_MSG_INVALIDNODE:
    return xfs_message_invalidnode(fd, (struct xfs_message_invalidnode *) message, message->size);
  default:
    printf("XFS PANIC Warning xfs_dev: Unknown message opcode == %d\n", message->opcode);
    return EINVAL;
  }
}

int
xfs_message_wakeup(int fd, struct xfs_message_wakeup *message, u_int size)
{
  struct xfs_channel *chan;
  struct xfs_link *sleepq;
  struct xfs_link *t;

  ASSERT(message != NULL);

  chan = (struct xfs_channel *)ddi_get_soft_state(xfs_dev_state, fd);
  if (chan == NULL)
      return ENXIO;

  sleepq = &chan->sleepq;
  t = chan->sleepq.next; /* Really first in q */
  XFSDEB(XDEBMSG, ("xfs_message_wakeup\n"));

  for (; t != sleepq; t = t->next) {
    ASSERT(t->message != NULL);
    if (t->message->sequence_num == message->sleepers_sequence_num)
      {
	if (t->message->size < size)
	  {
	    printf("XFS PANIC Error:"
		   " Could not wakeup requestor"
		   " with opcode = %d properly,"
		   " to small receive buffer.\n",
		   t->message->opcode);
	    t->error_or_size = ENOMEM;
	  }
	else
	  bcopy((caddr_t)message, (caddr_t)t->message, size);
	mutex_enter(&t->mutex);
	cv_signal (&t->cv);
	mutex_exit (&t->mutex);
	break;
      }
  }

  return 0;
}

int
xfs_message_wakeup_data(int fd,
			struct xfs_message_wakeup_data * message,
			u_int size)
{
    struct xfs_channel *chan;
    struct xfs_link *sleepq;
    struct xfs_link *t;

    ASSERT(message != NULL);

    chan = (struct xfs_channel *)ddi_get_soft_state(xfs_dev_state, fd);
    if (chan == NULL)
	return ENXIO;

    sleepq = &chan->sleepq;
    t = chan->sleepq.next; /* Really first in q */
    XFSDEB(XDEBMSG, ("xfs_message_wakeup_data\n"));

    for (; t != sleepq; t = t->next) {
	ASSERT(t->message != NULL);
	if (t->message->sequence_num == message->sleepers_sequence_num) {
	    if (t->message->size < size) {
		printf("XFS PANIC Error: Could not wakeup requestor with " 
		       "opcode = %d properly, to small receive buffer.\n", 
		       t->message->opcode);
		t->error_or_size = ENOMEM;
	    } else
		bcopy((caddr_t)message, (caddr_t)t->message, size);
	    mutex_enter(&t->mutex);
	    cv_signal (&t->cv);
	    mutex_exit (&t->mutex);
	    break;
	}
    }

    return 0;
}

int
xfs_dev_init(void)
{
    int ret;

    ret = ddi_soft_state_init(&xfs_dev_state,
			      sizeof(struct xfs_channel), NXFS);

    return ret;
}

int
xfs_dev_fini(void)
{
    ddi_soft_state_fini(&xfs_dev_state);
    return 0;
}

static int
xfs_dev_getinfo(dev_info_t *dip,
		ddi_info_cmd_t infocmd,
		void *arg,
		void **result)
{
    int ret;

    printf ("xfs_dev_getinfo\n");

    switch(infocmd) {
    case DDI_INFO_DEVT2INSTANCE : {
	dev_t dev = (dev_t)arg;
	*result = (void *)getminor(dev);
	ret = DDI_SUCCESS;
	break;
    }
    case DDI_INFO_DEVT2DEVINFO : {
	dev_t dev = (dev_t)arg;
	struct xfs_channel *chan;

	chan = (struct xfs_channel *)ddi_get_soft_state(xfs_dev_state,
							getminor(dev));

	if (chan == NULL) {
	    *result = NULL;
	    ret = DDI_FAILURE;
	} else {
	    *result = chan->dip;
	    ret = DDI_SUCCESS;
	}
	break;
    }
    default :
	ret = DDI_FAILURE;
	break;
    }
    return ret;
}

static int
xfs_dev_probe (dev_info_t *dip)
{
    printf ("xfs_dev_probe\n");

    return DDI_PROBE_SUCCESS;
}

static int
xfs_dev_attach(dev_info_t *dip,
	       ddi_attach_cmd_t cmd)
{
    int ret;

    printf ("xfs_dev_attach\n");

    switch(cmd) {
    case DDI_ATTACH : {
	int instance = ddi_get_instance(dip);
	struct xfs_channel *state;

	ret = ddi_soft_state_zalloc(xfs_dev_state, instance);
	if (ret != DDI_SUCCESS)
	    break;
	state = (struct xfs_channel *)ddi_get_soft_state(xfs_dev_state,
							 instance);

	ret = ddi_create_minor_node(dip, "", S_IFCHR, instance, NULL, 0);
	if (ret != DDI_SUCCESS) {
	    ddi_soft_state_free(xfs_dev_state, instance);
	    break;
	}
	
	state->dip = dip;
	xfs_initq(&state->messageq);
	xfs_initq(&state->sleepq);
	state->nsequence = 0;
	state->selecting_proc = NULL;
	state->message_buffer = 0;
	state->status = 0;
	/* how is the pollhead supposed to be initialized? */
	bzero((caddr_t)&state->pollhead, sizeof(state->pollhead));

	ddi_report_dev(dip);
	ret = DDI_SUCCESS;
	break;
    }	
#ifdef DDI_PM_RESUME
    case DDI_PM_RESUME :
#endif
    case DDI_RESUME :
	ret = DDI_SUCCESS;
	break;
    default :
	ret = DDI_FAILURE;
	break;
    }
    return ret;
}

static int
xfs_dev_detach(dev_info_t *dip, ddi_detach_cmd_t cmd)
{
    int ret;

    printf ("xfs_dev_detach\n");

    switch (cmd) {
    case DDI_DETACH : {
	int instance = ddi_get_instance(dip);
	struct xfs_channel *state;

	state = (struct xfs_channel *)ddi_get_soft_state(xfs_dev_state,
							 instance);
	ddi_remove_minor_node(dip, NULL);
	ddi_soft_state_free(xfs_dev_state, instance);
	ret = DDI_SUCCESS;
	break;
    }
    case DDI_PM_SUSPEND :
    case DDI_SUSPEND :
	ret = DDI_SUCCESS;
	break;
    default :
	ret = DDI_FAILURE;
	break;
    }
    return ret;
}

static struct cb_ops xfs_cb_ops = {
    xfs_devopen,		/* open */
    xfs_devclose,		/* close */
    nodev,			/* strategy */
    nodev,			/* print */
    nodev,			/* dump */
    xfs_devread,		/* read */
    xfs_devwrite,		/* write */
    xfs_devioctl,		/* ioctl */
    nodev,			/* devmap */
    nodev,			/* mmap */
    nodev,			/* segmap */
    xfs_chpoll,			/* chpoll */
    nodev,			/* prop_op */
    NULL,			/* cb_str */
    D_NEW | D_MP,		/* flag */
    0,				/* rev */
    nodev,			/* aread */
    nodev			/* awrite */
};

static struct dev_ops xfs_dev_ops = {
    DEVO_REV,			/* rev */
    0,				/* refcnt */
    xfs_dev_getinfo,		/* getinfo */
    nulldev,			/* identify */
    xfs_dev_probe,		/* probe */
    xfs_dev_attach,		/* attach */
    xfs_dev_detach,		/* detach */
    nodev,			/* reset */
    &xfs_cb_ops,		/* cb_ops */
    NULL,			/* bus_ops */
    NULL			/* power */
};

struct modldrv xfs_modldrv = {
    &mod_driverops,
    "xfs cdev driver",
    &xfs_dev_ops
};
