/*
 * Copyright (c) 1995, 1996, 1997, 1998 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* copyright (c) 2000 
   the regents of the university of michigan 
   all rights reserved 
   
   permission is granted to use, copy, create derivative works and  
   redistribute this software and such derivative works for any purpose,  
   so long as the name of the university of michigan is not used in any  
   advertising or publicity pertaining to the use or distribution of this  
   software without specific, written prior authorization.  if the above  
   copyright notice or any other identification of the university of  
   michigan is included in any copy of any portion of this software, then  
   the disclaimer below must also be included. 
   
   this software is provided as is, without representation from the  
   university of michigan as to its fitness for any purpose, and without  
   warranty by the university of michigan of any kind, either express or  
   implied, including without limitation the implied warranties of  
   merchantability and fitness for a particular purpose.  the regents of  
   the university of michigan shall not be liable for any damages,  
   including special, indirect, incidental, or consequential damages, with  
   respect to any claim arising out or in connection with the use of the  
   software, even if it has been or is hereafter advised of the  
   possibility of such damages. */


/*
 * XFS operations.
 */

#include "xfs_locl.h"
#include <xfs_message.h>
#include <xfs_dev.h>
#include <xfs_common.h>
#include <xfs_fs.h>
#include <xfs_deb.h>
#include <xfs_syscalls.h>

RCSID("$Id: xfs_vnodeops.c,v 1.4 1998/10/18 09:00:56 assar Exp $");

static int
xfs_open_valid(struct vnode *vp, struct cred *cred, u_int tok)
{
  struct xfs *xfsp = XFS_FROM_VNODE(vp);
  struct xfs_node *xn = VNODE_TO_XNODE(vp);
  int error = 0;
  
  XFSDEB(XDEBVFOPS, ("xfs_open_valid\n"));

  do {
    if (!XFS_TOKEN_GOT(xn, tok))
      {
	struct xfs_message_open msg;

	msg.header.opcode = XFS_MSG_OPEN;
	msg.cred.uid = cred->cr_uid;
	msg.cred.pag = xfs_get_pag(cred);
	msg.handle = xn->handle;
	msg.tokens = tok;
	error = xfs_message_rpc(xfsp->fd, &msg.header, sizeof(msg));
	if (error == 0)
	  error = ((struct xfs_message_wakeup *) &msg)->error;
      }
    else
      {
	goto done;
      }
  } while (error == 0);

 done:
  return error;
}

static int
xfs_attr_valid(struct vnode *vp, struct cred *cred, u_int tok)
{
  struct xfs *xfsp = XFS_FROM_VNODE(vp);
  struct xfs_node *xn = VNODE_TO_XNODE(vp);
  int error = 0;
  
  pag_t pag = xfs_get_pag(cred);

  do {
    if (!XFS_TOKEN_GOT(xn, tok))
      {
	struct xfs_message_getattr msg;
	msg.header.opcode = XFS_MSG_GETATTR;
	msg.cred.uid = cred->cr_uid;
	msg.cred.pag = pag;
	msg.handle = xn->handle;
	error = xfs_message_rpc(xfsp->fd, &msg.header, sizeof(msg));
	if (error == 0)
	  error = ((struct xfs_message_wakeup *) &msg)->error;
      }
    else
      {
	goto done;
      }
  } while (error == 0);

 done:
  return error;
}

static int
xfs_fetch_rights(struct vnode *vp, struct cred *cred)
{
    struct xfs *xfsp = XFS_FROM_VNODE(vp);
    struct xfs_node *xn = VNODE_TO_XNODE(vp);
    int error = 0;

    pag_t pag = xfs_get_pag(cred);

    struct xfs_message_getattr msg;

    msg.header.opcode = XFS_MSG_GETATTR;
    msg.cred.uid = cred->cr_uid;
    msg.cred.pag = pag;
    msg.handle = xn->handle;
    error = xfs_message_rpc(xfsp->fd, &msg.header, sizeof(msg));
    if (error == 0)
	error = ((struct xfs_message_wakeup *) & msg)->error;

    return (error);
}

static int
xfs_data_valid(struct vnode *vp, struct cred *cred, u_int tok)
{
  struct xfs *xfsp = XFS_FROM_VNODE(vp);
  struct xfs_node *xn = VNODE_TO_XNODE(vp);
  int error = 0;
  
  do {
    if (!XFS_TOKEN_GOT(xn, tok))
      {
	struct xfs_message_getdata msg;
	msg.header.opcode = XFS_MSG_GETDATA;
	msg.cred.uid = cred->cr_uid;
	msg.cred.pag = xfs_get_pag(cred);
	msg.handle = xn->handle;
	msg.tokens = tok;
	error = xfs_message_rpc(xfsp->fd, &msg.header, sizeof(msg));
	if (error == 0)
	  error = ((struct xfs_message_wakeup *) &msg)->error;
      }
    else
      {
	goto done;
      }
  } while (error == 0);

 done:
  return error;
}

static int
do_fsync(struct xfs *xfsp,
	 struct xfs_node *xn,
	 struct cred *cred,
	 u_int flag)
{
    int error;
    struct xfs_message_putdata msg;

    msg.header.opcode = XFS_MSG_PUTDATA;
    msg.cred.uid = cred->cr_uid;
    msg.cred.pag = xfs_get_pag(cred);
    msg.handle = xn->handle;

    msg.flag = flag;
    error = xfs_message_rpc(xfsp->fd, &msg.header, sizeof(msg));
    if (error == 0)
	error = ((struct xfs_message_wakeup *) & msg)->error;

    if (error == 0)
	xn->flags &= ~XFS_DATA_DIRTY;

    return error;
}

static int
xfs_open(struct vnode **vpp,
	 mode_t mode,
	 struct cred *cred)
{
  int error = 0;
  
  XFSDEB(XDEBVNOPS, ("xfs_open\n"));
  
  if (mode & VWRITE)
    error = xfs_open_valid(*vpp, cred, XFS_OPEN_NW);
  else
    error = xfs_open_valid(*vpp, cred, XFS_OPEN_NR);
  
  return error;
}

static int
xfs_close(struct vnode *vp,
	  int flag,
	  lastclose_t lastclose,
	  off_t offset,
	  struct cred *cred)
{
  struct xfs *xfsp = XFS_FROM_VNODE(vp);
  struct xfs_node *xn = VNODE_TO_XNODE(vp);
  int error = 0;

  XFSDEB(XDEBVNOPS, ("xfs_close\n"));
  
  if (flag & FWRITE && xn->flags & XFS_DATA_DIRTY)
    error = do_fsync (xfsp, xn, cred, XFS_WRITE);
  
  return error;
}

static int
xfs_read(struct vnode *vp,
	 struct uio *uio,
	 int ioflag,
	 struct cred *cred)
{
    int error = 0;

    XFSDEB(XDEBVNOPS, ("xfs_read\n"));

    error = xfs_data_valid(vp, cred, XFS_DATA_R);

    if (error == 0) {
	struct vnode *t = DATA_FROM_VNODE(vp);
	ASSERT(t != NULL);
	VOP_RWLOCK(t, 0);
	error = VOP_READ(t, uio, ioflag, cred);
	VOP_RWUNLOCK(t, 0);
    }

    return error;
}

static int
xfs_write(struct vnode *vp,
	  struct uio *uio,
	  int ioflag,
	  struct cred *cred)
{
    int error = 0;

    XFSDEB(XDEBVNOPS, ("xfs_write\n"));

    error = xfs_data_valid(vp, cred, XFS_DATA_W);

    if (error == 0) {
	struct xfs_node *xn = VNODE_TO_XNODE(vp);
	struct vnode *t = DATA_FROM_VNODE(vp);
	struct vattr sub_attr;
	int error2 = 0;

	ASSERT(t != NULL);
	VOP_RWLOCK(t, 1);
	error  = VOP_WRITE(t, uio, ioflag, cred);
	error2 = VOP_GETATTR(t, &sub_attr, 0, cred);
	VOP_RWUNLOCK(t, 1);
	VNODE_TO_XNODE(vp)->flags |= XFS_DATA_DIRTY;

	if (error2 == 0) {
	    xn->attr.va_size  = sub_attr.va_size;
	    xn->attr.va_mtime = sub_attr.va_mtime;
	}
    }

    return error;
}

static int
xfs_ioctl(struct vnode *vp,
	  int cmd,
	  void *arg,
	  int flag,
	  struct cred *cred,
	  int *result)
{
  XFSDEB(XDEBVNOPS, ("xfs_ioctl\n"));
  return ENOSYS;
}

static int
xfs_setfl(struct vnode *vp,
	  int oflags,
	  int nflags,
	  struct cred *cred)
{
  XFSDEB(XDEBVNOPS, ("xfs_setfl\n"));
  return fs_setfl (vp, oflags, nflags, cred);
}

static int
xfs_getattr(struct vnode *vp,
	    struct vattr *vap,
	    int flags,
	    struct cred *cred)     
{
  int error = 0;
  
  struct xfs_node *xn = VNODE_TO_XNODE(vp);

  XFSDEB(XDEBVNOPS, ("xfs_getattr\n"));
  
  error = xfs_attr_valid(vp, cred, XFS_ATTR_R);
  if (error == 0) {
      *vap = xn->attr;
  }
  
  return error;
}

static int
xfs_setattr(struct vnode *vp,
	    struct vattr *vap,
	    int flags,
	    struct cred *cred)
{
  struct xfs *xfsp = XFS_FROM_VNODE(vp);
  struct xfs_node *xn = VNODE_TO_XNODE(vp);
  int error = 0;
  
  XFSDEB(XDEBVNOPS, ("xfs_setattr\n"));
  if (XFS_TOKEN_GOT(xn, XFS_ATTR_W)) {
      /* Update attributes and mark them dirty. */
      VNODE_TO_XNODE(vp)->flags |= XFS_ATTR_DIRTY;
      error = ENOSYS;		/* XXX not yet implemented */
      goto done;
  } else {
      struct xfs_message_putattr msg;
      msg.header.opcode = XFS_MSG_PUTATTR;
      msg.cred.uid = cred->cr_uid;
      msg.cred.pag = xfs_get_pag(cred);
      msg.handle = xn->handle;
      vattr2xfs_attr (vap, &msg.attr);
      error = xfs_message_rpc(xfsp->fd, &msg.header, sizeof(msg));
      if (error == 0)
	error = ((struct xfs_message_wakeup *) &msg)->error;
      XFS_TOKEN_CLEAR(xn, XFS_ATTR_VALID, XFS_ATTR_MASK);
  }

 done:
  return error;
}

static int
check_rights (u_char rights, int mode)
{
    int error = 0;

    if (mode & VREAD)
	if ((rights & XFS_RIGHT_R) == 0)
	    error = EACCES;
    if (mode & VWRITE)
	if ((rights & XFS_RIGHT_W) == 0)
	    error = EACCES;
    if (mode & VEXEC)
	if ((rights & XFS_RIGHT_X) == 0)
	    error = EACCES;
    return error;
}

static int
xfs_access(struct vnode *vp,
	   int mode,
	   int flags,
	   struct cred *cred)     
{
  int error = 0;
  pag_t pag = xfs_get_pag(cred);

  XFSDEB(XDEBVNOPS, ("xfs_access mode = 0%o\n", mode));

  error = xfs_attr_valid(vp, cred, XFS_ATTR_R);
  if (error == 0) {
      struct xfs_node *xn = VNODE_TO_XNODE(vp);
      int i;

      error = check_rights (xn->anonrights, mode);
	
      if (error == 0)
	  goto done;

      XFSDEB(XDEBVNOPS, ("xfs_access anonaccess failed\n"));

      if (error != 0)
	  xfs_fetch_rights(vp, cred); /* ignore error */
	
      error = EACCES;
	
      for (i = 0; i < MAXRIGHTS; i++)
	  if (xn->id[i] == pag) {
	      error = check_rights (xn->rights[i], mode);
	      break;
	  }
  }

done:
    XFSDEB(XDEBVNOPS, ("xfs_access(0%o) = %d\n", mode, error));
    return error;

}

static int
xfs_lookup(struct vnode *dvp,
	   char *nm,
	   struct vnode **vpp,
	   struct pathname *pnp,
	   int flags,
	   struct vnode *rdir,
	   struct cred *cred)
{
  struct xfs_message_getnode msg;
  struct xfs *xfsp = XFS_FROM_VNODE(dvp);
  int error = 0;

  struct xfs_node *d = VNODE_TO_XNODE(dvp);
  struct vnode *v;
  
  XFSDEB(XDEBVNOPS, ("xfs_lookup\n"));
  
  do {
#ifdef notdef_but_correct
    error = xfs_access(dvp, VEXEC, cred);
    if (error != 0)
      goto done;
#endif
    v = xfs_dnlc_lookup(dvp, nm);
    if (!v)
      {
	msg.header.opcode = XFS_MSG_GETNODE;
	msg.cred.uid = cred->cr_uid;
	msg.cred.pag = xfs_get_pag(cred);
	msg.parent_handle = d->handle;
	strncpy(msg.name, nm, sizeof(msg.name));
	error = xfs_message_rpc(xfsp->fd, &msg.header, sizeof(msg));
	if (error == 0)
	  error = ((struct xfs_message_wakeup *) &msg)->error;
      }
    else
      {
	*vpp = v;
	VN_HOLD(v);
	goto done;
      }
  } while (error == 0);

 done:
  XFSDEB(XDEBVNOPS, ("xfs_lookup() = %d\n", error));
  return error;
}

static int
xfs_create(struct vnode *dvp,
	   char *nm,
	   struct vattr *va,
	   enum vcexcl exclusive,
	   int mode,
	   struct vnode **vpp,
	   struct cred *cred)     
{
  struct xfs *xfsp = XFS_FROM_VNODE(dvp);
  struct xfs_node *xn = VNODE_TO_XNODE(dvp);
  int error = 0;
  int do_trunc = 0;
  
  XFSDEB(XDEBVNOPS, ("xfs_create\n"));
  {
    struct xfs_message_create msg;
    msg.header.opcode = XFS_MSG_CREATE;
    msg.parent_handle = xn->handle;
    strncpy(msg.name, nm, 256);
    vattr2xfs_attr (va, &msg.attr);
#if 0
    msg.exclusive = exclusive;
#endif
    msg.mode = mode;
    msg.cred.uid = cred->cr_uid;
    msg.cred.pag = xfs_get_pag(cred);
    error = xfs_message_rpc(xfsp->fd, &msg.header, sizeof(msg));
    if (error == 0)
      error = ((struct xfs_message_wakeup *) &msg)->error;
  }

  if (error == EEXIST) {
      do_trunc = 1;
      error = 0;
  }

  if (error == 0)
    error = xfs_lookup(dvp, nm, vpp, /*pnp*/ NULL, /*flags*/ 0,
		       /*rdir*/ NULL, cred);

  if (error = 0 && do_trunc)
      error = xfs_setattr (*vpp, va, 0, cred);

  return error;
}

static int
xfs_remove(struct vnode *dvp,
	   char *nm,
	   struct cred *cred)
{
  struct xfs *xfsp = XFS_FROM_VNODE(dvp);
  struct xfs_node *xn = VNODE_TO_XNODE(dvp);
  struct xfs_message_remove msg;
  int error;

  XFSDEB(XDEBVNOPS, ("xfs_remove: %s\n", nm));

  msg.header.opcode = XFS_MSG_REMOVE;
  msg.parent_handle = xn->handle;
  strncpy(msg.name, nm, 256);
  msg.cred.uid = cred->cr_uid;
  msg.cred.pag = xfs_get_pag(cred);
  error = xfs_message_rpc(xfsp->fd, &msg.header, sizeof(msg));
  if (error == 0)
      error = ((struct xfs_message_wakeup *) &msg)->error;

  return error;
}

static int
xfs_link(struct vnode *vp,
	 struct vnode *tdvp,
	 char *tnm,
	 struct cred *cred)     
{
    struct xfs *xfsp = XFS_FROM_VNODE(vp);
    struct xfs_node *xn = VNODE_TO_XNODE(tdvp);
    struct xfs_node *xn2 = VNODE_TO_XNODE(vp);
    struct xfs_message_link msg;
    int error = 0;

    XFSDEB(XDEBVNOPS, ("xfs_link: (%s)\n", tnm));

    msg.header.opcode = XFS_MSG_LINK;
    msg.parent_handle = xn->handle;
    msg.from_handle   = xn2->handle;
    strncpy(msg.name, tnm, 256);
    msg.cred.uid = cred->cr_uid;
    msg.cred.pag = xfs_get_pag(cred);

    error = xfs_message_rpc(xfsp->fd, &msg.header, sizeof(msg));
    if (error == 0)
	error = ((struct xfs_message_wakeup *) & msg)->error;

    return error;
}

static int
xfs_rename(struct vnode *sdvp,
	   char *onm,
	   struct vnode *tdvp,
	   char *nnm,
	   struct pathname *npnp,
	   struct cred *cred)     
{
  struct xfs_message_rename msg;
  struct vnode **vpp;
  int error = 0;

  XFSDEB(XDEBVNOPS, ("xfs_rename\n"));

  error = xfs_lookup(tdvp, nnm, vpp, /*pnp*/ NULL, /*flags*/ 0,
		     /*rdir*/ NULL, cred);
  if (error != ENOENT) {
      /* the filename being moved to already exists */
      struct xfs_message_remove remmsg;
      
      remmsg.header.opcode = XFS_MSG_REMOVE;
      remmsg.parent_handle = VNODE_TO_XNODE(tdvp)->handle;
      strncpy(remmsg.name, nnm, 256);
      remmsg.cred.uid = cred->cr_uid;
      remmsg.cred.pag = xfs_get_pag(cred);
      error = xfs_message_rpc(XFS_FROM_VNODE(tdvp)->fd, &remmsg.header,
			      sizeof(remmsg));
      if (error == 0)
	  error = ((struct xfs_message_wakeup *) & remmsg)->error;
      if (error != 0)
	  return error;
  }

  msg.header.opcode = XFS_MSG_RENAME;
  msg.old_parent_handle = VNODE_TO_XNODE(sdvp)->handle;
  strncpy(msg.old_name, onm, 256);
  msg.new_parent_handle = VNODE_TO_XNODE(tdvp)->handle;
  strncpy(msg.new_name, nnm, 256);
  msg.cred.uid = cred->cr_uid;
  msg.cred.pag = xfs_get_pag(cred);
  error = xfs_message_rpc(XFS_FROM_VNODE(sdvp)->fd, &msg.header,
			  sizeof(msg));
  if (error == 0)
      error = ((struct xfs_message_wakeup *) & msg)->error;
  
  return error;
}

static int
xfs_mkdir(struct vnode *dvp,
	  char *nm,
	  struct vattr *va,
	  struct vnode **vpp,
	  struct cred *cred)     
{
  struct xfs *xfsp = XFS_FROM_VNODE(dvp);
  struct xfs_node *xn = VNODE_TO_XNODE(dvp);
  int error = 0;
  
  XFSDEB(XDEBVNOPS, ("xfs_mkdir\n"));
  {
    struct xfs_message_mkdir msg;
    msg.header.opcode = XFS_MSG_MKDIR;
    msg.parent_handle = xn->handle;
    strncpy(msg.name, nm, 256);
    vattr2xfs_attr (va, &msg.attr);
    msg.cred.uid = cred->cr_uid;
    msg.cred.pag = xfs_get_pag(cred);
    error = xfs_message_rpc(xfsp->fd, &msg.header, sizeof(msg));
    if (error == 0)
      error = ((struct xfs_message_wakeup *) &msg)->error;
  }
  if (error == 0)
      error = xfs_lookup(dvp, nm, vpp, /*pnp*/ NULL, /*flags*/ 0,
			 /*rdir*/ NULL, cred);
  return error;
}

static int
xfs_rmdir(struct vnode *dvp,
	  char *nm,
	  struct vnode *foo,
	  struct cred *cred)     
{
    struct xfs_message_rmdir msg;
    struct xfs *xfsp  = XFS_FROM_VNODE(dvp);
    struct xfs_node *xn = VNODE_TO_XNODE(dvp);
    int error = 0;


    XFSDEB(XDEBVNOPS, ("xfs_rmdir\n"));

    msg.header.opcode = XFS_MSG_RMDIR;
    msg.parent_handle = xn->handle;
    strncpy(msg.name, nm, 256);
    msg.cred.uid = cred->cr_uid;
    msg.cred.pag = xfs_get_pag(cred);
    error = xfs_message_rpc(xfsp->fd, &msg.header, sizeof(msg));
    if (error == 0)
	error = ((struct xfs_message_wakeup *) &msg)->error;

    return error;
}

static int
xfs_readdir(struct vnode *vp,
	    struct uio *uiop,
	    struct cred *cred,
	    int *eofp)		/* XXX */
{
  int error = 0;

  XFSDEB(XDEBVNOPS, ("xfs_readdir\n"));

  error = xfs_data_valid(vp, cred, XFS_DATA_R);
  if (error == 0) {
      struct vnode *t      = DATA_FROM_VNODE(vp);

      ASSERT(t != NULL);

      if(ABI_IS_64BIT(GETDENTS_ABI(u.u_procp->p_abi, uiop))) {
	  VOP_RWLOCK(t, 0);
	  error = VOP_READ(t, uiop, 0, cred);
	  VOP_RWUNLOCK(t, 0);
      } else {
	  struct uio tmp_uio;
	  struct iovec iovec;
	  char *buf;
	  size_t count = uiop->uio_resid;

	  buf = xfs_alloc (count);
	  iovec.iov_base     = buf;
	  iovec.iov_len      = count;
	  tmp_uio.uio_iov    = &iovec;
	  tmp_uio.uio_iovcnt = 1;
	  tmp_uio.uio_offset = uiop->uio_offset;
	  tmp_uio.uio_segflg = UIO_SYSSPACE;
	  tmp_uio.uio_fmode  = uiop->uio_fmode;
	  tmp_uio.uio_limit  = uiop->uio_limit;
	  tmp_uio.uio_resid  = count;
	  VOP_RWLOCK(t, 0);
	  error = VOP_READ(t, &tmp_uio, 0, cred);
	  VOP_RWUNLOCK(t, 0);

	  if (error == 0) {
	      char *ptr;
	      struct dirent64 *d;
	      size_t len = count - tmp_uio.uio_resid;
	      
	      for (ptr = buf;
		   ptr < buf + len;
		   ptr += d->d_reclen) {
		  char tmp_buf[1024];
		  struct irix5_dirent *d5 = (struct irix5_dirent *)tmp_buf;

		  d = (struct dirent64 *)ptr;

		  d5->d_ino    = d->d_ino;
		  d5->d_off    = d->d_off;
		  d5->d_reclen = DIRENTSIZE(strlen(d->d_name));
		  strcpy (d5->d_name, d->d_name);

		  error = uiomove (d5, d5->d_reclen, UIO_READ, uiop);
		  if (error) {
		      XFSDEB(XDEBVNOPS, ("xfs_readdir: uiomove failed: %d\n",
					 error));
		      break;
		  }
	      }
	  }
	  xfs_free (buf, count);
      }
  }

  return error;
}

static int
xfs_symlink(struct vnode *dvp,
	    char *lnm,
	    struct vattr *tva,
	    char *tnm,
	    struct cred *cred)
{
    struct xfs *xfsp  = XFS_FROM_VNODE(dvp);
    struct xfs_node *xn = VNODE_TO_XNODE(dvp);
    struct xfs_message_symlink msg;
    int error = 0;
 
    XFSDEB(XDEBVNOPS, ("xfs_symlink\n"));
   
    msg.header.opcode = XFS_MSG_SYMLINK;
    msg.parent_handle = xn->handle;
    strncpy(msg.name, lnm, 256);
    strncpy(msg.contents, tnm, sizeof(msg.contents));
    vattr2xfs_attr (tva, &msg.attr);
    msg.cred.uid = cred->cr_uid;
    msg.cred.pag = xfs_get_pag(cred);

    error = xfs_message_rpc(xfsp->fd, &msg.header, sizeof(msg));
    if (error == 0)
	error = ((struct xfs_message_wakeup *) & msg)->error;

    return error;
}

static int
xfs_readlink(struct vnode *vp,
	     struct uio *uiop,
	     struct cred *cred)
{
  int error = 0;

  XFSDEB(XDEBVNOPS, ("xfs_readlink\n"));

  error = xfs_data_valid(vp, cred, XFS_DATA_R);
  if (error == 0)
    {
      struct vnode *t = DATA_FROM_VNODE(vp);
      ASSERT(t != NULL);
      VOP_RWLOCK(t, 0);
      error = VOP_READ(t, uiop, 0, cred);
      VOP_RWUNLOCK(t, 0);
    }

  return error;
}

static int
xfs_fsync(struct vnode *vp,
	  int syncflag,
	  struct cred *cred)
{
  struct xfs *xfsp = XFS_FROM_VNODE(vp);
  struct xfs_node *xn = VNODE_TO_XNODE(vp);
  int error = 0;
  
  XFSDEB(XDEBVNOPS, ("xfs_fsync\n"));

  if (xn->flags & XFS_DATA_DIRTY)
      error = do_fsync (xfsp, xn, cred, XFS_WRITE);
  return error;
}

static void
xfs_inactive(struct vnode *vp,
	     struct cred *cred)
{
  struct xfs_message_inactivenode msg;
  struct xfs *xfsp = XFS_FROM_VNODE(vp);
  struct xfs_node *xn = VNODE_TO_XNODE(vp);

  XFSDEB(XDEBVNOPS, ("xfs_inactive\n"));

  msg.header.opcode = XFS_MSG_INACTIVENODE;
  msg.handle = xn->handle;
  msg.flag   = XFS_NOREFS; /* | XFS_DELETE; */
#if 0
  free_xfs_node(VNODE_TO_XNODE(vp));
#endif
  xfs_message_send(xfsp->fd, &msg.header, sizeof(msg));
}

static int
xfs_fid(struct vnode *vp,
	struct fid **fid)
{
  XFSDEB(XDEBVNOPS, ("xfs_fid\n"));
  return ENOSYS;
}

static int
xfs_fid2(struct vnode *vp,
	struct fid *fid)
{
  XFSDEB(XDEBVNOPS, ("xfs_fid2\n"));
  return ENOSYS;
}

static void
xfs_rwlock(struct vnode *vp,
	   vrwlock_t write_lock)
{
  XFSDEB(XDEBVNOPS, ("xfs_rwlock\n"));
}

static void
xfs_rwunlock(struct vnode *vp,
	     vrwlock_t write_lock)
{
  XFSDEB(XDEBVNOPS, ("xfs_rwunlock\n"));
}

static int
xfs_seek(struct vnode *vp,
	 off_t offset,
	 off_t *roffset)
{
  XFSDEB(XDEBVNOPS, ("xfs_seek\n"));
  return 0;
}

static int
xfs_cmp(struct vnode *vp1,
	struct vnode *vp2)
{
  XFSDEB(XDEBVNOPS, ("xfs_cmp\n"));
  return vp1 == vp2;
}

static int
xfs_frlock(struct vnode *vp,
	   int foo,
	   struct flock *fl,
	   int bar,
	   off_t off,
	   struct cred *cred)
{
  XFSDEB(XDEBVNOPS, ("xfs_frlock\n"));
  return ENOSYS;
}

static int
xfs_realvp(struct vnode *vp,
	   struct vnode **vpp)
{
  XFSDEB(XDEBVNOPS, ("xfs_realvp\n"));
  return ENOSYS;
}

static int
xfs_bmap (struct vnode *vp,
	  off_t off,
	  ssize_t sz,
	  int flags,
	  struct cred *cred,
	  struct bmapval *bv,
	  int *foo)
{
  XFSDEB(XDEBVNOPS, ("xfs_bmap\n"));
  return ENOSYS;

}

static void
xfs_strategy (struct vnode *vp,
	      struct buf *buf)
{
  XFSDEB(XDEBVNOPS, ("xfs_strategy\n"));
}

static int
xfs_map(struct vnode *vp,
	off_t off,
	struct pregion *pregion,
	char **a,
	size_t sz,
	u_int prot,
	u_int max_prot,
	u_int map_flag,
	struct cred *cred)
{
  int error = 0;
  struct xfs_node *xn = VNODE_TO_XNODE(vp);
  struct vattr *va = &xn->attr;

  XFSDEB(XDEBVNOPS, ("xfs_map\n"));

  error = fs_map_subr (vp, va->va_size, va->va_mode, off, pregion,
		       *a, sz, prot, max_prot, map_flag, cred);

  XFSDEB(XDEBVNOPS, ("xfs_map: error = %d\n", error));
}

#if 0
  if ((prot & PROT_WRITE) && (flags & MAP_SHARED))
    error = xfs_data_valid(vp, cred, XFS_DATA_W);
  else
    error = xfs_data_valid(vp, cred, XFS_DATA_R);
  
  if (error != 0)
    /* Can't map today */;
  else if (off + len > VNODE_TO_XNODE(vp)->attr.va_size)
    error = EINVAL;
#if 0
  else if ((prot & PROT_WRITE) && (flags & MAP_SHARED))
    error = EROFS;		/* XXX This is currently not supported */
#endif
  else
    {
      struct vnode *t = DATA_FROM_VNODE(vp);
      ASSERT(t != NULL);
      VOP_RWLOCK(t, 1);
      error = VOP_MAP(t, off, as, addrp, len, prot, maxprot, flags, cred);
      /* XXX Patch vnode so that we can intercept get/putpage and inactive. */
      VOP_RWUNLOCK(t, 1);
    }

  return error;
  XFSDEB(XDEBVNOPS, ("xfs_map\n"));
  return ENOSYS;

}
#endif

static int
xfs_addmap(struct vnode *vp,
	   off_t off,
	   struct pregion *as,
	   addr_t addr,
	   size_t len,
	   u_int prot,
	   u_int maxprot,
	   u_int flags,
	   struct cred *cred)
{
  XFSDEB(XDEBVNOPS, ("xfs_addmap\n"));
  return 0;
}

static int
xfs_delmap(struct vnode *vp,
	   off_t off,
	   struct pregion *as,
	   addr_t addr,
	   u_int len,
	   u_int prot,
	   u_int maxprot,
	   u_int flags,
	   struct cred *cred)
{
  XFSDEB(XDEBVNOPS, ("xfs_delmap\n"));
  return 0;
}


static int
xfs_poll(struct vnode *vp,
	 short events,
	 int anyyet,
	 short *revents,
	 struct pollhead **ph)
{
  XFSDEB(XDEBVNOPS, ("xfs_poll\n"));
  return fs_poll(vp, events, anyyet, revents, ph);
}

static int
xfs_dump(struct vnode *dumpvp,
	 caddr_t addr,
	 daddr_t darr,
	 u_int foo)
{
  XFSDEB(XDEBVNOPS, ("xfs_dump\n"));
  return ENOSYS;
}

static int
xfs_pathconf(struct vnode *vp,
	     int cmd,
	     u_long *valp,
	     struct cred *cred)
{
  XFSDEB(XDEBVNOPS, ("xfs_pathconf\n"));
  return fs_pathconf (vp, cmd, valp, cred);
}

static int
xfs_allocstore(struct vnode *vp,
	       off_t off,
	       size_t sz,
	       struct cred *cred)
{
  XFSDEB(XDEBVNOPS, ("xfs_allocstore\n"));
  return ENOSYS;
}

static int
xfs_fcntl(struct vnode *vp,
	  int cmd,
	  void *arg,
	  int foo,
	  off_t off,
	  struct cred *cred,
	  union rval *result)
{
  XFSDEB(XDEBVNOPS, ("xfs_fcntl\n"));
  return ENOSYS;
}

static int
xfs_reclaim (struct vnode *vp,
	     int foo)
{
  struct xfs_message_inactivenode msg;
  struct xfs *xfsp = XFS_FROM_VNODE(vp);
  struct xfs_node *xn = VNODE_TO_XNODE(vp);

  XFSDEB(XDEBVNOPS, ("xfs_reclaim\n"));

  msg.header.opcode = XFS_MSG_INACTIVENODE;
  msg.handle = xn->handle;
  msg.flag   = XFS_NOREFS | XFS_DELETE;
#if 1
  free_xfs_node(VNODE_TO_XNODE(vp));
#endif
  xfs_message_send(xfsp->fd, &msg.header, sizeof(msg));
  return 0;
}

static int
xfs_attr_get (struct vnode *vp,
	      char *a,
	      char *b,
	      int *c,
	      int d,
	      struct cred *cred)
{
  XFSDEB(XDEBVNOPS, ("xfs_attr_get\n"));
  return ENOSYS;
}

static int
xfs_attr_set (struct vnode *vp,
	      char *a,
	      char *b,
	      int c,
	      int d,
	      struct cred *cred)
{
  XFSDEB(XDEBVNOPS, ("xfs_attr_set\n"));
  return ENOSYS;
}

static int
xfs_attr_remove (struct vnode *vp,
		 char *a,
		 int b,
		 struct cred *cred)
{
  XFSDEB(XDEBVNOPS, ("xfs_attr_remove\n"));
  return ENOSYS;
}

static int
xfs_attr_list (struct vnode *vp,
	       char *a,
	       int b,
	       int c,
	       struct attrlist_cursor_kern *k,
	       struct cred *cred)
{
  XFSDEB(XDEBVNOPS, ("xfs_attr_list\n"));
  return ENOSYS;
}

struct vnodeops xfs_vnodeops = {
        xfs_open,
        xfs_close,
        xfs_read,
        xfs_write,
        xfs_ioctl,
	xfs_setfl,
        xfs_getattr,
        xfs_setattr,
        xfs_access,
        xfs_lookup,
        xfs_create,
        xfs_remove,
        xfs_link,
        xfs_rename,
        xfs_mkdir,
        xfs_rmdir,
        xfs_readdir,
        xfs_symlink,
        xfs_readlink,
        xfs_fsync,
        xfs_inactive,
	xfs_fid,
	xfs_fid2,
	xfs_rwlock,
	xfs_rwunlock,
	xfs_seek,
	xfs_cmp,
	xfs_frlock,
	xfs_realvp,
        xfs_bmap,
        xfs_strategy,
        xfs_map,
	xfs_addmap,
	xfs_delmap,
	xfs_poll,
        xfs_dump,
	xfs_pathconf,
	xfs_allocstore,
	xfs_fcntl,
	xfs_reclaim,
	xfs_attr_get,
	xfs_attr_set,
	xfs_attr_remove,
	xfs_attr_list
};
