
/* copyright (c) 2000 
   the regents of the university of michigan 
   all rights reserved 
   
   permission is granted to use, copy, create derivative works and  
   redistribute this software and such derivative works for any purpose,  
   so long as the name of the university of michigan is not used in any  
   advertising or publicity pertaining to the use or distribution of this  
   software without specific, written prior authorization.  if the above  
   copyright notice or any other identification of the university of  
   michigan is included in any copy of any portion of this software, then  
   the disclaimer below must also be included. 
   
   this software is provided as is, without representation from the  
   university of michigan as to its fitness for any purpose, and without  
   warranty by the university of michigan of any kind, either express or  
   implied, including without limitation the implied warranties of  
   merchantability and fitness for a particular purpose.  the regents of  
   the university of michigan shall not be liable for any damages,  
   including special, indirect, incidental, or consequential damages, with  
   respect to any claim arising out or in connection with the use of the  
   software, even if it has been or is hereafter advised of the  
   possibility of such damages. */

/*
    scfs.c
    Called by scfsd to access smartcard
    NI, 5/29, 1998
*/

#include "scfs.h"
#include "scrw.h"
#include "scfs_pioctl.h"
#include <sys/mount.h>

#define get_vfc_index(x) "xfs" /*MOUNT_XFS*/

/* Global Variables */
int fd;
unsigned char *cur_fid;
/*int flag_mounted;*/

/* extern */
extern struct scconfig scconfig;

int scfs_init(int port, int flag)
{
  int rv = 0;
  char atr[256];
  int mntflags = 0;
  
  /*arla_warnx (ADEBMSG, "scfs_init()");*/

  /*if (flag_mounted == FLAG_MOUNTED) {
    arla_warnx (ADEBERROR, "Trying to mount twice");
    exit(1);
  }
  flag_mounted = FLAG_MOUNTED; */

  /* reset smartcard */
  if ((rv = scfs_mount(port, atr, flag))) {
    arla_warnx (ADEBMSG, "cannot mount smartcard");
    return rv;
  }

  /* init scdir */
  if ((rv = scdir_init(atr, (char *)NULL))) {
    arla_warnx (ADEBMSG, "cannot initialize scdir information: %s", SCDFILE);
    return rv;
  }

  /* install 3f.00 */
  if ((rv = scdir_installroot())){ 
    arla_warnx (ADEBMSG, "cannot install scdir root");
    return rv;
  }

  /*arla_warnx (ADEBMSG, "Now mount it!!");*/

  /*if ((rv = mount(get_vfc_index ("xfs"), "/smartcard", mntflags, "/dev/xfs0"))) {
    perror("mount failed"); 
    return rv;
    }*/

#ifdef NIXX
  if ((pid = fork()) < 0) {
    arla_warnx (ADEBERROR, "scfs_init() - cannot create a process");
    return EIO; 
  } else if (pid == 0) { /* child */
    arla_warnx (ADEBMSG, "scfs_init() - child start");
    execl("/bin/sh", "sh", "-c", "/usr/arla/bin/mount_xfs /dev/xfs0 /smartcard",
	  (char *) 0);
    arla_warnx (ADEBMSG, "scfs_init() - child done");
    return 0; 
  }
#endif /* NIXX */

  arla_warnx (ADEBMSG, "Mount OK. :)");
  
  return 0; 
}
  
int scfs_mount (int port, char *buf, int flag)
{
  int vflag = 1, error, n;
    
  /*arla_warnx (ADEBMSG, "scfs_mount (%d) :)", port);*/

  fd = scopen(port, SCODSR, &error);
  /*printf("scopen on port %d returns %d\n", port, fd); */
  if (fd < 0) {
    arla_warnx (ADEBMSG, "can't open reader");
    errno = error; 
    perror("scopen");
    return error;
  }

  /* reset */
  n = screset(fd, buf, &error);
  arla_warnx (ADEBMSG, "screset returns %d, error=%d", n, error);
  if (!n || error != SCEOK) {
    arla_warnx (ADEBMSG, "%s (%d %d)", scerrtab[error], n, error);
    return EIO;
  }
  if (buf){
    if (vflag) {
      /*arla_warnx (ADEBMSG, "atr: ");*/
      dump_reply(buf, n, 0, 0);
    }
  }
  return 0;
}

/* called upon unmount.  free scdir struct, close fd */

int scfs_unmount()
{
  /*arla_warnx (ADEBMSG, "closing fd = %d", fd);*/
  scclose (fd);

  /*arla_warnx (ADEBMSG, "scfs_unmount");*/
  scdir_free();
  
  return 0;
}
     
int scfs_open_file(int f0, int f1)
{
  return open_file(fd, f0, f1);
}

int open_file(fd, f0, f1)
int fd, f0, f1;
{
    char buf[2];
    int len, r1, r2;

    buf[0] = f0 & 0xff;
    buf[1] = f1 & 0xff;
    /*fprintf(stderr, "open_file %x.%x\n", buf[0], buf[1]);*/
    len = scwrite(fd, scconfig.cla_select, 0xa4, 0, 0, 2, buf, &r1, &r2);
    if (len == 2 && ((r1 == 144 && r2 == 0) || (r1 == 0x61))) {
      return 0;
    }
    
    /*fprintf(stderr, "open->scwrite returns len %d %02x %02x\n", len, r1, r2);*/
    return -1;
}

/* take 2 characters, and convert it to a number (char) */

int a2h(char c, char d, unsigned char *val)
{
  *val = 0;
  if ((c >= '0') && (c <= '9')) *val += c - '0';
  else if ((c >= 'a') && (c <= 'f')) *val += c - 'a' + 10;
  else if ((c >= 'A') && (c <= 'F')) *val += c - 'A' + 10;
  else return -1;
  *val <<= 4;
  if ((d >= '0') && (d <= '9')) *val += d - '0';
  else if ((d >= 'a') && (d <= 'f')) *val += d - 'a' + 10;
  else if ((d >= 'A') && (d <= 'F')) *val += d - 'A' + 10;
  else return -1;
  return(*val);
}

/* Read smartcard.  Fill buf.  Return # of bytes read.*/

int scfs_read(unsigned char *fid, int offset, int length, unsigned char *buf, int rec_flag, unsigned char *parent_fid, dev_t t_dev)
{
  struct scdnode *node;
  int sw1, sw2, rv, tmp_offset, tmp_length, i;

  /*arla_warnx (ADEBMSG, "scfs_read on tty %s.", scfs_ttyname(t_dev));*/
  
  if (rec_flag == SCFS_IS_NOT_RECORD) {
    /* This is not record ... do as usual. */
    if ((node = get_scdnode(fid)) == NULL) {
      arla_warnx (ADEBMSG, "cannot find %02x.%02x", fid[0], fid[1]);
      return -1;
    }

    /* offset out of range */
    if (offset > node->fileent.l_size) return -1;
    if (offset == node->fileent.l_size) return 0;
	
    /* if requested length is longer than logical length of the file,
       read local length byte of data. */
    length = length + offset > node->fileent.l_size ?
      node->fileent.l_size - offset : length;
    
    if (godown(node) == -1) return -1;
    
    /* read it */
  readagain:
    /* loop to handle a long message*/
    tmp_offset = offset;
    for (i=0; i<length/SCMaxRWLen+1; i++){
      tmp_length = length - i*SCMaxRWLen > SCMaxRWLen
	? SCMaxRWLen : length - i*SCMaxRWLen;
      rv = scread(fd, scconfig.cla_readbinary, 0xb0, (tmp_offset >> 8) & 0xff,
		  tmp_offset & 0xff, tmp_length, buf, &sw1, &sw2);
      if (rv == -1 || sw1 != 0x90 || sw2 != 0x00) {
	if (sw1 == 0x69 && sw2 == 0x82) {
	  char *pin;
	  arla_warnx (ADEBWARN, "permission denied.  input PIN");
	  if (scconfig.multiflexpin == 0) {
	    arla_warnx (ADEBWARN, "This smartcard does not have MultiFlex type PIN support");
	  } else {
	    /*scfs_tcl_readpin(pin);*/
	    pin = getpin(scfs_ttyname(t_dev), "Input PIN: ");
	    /*arla_warnx (ADEBMSG, "Got PIN %02x %02x %02x %02x %02x %02x %02x %02x",
			pin[0], pin[1], pin[2], pin[3],
			pin[4], pin[5], pin[6], pin[7]);*/
	    rv = scwrite(fd, scconfig.cla_verifypin, 0x20, 0, 1, 8, pin, &sw1, &sw2);
	    if (rv != -1 && sw1 == 0x90 && sw2 == 0x00) {
	      /*arla_warnx (ADEBMSG, "PIN verified");*/
	      goto readagain;
	    } else {
	      arla_warnx (ADEBERROR, "incorrect PIN");
	      return -1; /* incorrect PIN.  return. */
	    }
	    arla_warnx (ADEBERROR, "incorrect PIN?");
	    return -1; /* incorrect PIN.  return. */
	  }
	} else {
	  if (rv == 0) return rv; /* 0 byte read.  return 0. */
	  return -1; /* error other than permission.  return. */
	}
      }
      tmp_offset += tmp_length;
      buf += tmp_length;
    }
  } else { /* This is record. */
    if ((node = get_scdnode(parent_fid)) == NULL) {
      arla_warnx (ADEBMSG, "cannot find %02x.%02x",
		  parent_fid[0], parent_fid[1]);
      return -1;
    }
    if (godown(node) == -1) return -1; /* go down to parent (=record file) */
  rec_readagain:
    rv = scread(fd, scconfig.cla_readrecord, 0xb2, fid[1] & 0xff, 0x04, length, buf, &sw1, &sw2);
    if (rv != -1 && sw1 == 0x90 && sw2 == 0x00) {
      return 0;
    } else if (sw1 == 0x69 && sw2 == 0x82) {
      char pin[16];
      arla_warnx (ADEBWARN, "permission denied.  input PIN");
      if (scconfig.multiflexpin == 0) {
	arla_warnx (ADEBWARN, "This smartcard does not have MultiFlex type PIN support");
	return -1;
      } else {
	scfs_tcl_readpin(pin);
	rv = scwrite(fd,scconfig.cla_verifypin, 0x20, 0, 1, 8, pin, &sw1, &sw2);
	if (rv != -1 && sw1 == 0x90 && sw2 == 0x00) {
	  /*arla_warnx (ADEBMSG, "PIN verified");*/
	  goto rec_readagain;
	} else {
	  arla_warnx (ADEBERROR, "incorrect PIN");
	  return -1;
	}
	arla_warnx (ADEBERROR, "incorrect PIN?");
	return -1;
      }
    }
  }

  return length;
}

/* Write to smartcard file.
   If the length is more than 0xff long, chop it into 0xff byte pieces. 
   Also, update directory entry of the parent.
 */

int scfs_write(unsigned char *fid, int offset, int length, char *buf, int rec_flag, unsigned char *parent_fid, dev_t t_dev)
{
  struct scdnode *node; 
  int rv, sw1, sw2, i, tmp_length, tmp_offset;

  assert(buf);
  
  /* offset out of range */

  if (rec_flag == SCFS_IS_NOT_RECORD) {
    /*arla_warnx (ADEBMSG, "scfs_write: not a record.\n");*/
    /* This is not record ... do as usual. */
    if ((node = get_scdnode(fid)) == NULL) {
      arla_warnx (ADEBMSG, "cannot find %02x.%02x", fid[0], fid[1]);
      return -1;
    }

    if (offset > node->fileent.l_size) {
      arla_warnx (ADEBMSG, "scfs_write: offset out of range 1.\n");
      return -1;
    }

    if (offset == node->fileent.l_size && offset != 0) {
      arla_warnx (ADEBMSG, "scfs_write: offset out of range 2.\n");
      return 0;
    }
    
    /* if requested length is longer than logical length of the file,
       read local length byte of data. */
    /*arla_warnx (ADEBMSG, "scfs_write: length in %d.\n", length);*/
    length = length + offset > node->fileent.p_size ?
      node->fileent.p_size - offset : length;
    /*arla_warnx (ADEBMSG, "scfs_write: length set %d.\n", length);*/
    /* select the file to be written */
    if (godown(node) == -1) {
      arla_warnx (ADEBMSG, "scfs_write: godown() failed.\n");
      return -1;
    }

    /* write it */
  writeagain:
    /* loop */
    tmp_offset = offset;
    for (i=0; i<length/SCMaxRWLen+1; i++){
      tmp_length = length - i*SCMaxRWLen > SCMaxRWLen
	? SCMaxRWLen : length - i*SCMaxRWLen;
      /*arla_warnx (ADEBWARN, "scfs_write: now write it ...");*/
      rv = scwrite(fd, scconfig.cla_updatebinary, 0xd6,
		   (tmp_offset >> 8) & 0xff, tmp_offset & 0xff, tmp_length,
		   buf, &sw1, &sw2);
      if (rv == -1 || sw1 != 0x90 || sw2 != 0x00) {
	if (sw1 == 0x69 && sw2 == 0x82) {
	  char *pin; 
	  arla_warnx (ADEBWARN, "permission denied.  input PIN");
	  if (scconfig.multiflexpin == 0) {
	    arla_warnx (ADEBWARN, "This smartcard does not have MultiFlex type PIN support");
	  } else {
	    /*scfs_tcl_readpin(pin);*/
	    pin = getpin(scfs_ttyname(t_dev), "Input PIN: ");
	    /*arla_warnx (ADEBMSG, "Got PIN %02x %02x %02x %02x %02x %02x %02x %02x",
			pin[0], pin[1], pin[2], pin[3],
			pin[4], pin[5], pin[6], pin[7]);*/
	    rv = scwrite(fd, scconfig.cla_verifypin, 0x20, 0, 1, 8, pin, &sw1, &sw2);
	    if (rv != -1 && sw1 == 0x90 && sw2 == 0x00) {
	      /*arla_warnx (ADEBMSG, "PIN verified");*/
	      goto writeagain;
	    } else {
	      arla_warnx (ADEBERROR, "incorrect PIN");
	      return EACCES; /* incorrect PIN.  return. */
	    }
	  }
	}
	arla_warnx (ADEBERROR, "write error.  smartcard returns %02x %02x",
		    sw1, sw2);
	if (sw1 == 0x6b && sw2 == 0x00)
	  arla_warnx (ADEBERROR, "Offset out of range.");
	return -1;
      }
      tmp_offset += tmp_length;
      buf += tmp_length;
    }

    arla_warnx (ADEBWARN, "scfs_write: writing a file is done.");

    node->fileent.l_size = offset + length;

    /* update directory entry */
    if ((rv = scfs_update_dirent(node))) return rv;
  }
  return rv;
}

int scfs_verifypin(int insize, char *pin)
{
  int len, r1, r2;

  if (scconfig.cla_verifypin == CLA_NOSUPP) {
    arla_warnx (ADEBERROR, "verify PIN APDU is not supported in this smartcard.");
    return EINVAL;
  }
  /*
  for (i=0; i<insize; i++) {
    fprintf(stderr, "%02x ", pin[i]);
  }
  fprintf(stderr, "\n");*/
  
  len = scwrite(fd, scconfig.cla_verifypin, 0x20, 0, 0x01, insize, pin, &r1, &r2);
  if (r1 == 144 && r2 == 0) {
    return 0;
  }
  else {
    arla_warnx (ADEBERROR, "verify PIN -> scwrite returns len %d %02x %02x", len, r1, r2);
    return EINVAL;
  }
}

int scfs_inauth(int keynum, int insize, char *key)
{
  int len, r1, r2;

  if (scconfig.cla_inauth == CLA_NOSUPP) {
    arla_warnx (ADEBERROR, "Internal Authentication APDU is not supported in this smartcard.");
    return EINVAL;
  }

  /*  for (i=0; i<insize; i++) {
    fprintf(stderr, "%02x ", key[i]);
  }
  fprintf(stderr, "\n");*/
  
  len = scwrite(fd, scconfig.cla_inauth, 0x88, 0, keynum, insize, key, &r1, &r2);
  if ((r1 == 0x90 && r2 == 0) || r1 == 0x61) {
    return 0;
  }
  else {
    arla_warnx (ADEBERROR, "Internal Authentication Error -> scwrite returns len %d %02x %02x", len, r1, r2);
    return EINVAL;
  }
}

int scfs_getresponse(char *str, int outsize)
{
  int len, r1, r2;

  if (scconfig.cla_getresponse == CLA_NOSUPP) {
    arla_warnx (ADEBERROR, "Get Response APDU is not supported in this smartcard.");
    return EINVAL;
  }
    
  len = scread(fd, scconfig.cla_getresponse, 0xc0, 0, 0, outsize, str, &r1, &r2);
  if (r1 == 0x90 && r2 == 0) {
    return 0;
  }
  else {
    arla_warnx (ADEBERROR, "Get Response Error -> scwrite returns len %d %02x %02x", len, r1, r2);
    return EINVAL;
  }
}

int scfs_getchallenge(char *str, int outsize)
{
  int len, r1, r2;
  
  if (scconfig.cla_getchallenge == CLA_NOSUPP) {
    arla_warnx (ADEBERROR, "Get Challenge APDU is not supported in this smartcard.");
    return EINVAL;
  }
  
  len = scread(fd, scconfig.cla_getchallenge, 0x84, 0, 0, outsize, str, &r1, &r2);
  if (r1 == 0x90 && r2 == 0) {
    return 0;
  }
  else {
    arla_warnx (ADEBERROR, "Get Challenge Error -> scwrite returns len %d %02x %02x", len, r1, r2);
    return EINVAL;
  }
}

int scfs_exauth(int keynum, int insize, char *response)
{
  int len, r1, r2, i;
  char buf[7];
  
  if (scconfig.cla_exauth == CLA_NOSUPP) {
    arla_warnx (ADEBERROR, "External Authentication APDU is not supported in this smartcard.");
    return EINVAL;
  }
    
  buf[0] = (char)keynum;
  for (i=0; i<insize; i++) {
    buf[i+1] = response[i];
    /*fprintf(stderr, "%02x ", buf[i+1]);*/
  }
  /*fprintf(stderr, "\n");*/
  
  len = scwrite(fd, scconfig.cla_exauth, 0x82, 0, 0, 7, buf, &r1, &r2);
  if (r1 == 0x90 && r2 == 0) {
    return 0;
  }
  else {
    arla_warnx (ADEBERROR, "External Authentication Error -> scwrite returns len %d %02x %02x", len, r1, r2);

    return EINVAL;
  }
}

int scfs_tcl_readpin(char *pin)
{
  int n, fd[2], i;
  pid_t pid;
  char line[256];
  char *systest = "/home/itoi/tcl/test/pin.tcl";
  
  if (pipe(fd) < 0) {
    arla_warnx (ADEBERROR, "pipe error");
    return -1;
  }
  if ((pid=fork()) < 0) {
    arla_warnx (ADEBERROR, "fork error");
    close (fd[0]);
    close (fd[1]);
    return -1;
  }
  if (pid>0) {/* parent read */
    close (fd[1]);
    n = read (fd[0], line, 256);
    line[n] = NULL;
    strncpy (pin, line, n);
    for (i=n; i<16; i++) pin[i] = 0xff;
  } else { /* child write */
    close(fd[0]);
    if (fd[1] != STDOUT_FILENO) {
      if (dup2(fd[1], STDOUT_FILENO) != STDOUT_FILENO){
	arla_warnx(ADEBERROR, "dup2 error to stdout");
	close (fd[1]);
	return -1;
      }
      close (fd[1]);
    }
    
    if (execl(systest, systest, (char *)0) < 0) {
      arla_warnx(ADEBERROR, "execl error for %s\n", systest);
      close (fd[0]);
      close (fd[1]);
      return (-1);
    }
  }
  return 0;
}

int scfs_query_file(struct scfs_response *scfs_res, struct scdnode *node)
{
  char str[16];

  if (scconfig.multiflexgetres == 0) {
    scfs_res->size = 256;
    return 0;
  }
  if (godown(node) == -1) return -1;
  if (scfs_getresponse(str, 0x0f) != 0) {
    return -1;
  }
  /*for (i=0; i<15; i++) {fprintf(stderr, "%02x ", str[i]);}
    fprintf(stderr, "\n");*/
    
  scfs_res->size = (str[2] & 0xff) * 0x100 + (str[3] & 0xff);
  /*fprintf(stderr, "size=%x\n", scfs_res->size);*/
  return 0;
}

/* optimized godown */
int godown(struct scdnode *node)
{
  int i, j;
  struct scdnode *tmp;
  int path[MaxDepth][2];

  /* Optimized godown() */
  /* if the file is already selected, do nothing. */
  if (fidcmp(node->fileent.fid, cur_fid)) {
    /*return 0; */
    return open_file(fd, node->fileent.fid[0], node->fileent.fid[1]);
  }

  if (node->fileent.d_type != SCFS_TYPE_ELFILE) goto select_from_root;

  /* if the file is under current directory, just select it. */
  if (fidcmp(node->parent->fileent.fid, cur_fid)) {
    return open_file(fd, node->fileent.fid[0], node->fileent.fid[1]);
  }

  if (!(tmp = get_scdnode(cur_fid))) {
    arla_warnx (ADEBERROR, "cur_fid %02x.%02x is not found?\n",
		cur_fid[0], cur_fid[1]);
    return -1;
  }
  if (tmp->fileent.d_type != SCFS_TYPE_ELFILE) goto select_from_root;
  /* if the file is current's sibling, just select it. */
  if (fidcmp(node->parent->fileent.fid, tmp->parent->fileent.fid)) {
    return open_file(fd, node->fileent.fid[0], node->fileent.fid[1]);
  }

select_from_root:
  for (tmp = node, i=0; tmp != NULL; tmp = tmp->parent, i++) {
    path[i][0] = tmp->fileent.fid[0];
    path[i][1] = tmp->fileent.fid[1];
  }

  for (j=i-1; j>=0; j--) {
    /*arla_warnx (ADEBMSG, "select %02x.%02x ", path[j][0], path[j][1]);*/
    if (open_file(fd, path[j][0], path[j][1]) != 0) {
      fprintf(stderr, "cannot open %02x %02x\n", path[j][0], path[j][1]);
      return -1;
    }
  }
  cur_fid = node->fileent.fid;
  return 0;
}

/* Update a directory entry for 'node'.
   The entry is in the parent's meta data (.i) */

int scfs_update_dirent(struct scdnode *node)
{
  struct scdnode *parent;
  int i, rv;

  /* do not update directory entry if the updated file is .i */
  if (node->fileent.fid[0] == DIRENT_FILE0 &&
      node->fileent.fid[1] == DIRENT_FILE1) return 0;

  assert(node);
  /* get parent */
  if ((parent = node->parent) == NULL){
    arla_warnx (ADEBERROR, "No parent??");
    return -1;
  }

  /* I'm the i'th child of the parent */
  if ((i = find_child_in_parent(parent, node)) == -1) {
    arla_warnx (ADEBERROR, "Can't find??");
    return -1;
  }
  if ((rv = update_ith_dirent(parent, i, &node->fileent))) return rv;
    
  return 0;
}  

/* update i'th entry of parent_node/.i (meta data) */
/* this i'th includes .i
   for example, if the directory contains (.i aa bb), then .i is 0'th,
   aa is 1'th, and bb is 2'th entry. */
int update_ith_dirent(struct scdnode *parent_node, int ith, struct fileent *fe)
{
  int de_offset, rv, sw1, sw2;

  assert(parent_node);
  assert(ith >=0);
  assert(ith < MaxNumChildren + 1); /* +1 for .i */

  if ((rv = godown(parent_node))) return rv;
  /*arla_warnx (ADEBMSG, "I am %d'th child of the parent.", ith);*/
  de_offset = (ith - 1) * sizeof(struct fileent); /* ith -1 because of .i */

  /* open .i */
  if (open_file(fd, DIRENT_FILE0, DIRENT_FILE1) != 0) {
    arla_warnx (ADEBERROR, "Cannot open directory entry %02x.%02x??",
		  DIRENT_FILE0, DIRENT_FILE1);
      return -1;
  }
  rv = scwrite(fd, scconfig.cla_updatebinary, 0xd6, (de_offset >> 8) & 0xff, de_offset & 0xff, sizeof(struct fileent), (char *)fe, &sw1, &sw2);
  if (rv == -1) {
    arla_warnx (ADEBERROR, "write error.  smartcard returns %02x %02x", sw1, sw2);
    if (sw1 == 0x6b && sw2 == 0x00)
      arla_warnx (ADEBERROR, "Offset out of range.");
    return -1;
  }
  return 0;
}
  
int scfs_add_dirent(struct scdnode *parent_node, struct scdnode *new_node)
{
  int i, de_offset, sw1, sw2, rv; 

  assert(parent_node);
  assert(new_node);
  
  /* I'm the i'th child of the parent */
  /*arla_warnx (ADEBMSG, "I am %d'th child of the parent.", parent_node->num_children);*/
  if (parent_node->num_children >= MaxNumChildren + 1) return EIO;
  i = parent_node->num_children;
  
  de_offset = (i - 1) * sizeof(struct fileent); /* i-1 because of .i */

  /* open parent, then meta data (.i) */
  godown(parent_node);
  if (open_file(fd, DIRENT_FILE0, DIRENT_FILE1) != 0) {
    arla_warnx (ADEBERROR, "Cannot open directory entry %02x.%02x??",
		DIRENT_FILE0, DIRENT_FILE1);
    return -1;
  }
  
  rv = scwrite(fd, scconfig.cla_updatebinary, 0xd6, (de_offset >> 8) & 0xff, de_offset & 0xff, sizeof(struct fileent), (char *)&new_node->fileent, &sw1, &sw2);
  if (sw1 == 0x69 || sw2 == 82) return EACCES;
  if (rv == -1) {
    arla_warnx (ADEBERROR, "write error.  smartcard returns %02x %02x", sw1, sw2);
    if (sw1 == 0x6b && sw2 == 0x00)
      arla_warnx (ADEBERROR, "Offset out of range.");
    return EIO;
  }

  return 0;
}

int scfs_remove(xfs_handle parent_fid, char *name)
{
  struct scdnode *parent_node;
  xfs_handle child_fid;
  int sw1, sw2, rv;
  
  assert(parent_fid);
  assert(name);

  if (name[0] == DIRENT_FILE0 && name[1] == DIRENT_FILE1) return EPERM;

  /*arla_warnx (ADEBMSG, "scfs_remove parent: %02x.%02x.%02x.%02x.%02x.%02x.%02x.%02x\n",
	      parent_fid[0], parent_fid[1],
	      parent_fid[2], parent_fid[3],
	      parent_fid[4], parent_fid[5],
	      parent_fid[6], parent_fid[7]);*/

  /* First go down to parent directory. */
  if ((parent_node = get_scdnode(parent_fid)) == NULL) {
    arla_warnx (ADEBERROR, "cannot find %02x.%02x",
		parent_fid[0], parent_fid[1]);
    return ENOENT;
  }
  if (godown(parent_node) == -1) return ENOENT; /* go down to parent (=record file) */
  /*arla_warnx (ADEBMSG, "scfs_remove name: %s\n", name);*/

  /* Then delete the file in a smartcard. */
  if ((rv = name_to_fid(parent_fid, name, &child_fid))) return rv;
  /*arla_warnx (ADEBMSG, "scfs_remove child: %02x.%02x.%02x.%02x.%02x.%02x.%02x.%02x\n",
	      child_fid[0], child_fid[1],
	      child_fid[2], child_fid[3],
	      child_fid[4], child_fid[5],
	      child_fid[6], child_fid[7]);*/

  rv = scwrite(fd, scconfig.cla_deletefile, 0xe4, 0x00, 0x00, 2, child_fid, &sw1, &sw2);
  arla_warnx (ADEBMSG, "smartcard returns %02x.%02x\n", sw1, sw2);
  if (sw1 == 0x69 || sw2 == 82) return EACCES;
  if (rv == -1) return EIO;
  if (sw1 != 0x90 || sw2 != 00) return EIO;

  /*if ((remove_node(parent_node, ))) return EIO;*/
  /* remove a file entry in the scfs database */
  if ((remove_node(parent_node, child_fid))) return EIO;

  return 0;
}

int scfs_rmdir(xfs_handle parent_fid, char *name)
{
  struct scdnode *parent_node, *removed_node;
  xfs_handle removed_fid;
  int sw1, sw2, rv;
  unsigned char file[2];
  
  assert(parent_fid);
  assert(name);

  /*arla_warnx (ADEBMSG, "scfs_rmdir parent: %02x.%02x.%02x.%02x.%02x.%02x.%02x.%02x\n",
	      parent_fid[0], parent_fid[1],
	      parent_fid[2], parent_fid[3],
	      parent_fid[4], parent_fid[5],
	      parent_fid[6], parent_fid[7]);*/

  if ((parent_node = get_scdnode(parent_fid)) == NULL) {
    arla_warnx (ADEBERROR, "cannot find %02x.%02x",
		parent_fid[0], parent_fid[1]);
    return ENOENT;
  }

  /* Get the removed node */
  if ((rv = name_to_fid(parent_fid, name, &removed_fid))) return rv;
  /*arla_warnx (ADEBMSG, "scfs_rmdir removed_node: %02x.%02x.%02x.%02x.%02x.%02x.%02x.%02x\n",
	      removed_fid[0], removed_fid[1],
	      removed_fid[2], removed_fid[3],
	      removed_fid[4], removed_fid[5],
	      removed_fid[6], removed_fid[7]);*/

  if ((removed_node = get_scdnode(removed_fid)) == NULL) {
    arla_warnx (ADEBERROR, "cannot find %02x.%02x",
		removed_fid[0], removed_fid[1]);
    return ENOENT;
  }
  /* go down to the removed node */
  if (godown(removed_node) == -1) return ENOENT;

  /* now remove meta data (.i) */
  file[0] = DIRENT_FILE0;
  file[1] = DIRENT_FILE1;
  rv = scwrite(fd, scconfig.cla_deletefile, 0xe4, 0x00, 0x00, 2, file, &sw1, &sw2);
  /*arla_warnx (ADEBMSG, "rmdir1: smartcard returns %02x.%02x\n", sw1, sw2);*/
  if (sw1 == 0x6a || sw2 == 0x82) {
    /* no meta data ... don't care */
    goto rmdir2;
  }
  if (sw1 == 0x69 || sw2 == 0x82) return EACCES;
  if (rv == -1 || sw1 != 0x90 || sw2 != 00) return EIO;
  
 rmdir2:
  /* go down to the parent node */
  if (godown(parent_node) == -1) return ENOENT;

  /* remove the removed node */
  rv = scwrite(fd, scconfig.cla_deletefile, 0xe4, 0x00, 0x00, 2, removed_fid, &sw1, &sw2);
  /*arla_warnx (ADEBMSG, "rmdir2: smartcard returns %02x.%02x\n", sw1, sw2);*/
  if (sw1 == 0x69 || sw2 == 0x82) return EACCES;
  if (rv == -1 || sw1 != 0x90 || sw2 != 00) return EIO;

  /* remove a file entry in the scfs database */
  if ((remove_node(parent_node, removed_fid))) return EIO;

  return 0;
}

int name_to_fid(xfs_handle parent_fid, char *name, xfs_handle *child_fid)
{
  int i;
 
  assert(parent_fid);
  assert(name);
  assert(*child_fid);

  if (strlen(name) != FILE_NAME_LEN) return EINVAL;
  (*child_fid)[0] = name[0];
  (*child_fid)[1] = name[1];
  for (i=0; i<FID_LEN-2; i++){
    (*child_fid)[i+2] = parent_fid[i];
  }

  return 0; 
}
 
int remove_node(struct scdnode *parent_node, unsigned char *child_fid)
{
  int i, ith, rv;
  struct scdnode *child_node;
  char *clean_buf;
  
  if ((child_node = get_scdnode(child_fid)) == NULL) {
    arla_warnx (ADEBERROR, "cannot find %02x.%02x",
		child_fid[0], child_fid[1]);
    return ENOENT;
  }
      
  if ((ith = find_child_in_parent(parent_node, child_node)) == -1) return -1;

  arla_warnx (ADEBMSG, "remove_node: parent_num_children=%d, ith=%d",
	      parent_node->num_children, ith); 

  /* remove it from scfs database */
  assert(parent_node->num_children > 1); /* 1 because of .i */
  parent_node->num_children--;
  free(parent_node->child[ith]);
  for (i=ith; i<parent_node->num_children; i++)
    parent_node->child[i] = parent_node->child[i+1];

  for (i=ith; i<parent_node->num_children; i++){
    if ((rv = update_ith_dirent(parent_node, i, &parent_node->child[i]->fileent)))
      return rv;
  }
  
  /* set clean_buf */
  clean_buf = (char *)malloc(sizeof(struct fileent));
  memset (clean_buf, 0, sizeof(struct fileent));
  
  /* now remove #children'th entry in meta data */
  if ((rv = update_ith_dirent(parent_node, parent_node->num_children, (struct fileent *)clean_buf))){
    free(clean_buf);
    return rv;
  }
  
  free(clean_buf);
  return 0; 
}

int find_child_in_parent(struct scdnode *parent_node, struct scdnode *child_node)
{
  int i;

  /* I'm the i'th child of the parent */
  for (i=0; i<parent_node->num_children; i++){
    if (parent_node->child[i] == child_node) break;
  }
  if (i == parent_node->num_children){
    arla_warnx (ADEBERROR, "Cannot find myself in parent??");
    return -1;
  }
  return i;
}

int scfs_rename(xfs_handle old_parent_fid, char *old_name,
		xfs_handle new_parent_fid, char *new_name,
		dev_t t_dev)
{
#ifdef NIXX
  struct scdnode *old_parent_node, *new_parent_node,
    *old_child_node, *new_child_node;
  xfs_handle old_child_fid, new_child_fid;
  int sw1, sw2, rv;
  unsigned char *buf; 
  
  assert(old_parent_fid);
  assert(new_parent_fid);
  assert(old_name);
  assert(new_name);

  /* get parents */
  if ((old_parent_node = get_scdnode(old_parent_fid)) == NULL) {
    arla_warnx (ADEBERROR, "cannot find %02x.%02x",
		old_parent_fid[0], old_parent_fid[1]);
    return ENOENT;
  }

  if ((new_parent_node = get_scdnode(new_parent_fid)) == NULL) {
    arla_warnx (ADEBERROR, "cannot find %02x.%02x",
		new_parent_fid[0], new_parent_fid[1]);
    return ENOENT;
  }

  if ((rv = name_to_fid(old_parent_fid, old_name, &old_child_fid))) return rv;
  if ((rv = name_to_fid(new_parent_fid, new_name, &new_child_fid))) return rv;
  /*arla_warnx (ADEBMSG, "scfs_rename move %02x.%02x.%02x.%02x.%02x.%02x.%02x.%02x to %02x.%02x.%02x.%02x.%02x.%02x.%02x.%02x.\n",
	      old_child_fid[0], old_child_fid[1],
	      old_child_fid[2], old_child_fid[3],
	      old_child_fid[4], old_child_fid[5],
	      old_child_fid[6], old_child_fid[7],
	      new_child_fid[0], new_child_fid[1],
	      new_child_fid[2], new_child_fid[3],
	      new_child_fid[4], new_child_fid[5],
	      new_child_fid[6], new_child_fid[7]);*/

  /* malloc copy buffer */
  if (!(buf = malloc(old_parent_node->fileent.l_size))) {
    /*arla_warnx (ADEBERROR, "cannot malloc %d bytes.\n",
      old_parent_node->fileent.l_size);*/
    free(buf); 
    return ENOMEM;
  } 

  /* read it */
  if (godown(old_parent_node) == -1) return ENOENT; /* go down to parent (=record file) */

  /* get child */
  if ((old_child_node = get_scdnode(old_child_fid)) == NULL) {
    arla_warnx (ADEBERROR, "cannot find %02x.%02x",
		old_child_fid[0], old_child_fid[1]);
    free(buf); 
    return ENOENT;
  }

  /* can't rename directories */
  if (old_child_node->fileent.d_type == SCFS_TYPE_DIR) {
    arla_warnx (ADEBERROR, "cannot rename directory.");
    free(buf); 
    return EISDIR;
  }

  /* read data */
  if ((rv = scfs_read(old_child_fid, 0, old_child_node->fileent.l_size, buf,
		      SCFS_IS_NOT_RECORD, old_parent_fid, t_dev)) < 0) {
    arla_warnx (ADEBERROR, "cannot read %02x.%02x.",
		old_child_fid[0], old_child_fid[1]);
    free(buf); 
    return EIO;
  }

  /* Get the destination.
     If there is not a file in the destination, create one. */
  if ((new_child_node = get_scdnode(new_child_fid)) == NULL) {
    struct scfs_creat_arg ca;
    int i;
    
    ca.path[0] = new_child_fid[0];
    ca.path[1] = new_child_fid[1];
    ca.size = old_child_node->fileent.p_size;
    if ((scfs_create_sub(&ca, new_parent_node))) {
      arla_warnx (ADEBERROR, "cannot create %02x.%02x.",
		  new_child_fid[0], new_child_fid[1]);
      free(buf); 
      return EIO;
    }
    /*
    new_child_node = old_child_node;

    if ((i = find_child_in_parent(new_parent_node, new_child_node)) == -1) {
      arla_warnx (ADEBERROR, "Can't find??");
      return -1;
    }
    fidcpy(new_child_node->fileent.fid, new_child_fid);
    if ((rv = update_ith_dirent(new_parent_node, i, &new_child_node->fileent))) return rv;
    */

    /* get new_child_node */
    if ((new_child_node = get_scdnode(new_child_fid)) == NULL) {
      arla_warnx (ADEBERROR, "How come no child?!",
		  new_child_fid[0], new_child_fid[1]);
      free(buf); 
      return EIO;
    }
    
    /* write data */
    if ((rv = scfs_write(new_child_fid, 0, old_child_node->fileent.l_size, buf,
			 SCFS_IS_NOT_RECORD, new_parent_fid, t_dev)) < 0) {
      arla_warnx (ADEBERROR, "cannot write %02x.%02x.",
		  new_child_fid[0], new_child_fid[1]);
      free(buf); 
      return EIO;
    }
    
    /*
      TODO: 
      DONE get new_child_node,
      DONE copy data, 
      remove old_child_node's entry in meta data, 
      remove old_child_node,
      update meta data of new_parent.
     */
  }
  
#endif /* NIXX */
  return 0;
}

#ifdef NIXX
int scfs_restart()
{
  int rv;
  
  arla_warnx (ADEBMSG, "card removed");
  
  /* store state */
  /*store_state ();*/

  /* unmount the filesystem */
  arla_warnx (ADEBMSG, "Now unmount the filesystem!");

  /*
  if (unmount("/smartcard", 0))
    perror("scfsd: cannot unmount smartcard");
  */

  xfs_message_unmount_force();

  rv = scfs_unmount();
  if (rv) {
    perror("scfsd: cannot unmount smartcard");
    return; 
  }

  free(buf); 
  return 0;

  /*
  arla_warnx (ADEBMSG, "And initialize it again.");

  return scfs_init(2 - 1);
  */
}
#endif
