#include "cfs_cache.h"
#include <signal.h>
#include <unistd.h>

static struct cfs_cache_str cfs_cache = {NULL, 0, 0, NULL, 0}; 

/* return 1 if the requested file "fid" is in cache.
   return 0 otherwise */
static int cache_query (cfs_fileid *fid)
{
#ifdef DEBUGXX
  printf ("cache_query (%x, %x)\n", fid, cfs_cache.fid);
#endif /* DEBUGXX */
  if (fid == cfs_cache.fid) return 1;
  else return 0; 
}

/* read the requested file "fid" into cache and decrypt it */
int cache_fetch (cfs_fileid *fid, int fd, int *err)
{
  int rv;
  sigset_t newmask, oldmask; 

  sigemptyset (&newmask);
  sigaddset (&newmask, SIGALRM);
  sigprocmask (SIG_BLOCK, &newmask, &oldmask); 

  rv = cache_fetch_int (fid, fd, err);

  sigprocmask (SIG_SETMASK, &oldmask, NULL); 
  
  return rv; 
}

int cache_fetch_int (cfs_fileid *fid, int fd, int *err)
{
  int rv, i;
  struct stat tmp_stat;
  unsigned long entire_len;
  int crypt_len; /* fixed_len cut down to 8 multiple */
  unsigned char sc_key[DES_LEN];
  
  if (fid == NULL) return 0;

  /* make sure it's not in the cache, first. */
  rv = cache_query (fid);
  if (rv == 1) {
#ifdef DEBUG
    printf ("cache_fetch: file (%s) is in cache.  NO need to fetch.  size=%d\n",
	    fid->name, cfs_cache.cache_size);
#endif /* DEBUG */
    return cfs_cache.cache_size;
  }
#ifdef DEBUG
  printf ("cache_fetch: file (%s) is not in cache.  need to fetch.\n",
	  fid->name);
#endif /* DEBUG */

  /* if something is in, purge it */
  if (cfs_cache.fid != NULL) {
    rv = cache_purge (err);
    if (rv != 0) 
      goto err_out; 
  }
  
  /* stat the file */
  rv = fstat (fd, &tmp_stat);
  if (rv != 0) {
    perror ("cache_fetch: fstat()");
    *err = cfsno(errno); 
    goto err_out; 
  }

  entire_len = tmp_stat.st_size;

  if (entire_len < 0) {
    *err = NFSERR_NOENT; 
    goto err_out; 
  }

  /* if the file len is 0 (probably it has just been created),
     no need to read */
  
  if (entire_len == 0) {
    cfs_cache.data = NULL;
    cfs_cache.cache_size = entire_len;
    cfs_cache.fid = fid;

    return 0; 
  }

  /* allocate the buffer */
  cfs_cache.data = (unsigned char *) malloc (entire_len + CFSBLOCK);
  if (cfs_cache.data == NULL) {
    *err = NFSERR_NOSPC; 
    goto err_out; 
  }
  bzero (cfs_cache.data, entire_len);

  /* somehow, this lseek doesn't work on OpenBSD.  So skip. */
#ifndef __OpenBSD__    
  rv = lseek (fd, 0, SEEK_SET);
  if (rv < 0) {
    perror ("lseek in cache_fetch_int");
    *err = cfsno(errno);
    goto err_out; 
  }
#endif /* !__OpenBSD__ */
  
  rv = read (fd, cfs_cache.data, entire_len);
  if (rv < 0) {
    perror ("reading entire file");
    *err = cfsno(errno);
    goto err_out; 
  }

  /* TODO: decrypt the entire file */
  crypt_len = entire_len - (entire_len & (CFSBLOCK - 1));

  generate_key (1, fid->vect, sc_key); /* check key cache */
#ifdef DEBUG
  printf ("read: decrypt with key : ");
  for (i = 0 ; i < DES_LEN; i ++) printf ("%02x ", sc_key[i]);
  printf ("\n");
#endif /* DEBUG */
  
  smartcard_encrypt (cfs_cache.data, cfs_cache.data,
		     sc_key, crypt_len, 1);

  cfs_cache.cache_size = entire_len;
  cfs_cache.fid = fid;

  return entire_len;

 err_out:
  return -1;
}

/* copy data from cache into buf
   returns size of the data read */
int cache_read (unsigned char *buf, int offset, int len, int *err) 
{
  int rv;
  int copylen; 

  copylen = cfs_cache.cache_size - offset < len ?
    cfs_cache.cache_size - offset : len;

  bcopy (cfs_cache.data + offset, buf, copylen);

  return copylen;
}

/* copy data from buf into cache

   WORK:
   realloc the buffer
   bcopy
   set written bit
   set length
*/
int cache_write (unsigned char *buf, int offset, int len, int *err) 
{
  int rv, i;

  /* realloc the buffer */
  if (offset + len > cfs_cache.cache_size) {
    cfs_cache.data = (unsigned char *)
      realloc (cfs_cache.data, offset + len + CFSBLOCK);
    if (cfs_cache.data == NULL) {
      *err = NFSERR_NOSPC; 
      return -1; 
    }
  }

#ifdef DEBUG
  printf ("cache_write: (cache_size %d, off+len %d, ",
	  cfs_cache.cache_size, offset + len);
#endif /* DEBUG */

  bcopy (buf, cfs_cache.data + offset, len);
  cfs_cache.cache_size = (cfs_cache.cache_size > offset + len) ?
    cfs_cache.cache_size : (offset + len);

#ifdef DEBUG
  printf ("->cache_size %d)\n",
	  cfs_cache.cache_size);
#endif /* DEBUG */


  cfs_cache.written = 1;

  return 0; 
}

/* write back the cache content, and then remote the entry from cache */
int cache_purge (int *err)
{
  int rv;

  /* if cache is empty, there's nothing to do. */
  if (cfs_cache.fid == NULL) return 0;

  rv = cache_write_back (err);
  if (rv != 0) return rv; 

  if (cfs_cache.data)
    free (cfs_cache.data);
  cfs_cache.fid = NULL;

  return 0; 
}

extern void grimreap(); /* in cfs.c */

int cache_write_back (int *err)
{
  int rv;
  sigset_t newmask, oldmask; 

  sigemptyset (&newmask);
  sigaddset (&newmask, SIGALRM);
  sigprocmask (SIG_BLOCK, &newmask, &oldmask); 

  rv = cache_write_back_int(err);

  sigprocmask (SIG_SETMASK, &oldmask, NULL); 
  
  return rv; 
}
     
/* write back the cache content */
int cache_write_back_int (int *err)
{
  int rv, i; 
  int fd;
  int fixed_len; /* real len + 8 if the real len is not divided by 8 */
  int crypt_len; /* fixed_len cut down to 8 multiple */
  unsigned char *tmp_buf;
  struct stat tmp_stat;
  unsigned char sc_key[DES_LEN];

  /* if cache is empty, there's nothing to do. */
  if (cfs_cache.fid == NULL) return 0;

#ifdef DEBUGXX
  fprintf (stderr, "cache_write_back (%s)\n", cfs_cache.fid->name);
#endif /* DEBUG */
    
  if (cfs_cache.written) {
    cfs_cache.written = 0;

    /* TODO: encrypt */
    if (isbndry (cfs_cache.cache_size))
      fixed_len = cfs_cache.cache_size;
    else
      fixed_len = cfs_cache.cache_size + CFSBLOCK;
    
    crypt_len = fixed_len - (fixed_len & (CFSBLOCK - 1));


    fd = open (cfs_cache.fid->name, O_WRONLY);
    if (fd < 0) {
      if (errno == ENOENT) {
	/* file is already removed */
	close (fd);
	return 0;
      }
      perror ("open in cfs_purge()");
      *err = cfsno(errno);
      goto write_back_err; 
    }

    /* somehow, this lseek doesn't work on OpenBSD.  So skip. */
#ifndef __OpenBSD__    
    rv = lseek (fd, 0, SEEK_SET);
    if (rv < 0) {
      perror ("lseek in cfs_purge()");
      *err = cfsno(errno);
      goto write_back_err; 
    }
#endif /* !__OpenBSD__ */
    
    /* stat the file */
    rv = fstat (fd, &tmp_stat);
    if (rv != 0) {
      perror ("cache_write_back: fstat()");
      *err = cfsno(errno);
      goto write_back_err; 
    }
       
    rv = smartcard_newseed (cfs_cache.fid, (unsigned long)tmp_stat.st_ino,
			    &err);
    if (rv != 0) {
      printf ("smartcard_newseed returns error\n");
      goto write_back_err;
    }

    tmp_buf = (unsigned char *) malloc (crypt_len);

    generate_key (0, cfs_cache.fid->vect, sc_key); /* don't try key cache */
    
    smartcard_encrypt (cfs_cache.data, tmp_buf,
		       sc_key, crypt_len, 0);
    rv = write (fd, tmp_buf, fixed_len);
    free (tmp_buf);
    if (rv < 0) {
      perror ("writing entire file in cfs_purge()");
      *err = cfsno(errno);
      goto write_back_err; 
    } else {
    }

    fflush (NULL);
    close (fd);
#ifdef DEBUG
    printf ("write back done (crypt_len=%d, fixed_len=%d)",
	    crypt_len, fixed_len);
#endif /* DEBUG */
    
  }
  
#ifdef DEBUGXX
  printf (stderr, "cache_write_back (%s) return success\n", cfs_cache.fid->name);
#endif /* DEBUG */
  return 0;

 write_back_err:
  close (fd);
  fprintf (stderr, "cache_write_back (%s) return error\n", cfs_cache.fid->name);
  /*cfs_cache.written = 1; /* why 1 ? */
  return -1; 
}

/* remove entry fid, if this is in cache. */
int cache_remove (cfs_fileid *fid) 
{
  printf ("cache_remove (fid->fileid %d, cc.fid->fileid %d\n",
	  fid->fileid, cfs_cache.fid->fileid);
  if (fid->fileid == cfs_cache.fid->fileid) {
    printf ("free!\n");
    free (cfs_cache.data);
    cfs_cache.fid = NULL;
  } else 
    printf ("n/n\n");
  
  return 0; 
}
