/*
 *
 * NAME:
 *      ctapi.c
 *      Copyright (C) 1998 Carlos Prados
 *
 * DESCRIPTION:
 *	Implementation of CT-API for Towitoko terminals
 *
 * AUTHOR:
 *      Carlos Prados Bocos, October 1998. 
 *	cprados@atos-ods.es
 *
 * LICENSE:
 *      See file COPYING
 *
 */

#include <stdio.h>
#include "defines.h"
#include "ctapi.h"
#include "serial.h"
#include "towitoko.h"
#include "atr.h"
#include "ctbcs.h"


#define MAX(a,b)	((a) > (b) ? a : b)

/*
 * Internal functions to perform specific command types
 */

static int
CTBCS_Command(int lc, BYTE* cmd, int* lr, BYTE* rsp);

static int
T0_Command(int lc, BYTE* cmd, int* lr, BYTE*rsp);

static int
T1_Command(int lc, BYTE* cmd, int* lr, BYTE*rsp);

/*
 * Answer to reset of the currently inserted card
 */

static int Atr_Len=0;
static BYTE Atr[ATR_MAX_SIZE];


/* 
 * Initializes the port on which the reader resides 
 * Parameters:
 *	Ctn: logical cardterminal number
 *	pn: serial port number
 * Returns:
 *	OK or CT-API error code.
 */
  
int  CT_init( unsigned int Ctn,  unsigned int pn ) {

  BYTE type;

  /* Open serial port */ 
  switch( pn ) {
    
    case PORT_COM1:

      #ifdef OS_SOLARIS
      if(!IO_InitializePort(9600,8,'E',"/dev/cua/a"))
        return(ERR_MEMORY);
      #endif

      #ifndef OS_SOLARIS
      if(!IO_InitializePort(9600,8,'E',"/dev/ttyS0"))
        return(ERR_MEMORY);
      #endif

      break;

    case PORT_COM2:

      #ifdef OS_SOLARIS
      if(!IO_InitializePort(9600,8,'E',"/dev/cua/b"))
        return(ERR_MEMORY);
      #endif

      #ifndef OS_SOLARIS
      if(!IO_InitializePort(9600,8,'E',"/dev/ttyS1"))
        return(ERR_MEMORY);
      #endif

      break;

    case PORT_COM3:

      #ifdef OS_SOLARIS
      if(!IO_InitializePort(9600,8,'E',"/dev/cua/c"))
        return(ERR_MEMORY);
      #endif

      #ifndef OS_SOLARIS
      if(!IO_InitializePort(9600,8,'E',"/dev/ttyS2"))
        return(ERR_MEMORY);
      #endif

      break;

    case PORT_COM4:

      #ifdef OS_SOLARIS
      if(!IO_InitializePort(9600,8,'E',"/dev/cua/d"))
        return(ERR_MEMORY);
      #endif

      #ifndef OS_SOLARIS
      if(!IO_InitializePort(9600,8,'E',"/dev/ttyS3"))
        return(ERR_MEMORY);
      #endif

      break;

    default:

      if(!IO_InitializePort(9600,8,'E',"/dev/smartcard"))
        return(ERR_MEMORY);

      break;
  }

  /* Determine reader type */
  if(!Towitoko_GetReaderType(&type))
    return(ERR_TRANS);

  #ifdef DEBUG
  printf("---- Reader type: 0x%X\n",type);
  #endif

  /* Uncomment the following if you need
   * KartenZwerg support. New Chipdrives that
   * share the same type identification 
   * will not work with this
   * /
//  if(type==TOWITOKO_KARTENZWERG) {
//
//    /* Set Serial port parity */
//    IO_UpdateParity('O');
//
//    /* Set number of stop bits */
//    IO_UpdateStopBits(1);
//
//    /* Initialize reader parity */   
//    if(!Towitoko_SetTerminalParity('O'))
//      return(ERR_TRANS);
//  }

  return(OK);  
}

/* 
 * Closes the port in which the reader resides 
 * Parameter:
 *      Ctn: logical cardterminal number
 * Returns:
 *      OK or CT-API error code.
 */

int  CT_close( unsigned int Ctn ) {

  /* Close serial port */  
  if(IO_Close()) 
    return OK;
  else 
    return ERR_CT;
}            

/* 
 * Sends/Receives Data to/from the Reader
 * Parameters:
 *      ctn: logical cardterminal number.
 *	dad: destination address.
 *	sad: source address.
 *	lc: cmd length in bytes.
 *	cmd: ICC cmd or CT cmd.
 *	lr: pasing of the max. buffer size of the rsp and 
 *	return of the actual length of the rsp.
 * 	rsp: rsp to the cmd.
 * Returns:
 *      OK or CT-API error code.
 */ 

int  CT_data( unsigned int ctn, unsigned char *dad, unsigned char *sad,
              unsigned int  lc, unsigned char *cmd, unsigned int  *lr,
              unsigned char *rsp ) {

  BYTE status;
  BYTE protocol;

  /* Command goes to the reader */ 
  if (*dad == 1) {             

    /* Response goes to the host */
    *sad = 1;                   
    *dad = 2;

    if(cmd[0]==CTBCS_CLA) {                  

      /* Perform a CT-BCS cmd */
      return(CTBCS_Command(lc,cmd,lr,rsp));

   }  else {

      if(*lr<CTBCS_MIN_RESPONSE_SIZE) {
        *lr=0;
        return(ERR_MEMORY);
      }

      *lr=CTBCS_MIN_RESPONSE_SIZE;
      rsp[0]=CTBCS_SW1_WRONG_CLA;
      rsp[1]=CTBCS_SW2_WRONG_CLA;
      return(ERR_INVALID);
    }
  } 

  /* Command goes to the card */ 
  else if(*dad == 0) {  
    
    /* Response goes to host */
    *sad = 0; 
    *dad = 2;

    /* See if card has changed */
    if(!Towitoko_GetRdrStatus(&status)) {

      /* No card detected: LED off */
      Towitoko_SetLED(TOWITOKO_LED_OFF);

      Atr_Len=*lr=0;
      return(ERR_TRANS);
    }
    if(!TOWITOKO_CARD(status)) {
      Atr_Len=*lr=0;
      return(ERR_TRANS);
    }

    /* Card shouldn't de removed: LED red */
    Towitoko_SetLED(TOWITOKO_LED_RED);

    /* If card has changed or ATR hasn't been stored yet */
    if((status==TOWITOKO_CARD_CHANGE) || (Atr_Len==0)) {

      /* Activate card */
      if(!Towitoko_ActivateCard()) {
        Atr_Len=*lr=0;
        return(ERR_TRANS);
      }

      /* Reset card and store it's ATR*/
      if(!Towitoko_ResetCard(Atr,&Atr_Len)) {
        Atr_Len=*lr=0;
        return(ERR_TRANS);
      }
    }

    /* Determine card protocol */
    if(ATR_Protocol(Atr,Atr_Len,&protocol)==ATR_MALFORMED) {
      Atr_Len=*lr=0;
      return(ERR_INVALID);
    }

    /* Protocol is T=0 */
    if(protocol==ATR_PROTOCOL_T0) {

      /* Sends a T=O TPDU and gets Card TPDU response */
      return(T0_Command(lc,cmd,lr,rsp));
  
   } else if(protocol==ATR_PROTOCOL_T1) 

      /* Sends a T=O TPDU and gets Card TPDU response */
      return(T1_Command(lc,cmd,lr,rsp));

    else {
      *lr=0;
      return(ERR_INVALID);
    }
  }

  /* Invalid SAD/DAD Address */

  *lr=0;
  return(ERR_INVALID);  
}

/* 
 * Performs a CT-BCS command and returns response
 * Parameters: 
 *	lc: length of the command.
 *	cmd: command to send to the card.
 *      lr: length of the response.
 *	rsp: returned response of the card.
 * Returns:
 *	CT-API operation status code.
 */

static int CTBCS_Command(int lc, BYTE* cmd, int* lr, BYTE* rsp) {

  BYTE status;
  BYTE hist[ATR_MAX_SIZE];
  int lhist;

  /* Check cmd is a valid CTBCS command*/
  if(lc < CTBCS_MIN_COMMAND_SIZE) {

    if(*lr<CTBCS_MIN_RESPONSE_SIZE) {
      *lr=0;
      return(ERR_MEMORY);
    }

    *lr=2;
    rsp[0]=CTBCS_SW1_WRONG_LENGTH;
    rsp[1]=CTBCS_SW2_WRONG_LENGTH;
    return(ERR_INVALID);
  }
  if(cmd[0]!=CTBCS_CLA) {

    if(*lr<CTBCS_MIN_RESPONSE_SIZE) {
      *lr=0;
      return(ERR_MEMORY);
    }

    *lr=CTBCS_MIN_RESPONSE_SIZE;
    rsp[0]=CTBCS_SW1_WRONG_CLA;
    rsp[1]=CTBCS_SW2_WRONG_CLA;
    return(ERR_INVALID);
  }

  /* Request ICC: Turn on the reader/card */
  if(cmd[1] == CTBCS_INS_REQUEST) {

    /* See if the card is inserted */
    if(!Towitoko_GetRdrStatus(&status)) {
      Atr_Len=0;
      *lr=0;      
      return(ERR_TRANS);      
    }
    if(!TOWITOKO_CARD(status)) {

      /* No card present */
      Atr_Len=0;

      if(*lr<CTBCS_MIN_RESPONSE_SIZE) {
        *lr=0;
        return(ERR_MEMORY);
      }
      *lr = 2;
      rsp[0] = CTBCS_SW1_REQUEST_NO_CARD;
      rsp[1] = CTBCS_SW2_REQUEST_NO_CARD;

      /* No card detected: LED off */
      Towitoko_SetLED(TOWITOKO_LED_OFF);

      return(OK);
    }

    /* Card shouldn't be removed: LED RED */
    Towitoko_SetLED(TOWITOKO_LED_RED);

    /* Activate card */
    if(!Towitoko_ActivateCard()) {
      Atr_Len=0;
      *lr=0;      
      return(ERR_TRANS);
    }
   
    /* Return compete ATR of the card */ 
    if((lc>2)&&(cmd[3]==CTBCS_P2_REQUEST_GET_ATR)) {

      /* Reset card and store it's ATR*/
      if(!Towitoko_ResetCard(Atr,&Atr_Len)) {
        Atr_Len=0;
        *lr=0;      
        return(ERR_TRANS);
      }
      if(*lr<Atr_Len+2) {
        Atr_Len=0;
        *lr=0;
        return(ERR_MEMORY);
      }
      *lr=Atr_Len+2;
      memcpy(rsp,Atr,Atr_Len);
      rsp[Atr_Len]=CTBCS_SW1_REQUEST_ASYNC_OK;
      rsp[Atr_Len+1]=CTBCS_SW2_REQUEST_ASYNC_OK;

      /* Card can be safely removed: LED green */
      Towitoko_SetLED(TOWITOKO_LED_GREEN);

      return(OK);
    }

    /* Return historical bytes of the ATR */
    else if((lc>2)&&(cmd[3]==CTBCS_P2_REQUEST_GET_HIST)) {

      /* Reset card and store it's ATR*/
      if(!Towitoko_ResetCard(Atr,&Atr_Len)) {
        Atr_Len=0;
        *lr=0;      
        return(ERR_TRANS);
      }
     
      ATR_HistoricalBytes(Atr,Atr_Len,hist,&lhist);

      if(*lr<lhist+2) {
        Atr_Len=*lr=0;
        return(ERR_MEMORY);
      }   
      
      *lr=lhist+2;
      rsp[lhist]=CTBCS_SW1_REQUEST_ASYNC_OK;
      rsp[lhist+1]=CTBCS_SW2_REQUEST_ASYNC_OK;

      /* Card can be safely removed: LED green */
      Towitoko_SetLED(TOWITOKO_LED_GREEN);

      return(OK);    
    }

    /* Return no data bytes */
    else {
    
      /* Synchronous ICC presented */
      if(*lr<CTBCS_MIN_RESPONSE_SIZE) {
        *lr=0;
        return(ERR_MEMORY);
      }
      *lr = 2;
      rsp[0]=CTBCS_SW1_REQUEST_ASYNC_OK;
      rsp[1]=CTBCS_SW2_REQUEST_ASYNC_OK;

      /* Card can be safely removed: LED green */
      Towitoko_SetLED(TOWITOKO_LED_GREEN);

      return(OK);
    }
  }

  /* Reset CT: Resets the Card/Terminal and returns Atr */
  else if(cmd[1] == CTBCS_INS_RESET) {

    /* See if the card is inserted */
    if(!Towitoko_GetRdrStatus(&status)) {
      Atr_Len=0;
      *lr=0;      
      return(ERR_TRANS);      
    }
    if(!TOWITOKO_CARD(status)) {

      /* No card present */
      Atr_Len=0;
      if(*lr<CTBCS_MIN_RESPONSE_SIZE) {
        *lr=0;
        return(ERR_MEMORY);
      }
      *lr = 2;
      rsp[0] = CTBCS_SW1_RESET_ERROR;
      rsp[1] = CTBCS_SW2_RESET_ERROR;

      /* No card detected: LED off */
      Towitoko_SetLED(TOWITOKO_LED_OFF);

      return(OK);
    }

    /* Card shouldn't be removed: LED RED */
    Towitoko_SetLED(TOWITOKO_LED_RED);

    /* Activate card */
    if(!Towitoko_ActivateCard()) {
      Atr_Len=0;
      *lr=0;      
      return(ERR_TRANS);
    }

    /* Reset card and store it's ATR*/
    if(!Towitoko_ResetCard(Atr,&Atr_Len)) {
      Atr_Len=0;
      *lr=0;      
      return(ERR_TRANS);
    }
  
    /* Return complete ATR of the card */  
    if((lc>2)&&(cmd[3]==CTBCS_P2_RESET_GET_ATR)) {

      if(*lr<Atr_Len+2) {
        Atr_Len=0;
        *lr=0;
        return(ERR_MEMORY);
      }
      *lr=Atr_Len+2;
      memcpy(rsp,Atr,Atr_Len);
      rsp[Atr_Len]=CTBCS_SW1_REQUEST_ASYNC_OK;
      rsp[Atr_Len+1]=CTBCS_SW2_REQUEST_ASYNC_OK;

      /* Card can be safely removed: LED green */
      Towitoko_SetLED(TOWITOKO_LED_GREEN);

      return(OK);
    }
 
    /* Return only historical bytes of the ATR */
    else if((lc>2)&&(cmd[3]==CTBCS_P2_RESET_GET_HIST)) {
     
      ATR_HistoricalBytes(Atr,Atr_Len,hist,&lhist);

      if(*lr<lhist+2) {
        Atr_Len=*lr=0;
        return(ERR_MEMORY);
      }   
      
      *lr=lhist+2;
      rsp[lhist]=CTBCS_SW1_RESET_ASYNC_OK;
      rsp[lhist+1]=CTBCS_SW2_RESET_ASYNC_OK;

      /* Card can be safely removed: LED green */
      Towitoko_SetLED(TOWITOKO_LED_GREEN);

      return(OK);    
    }

    /* Return no data bytes */
    else {
    
      /* Synchronous ICC presented */
      if(*lr<CTBCS_MIN_RESPONSE_SIZE) {
        *lr=0;
        return(ERR_MEMORY);
      }
      *lr = 2;
      rsp[0]=CTBCS_SW1_REQUEST_ASYNC_OK;
      rsp[1]=CTBCS_SW2_REQUEST_ASYNC_OK;

      /* Card can be safely removed: LED green */
      Towitoko_SetLED(TOWITOKO_LED_GREEN);

      return(OK);
    }
  }

  /* Get Status: Gets reader status */
  else if(cmd[1] == CTBCS_INS_STATUS) {

    /* See if the card is inserted */
    if(!Towitoko_GetRdrStatus(&status)) {
      Atr_Len=*lr=0;
      return(ERR_TRANS);
    }

    if(TOWITOKO_CARD(status)) {

      /* ICC inserted, Data = CTBCS_DATA_STATUS_CARD */
      if(*lr<3) {
        *lr=0;
        return(ERR_MEMORY);
      }
      *lr = 3;
      rsp[0] = CTBCS_DATA_STATUS_CARD;
      rsp[1] = CTBCS_SW1_OK;
      rsp[2] = CTBCS_SW2_OK;

      /* Card detected: LED green */
      Towitoko_SetLED(TOWITOKO_LED_GREEN);

      /* If card has changed */
      if(status==TOWITOKO_CARD_CHANGE) {

        /* Clear previously stored ATR */
        Atr_Len=0;

        /* Activate card */
        if(!Towitoko_ActivateCard()) {
          Atr_Len=*lr=0;
          return(ERR_TRANS);
        }
      }
    }
    else {

      /* No ICC inserted: Data = CTBCS_DATA_STATUS_NOCARD */
      if(*lr<3) {
        *lr=0;
        return(ERR_MEMORY);
      }
      *lr = 3;
      rsp[0] = CTBCS_DATA_STATUS_NOCARD;
      rsp[1] = CTBCS_SW1_OK;
      rsp[2] = CTBCS_SW2_OK;

      /* No card detected: LED off */
      Towitoko_SetLED(TOWITOKO_LED_OFF);
    }

    return(OK);
  }

  /* Eject ICC: Deactivates Reader  */
  else if(cmd[1] == CTBCS_INS_EJECT) {

    /* Turn off LED */
    Towitoko_SetLED(TOWITOKO_LED_OFF);

    /* See if the card is inserted */
    if(!Towitoko_GetRdrStatus(&status)) {
      Atr_Len=*lr=0;
      return(ERR_TRANS);
    }
    if(TOWITOKO_CARD(status))

      /* Deactivates card */
      if(!Towitoko_DeactivateCard()) {
        *lr=0;
        return(ERR_TRANS);
      }

    /* Command successful */
    if(*lr<CTBCS_MIN_RESPONSE_SIZE) {
      *lr=0;
      return(ERR_MEMORY);
    }
    *lr = 2;
    rsp[0] = CTBCS_SW1_OK;
    rsp[1] = CTBCS_SW2_OK;

    /* Clears currently stored ATR */
    Atr_Len=0;

    return(OK);
  }

  /* Other command */
  if(*lr<CTBCS_MIN_RESPONSE_SIZE) {
    *lr=0;
    return(ERR_MEMORY);
  }
  *lr = 2;
  rsp[0] = CTBCS_SW1_WRONG_INS;
  rsp[1] = CTBCS_SW2_WRONG_INS;
  return(ERR_INVALID);
}

/*
 * Performs a T0 command and returns Card response. 
 * Parameters: 
 *	lc: length of the cmd.
 *	cmd: T0 cmd to send to the card.
 *      lr: length of the rsp.
 *	rsp: returned rsp of the card.
 * Returns:
 *	CT-API operation status code.
 */

static int T0_Command(int lc, BYTE* cmd, int* lr, BYTE* rsp) {

	BYTE status;
	int null_bytes=0;
	int sent_bytes=0;
	int recv_bytes=0;

	// Check cmd buffer size
	if((cmd == NULL) || (lc < 5)) {
		*lr=0;
		return(ERR_INVALID);
	}

  	// Check rsp buffer size
	if ( (rsp == NULL && *lr != 0) || (lc==5) && *lr<(cmd[4]) ) {
		*lr=0;
		return(ERR_MEMORY);
	}

	// Send header bytes 
	if(!Towitoko_CardCommand(cmd,sent_bytes=5)) {
		*lr=0;
		return(ERR_TRANS);
	}

	while(TRUE) {

		// Read one procedure byte 
		if(!IO_Read(1,&status)) {
			*lr=0;
			return(ERR_TRANS);
		}


		// If NULL byte (0x60) is received
		if(status==0x60) {

			if(++null_bytes>=200) {
				*lr=0;
				return(ERR_TRANS);
			}

			continue;
		}

		// If SW1 byte is received
		else if(((status&0xF0)==0x60) || ((status&0xF0)==0x90)) {

			// Copy SW1 byte to rsp
			if(*lr<recv_bytes+2) {
				*lr=0;
				return(ERR_MEMORY);
			}
			rsp[recv_bytes++]=status;

			// Read SW2 byte 
			if(!IO_Read(1,&status)) {
				*lr=0;
				return(ERR_TRANS);
			}
			
			// Copy SW2 byte to rsp
			rsp[recv_bytes++]=status;
			*lr=recv_bytes;

			break;
		}


		// If ACK byte is received
		// else if((status&0xFE)==(cmd[1]&0xFE)) {
		
		/**********************************************************
		 * THIS IS FOR CRYPTOFLEX:
		 * Check only 4 less significant bytes of ACK
		 **********************************************************/

		else if ( (status&0x0E) == (cmd[1]&0x0E) ) {

			// Case 3 command
			if(lc>5) {
				// Send all data bytes
				if(!Towitoko_CardCommand(cmd + sent_bytes,MAX(lc - sent_bytes,0))) {
					*lr=0;
					return(ERR_TRANS);
				}
				sent_bytes = lc;
				continue;
			}
			
			// Case 2 command
			else {

				// Check we don't overflow rsp buffer
				if ( recv_bytes >= (*lr) ) {
					*lr=0;
					return(ERR_MEMORY);
				}

				// Read all data bytes
				if( !IO_Read(MAX(cmd[4] - recv_bytes, 0), rsp + recv_bytes )) { 
					*lr=0;
					return(ERR_TRANS);
				}
				recv_bytes = cmd[4];
				continue;
			}
		}

		// If ~ACK is received
		// else if ( (status&0xFE) == ((~cmd[1])&0xFE) ) {

		/**********************************************************
		 * THIS IS FOR CRYPTOFLEX:
		 * Check only 4 less significant bytes of ACK
		 **********************************************************/

		else if ( (status&0x0E) == ((~cmd[1])&0x0E) ) {

			// Case 3 command
			if (lc > 5) {

				// Send next data byte
				if ( !Towitoko_CardCommand(cmd + sent_bytes, 1) ) {
					*lr=0;
					return(ERR_TRANS);
				}
				sent_bytes++;
				printf("sent_bytes = %d \n", sent_bytes);
				continue;
			}
			
			// Case 2 command
			else {
				// Check we don't overflow rsp buffer
				if ( recv_bytes >= (*lr) ) {
					*lr=0;
					return(ERR_MEMORY);
				}

				// Read next data byte
				if(!IO_Read(1 , rsp + recv_bytes)) { 
					*lr=0;
					return(ERR_TRANS);
				}
				recv_bytes++;
				continue;
			}
		}


		// If anything else is received 
		else {
			*lr=0;
			return(ERR_TRANS);
		}    
	}


	/* Card can be safely removed: LED green */
	Towitoko_SetLED(TOWITOKO_LED_GREEN);

	return(OK);
}

/*
 * Performs a T1 command and returns Card response
 * Parameters:
 *      lc: length of the cmd.
 *      cmd: T1 command to send to the card.
 *      lr: length of the response.
 *      rsp: returned response of the card.
 * Returns:
 *      CT-API operation status code.
 */

static int T1_Command(int lc, BYTE* cmd, int* lr, BYTE*rsp) {

  /*
   * CODE FOR T=1 PROTOCOL SMARTCARDS
   * Not implemented
   */
  return(OK);
}
