/* $Id: $ */

/*
copyright 1999
the regents of the university of michigan
all rights reserved

permission is granted to use, copy, create derivative works 
and redistribute this software and such derivative works 
for any purpose, so long as the name of the university of 
michigan is not used in any advertising or publicity 
pertaining to the use or distribution of this software 
without specific, written prior authorization.  if the 
above copyright notice or any other identification of the 
university of michigan is included in any copy of any 
portion of this software, then the disclaimer below must 
also be included.

this software is provided as is, without representation 
from the university of michigan as to its fitness for any 
purpose, and without warranty by the university of 
michigan of any kind, either express or implied, including 
without limitation the implied warranties of 
merchantability and fitness for a particular purpose. the 
regents of the university of michigan shall not be liable 
for any damages, including special, indirect, incidental, or 
consequential damages, with respect to any claim arising 
out of or in connection with the use of the software, even 
if it has been or is hereafter advised of the possibility of 
such damages.
*/

/*
 * Palm serial IO interface for David Corcoran's ct-api
 * Written by Jim Rees, University of Michigan, February 1999
 */

#include <Common.h>
#include <System/SysAll.h>
#include <System/SerialMgr.h>
#include <System/Unix/unix_stdlib.h>
#include <System/Unix/unix_string.h>
#include <stdio.h>

static IO_UpdateParityFlags(char parity);

static UInt refNum;
static int IO_isopen;
static int IO_ttyn;
static SerSettingsType IO_settings;
int IO_blocktime; /* milliseconds */

int
IO_InitializePort(int bps, int bpc, char parity, char *ttyname)
{
    Err err;

    if (IO_isopen)
	return 1;

    if (err = SysLibFind("Serial Library", &refNum))
	goto bad;

    if (ttyname && *ttyname)
	IO_ttyn = ttyname[strlen(ttyname)-1] - '0';
    else
	IO_ttyn = 0;

    if (err = SerOpen(refNum, (UInt) IO_ttyn, (ULong) bps))
	goto bad;
    IO_isopen = 1;

    if (err = SerGetSettings(refNum, &IO_settings))
	goto bad;

    IO_settings.flags = serSettingsFlagStopBits2;
    IO_UpdateParityFlags(parity);
    IO_settings.flags |= ((bpc == 7) ? serSettingsFlagBitsPerChar7 : serSettingsFlagBitsPerChar8);
    IO_blocktime = 250;

    if (err = SerSetSettings(refNum, &IO_settings))
	goto bad;

    IO_FlushBuffer();
#ifdef DEBUG
    printf("open ok\n");
#endif
    return 1;

 bad:
    if (err == serErrAlreadyOpen || IO_isopen) {
	SerClose(refNum);
	IO_isopen = 0;
    }
    return 0;
}

int
IO_Read(int n, unsigned char *buf)
{
    Long ticks;
    Err err;

    if (IO_blocktime <= 0)
	ticks = IO_blocktime;
    else
	ticks = ((long) IO_blocktime * SysTicksPerSecond() + 999L) / 1000L;

    err = SerReceiveWait(refNum, (long) n, ticks);
    if (!err)
	SerReceive(refNum, buf, (long) n, ticks, &err);
    if (err == serErrLineErr)
	SerReceiveFlush(refNum, 1L);

#ifdef DEBUG
    printf("read %d %d err %d\n", n, (int) buf[0], err);
#endif
    return (!err);
}

int
IO_Write(int n, unsigned char *buf)
{
    Err err;

    SerSend(refNum, buf, (long) n, &err);
    if (err == serErrLineErr)
	SerReceiveFlush(refNum, 1L);
    if (!err)
	SerSendWait(refNum, -1L);

#ifdef DEBUG
    printf("write %d err %d\n", n, err);
#endif
    return (!err);
}

int
IO_UpdateReturnBlock(int blocktime)
{
    IO_blocktime = blocktime * 1000;
    return blocktime;
}

int
IO_ReturnBaudRate()
{
    return IO_settings.baudRate;
}

int
IO_FlushBuffer()
{
    SerReceiveFlush(refNum, SysTicksPerSecond() / 4L);
    return 1;
}

char
IO_ReturnParity()
{
    if (!(IO_settings.flags & serSettingsFlagParityOnM))
	return 'N';
    if (IO_settings.flags & serSettingsFlagParityEvenM)
	return 'E';
    return 'O';
}

static int
IO_UpdateParityFlags(char parity)
{
    if (parity == 'N')
	IO_settings.flags &= ~serSettingsFlagParityOnM;
    else
	IO_settings.flags |= serSettingsFlagParityOnM;
    if (parity == 'E')
	IO_settings.flags |= serSettingsFlagParityEvenM;
    else
	IO_settings.flags &= ~serSettingsFlagParityEvenM;
}

char
IO_UpdateParity(char parity)
{
    IO_UpdateParityFlags(parity);
    SerSetSettings(refNum, &IO_settings);
    return IO_ReturnParity();
}

int
IO_UpdateStopBits(int stopbits)
{
    IO_settings.flags &= ~serSettingsFlagStopBitsM;
    if (stopbits == 1)
	IO_settings.flags |= serSettingsFlagStopBits1;
    else if (stopbits == 2)
	IO_settings.flags |= serSettingsFlagStopBits2;
    else
	stopbits = -1;

    if (stopbits != -1)
	SerSetSettings(refNum, &IO_settings);

    return stopbits;
}

int
IO_Close()
{
    if (!IO_isopen)
	return 0;
    IO_isopen = 0;
    return (!SerClose(refNum));
}
