/*
 * Send commands to a smartcard.
 *
 * Jim Rees, University of Michigan, February 1999
 */
/* $Id: lewis.c,v 1.40 2001/09/28 16:57:34 rees Exp $ */

#pragma pack(2)

#include <Common.h>
#include <System/SysAll.h>
#include <System/DataMgr.h>
#include <System/SysEvtMgr.h>
#include <UI/UIAll.h>

#include <System/Unix/unix_stdio.h>
#include <System/Unix/unix_stdlib.h>
#include <System/Unix/sys_types.h>
#include <string.h>

#include "sectok.h"
#include "sc7816.h"
#include "field.h"

#include "lewis.h"

/* This will pull in the clib sprintf instead of using the crippled Pilot rom one */
#undef sprintf
#undef vsprintf

#define MAXFIELD 1000

unsigned short random_sw[] = {
    0x9000, 0x0601, 0x0602, 0x0603, 0x0604, 0x0605, 0x0606, 0x0607,
    0x0608, 0x0609, 0x060a, 0x060c, 0x060d, 0x61ff, 0x6234, 0x6239,
    0x6255, 0x6257, 0x6258, 0x6281, 0x6283, 0x6300, 0x6381, 0x67ff,
    0x6980, 0x6982, 0x6983, 0x6984, 0x6985, 0x6987, 0x6988, 0x6a80,
    0x6a81, 0x6a82, 0x6a83, 0x6b00, 0x6cff, 0x6d00, 0x6e00, 0x6f14,
    0x6f15, 0x6f19, 0x6fb0, 0x9481
};

static Boolean lewis(EventPtr event);
int stt(char c);
void initmainform(FormPtr form);
int paycmd(int id);
void rst();
void listcmd(int id);
void go();
int docmd(int fd, int inout, unsigned char *cmd, unsigned char *buf, unsigned char *s, int *r1p, int *r2p);

/* These should match the order of list items in the resource file */
/* dir(ection): 0 in; 1 out; 2 in if data present else out */
static struct cmds {
    char cmd[5][3];
    int dir;
} cmds[] = {
    { { "0", "a4", "0", "0",  "2"}, 0}, /* select */
    { { "0", "c0", "0", "0", "28"}, 1}, /* get response */
    { { "0", "84", "0", "0",  "8"}, 1}, /* get challenge */
    { { "0", "b2", "0", "0",  "6"}, 1}, /* read rec */
    { { "0", "b0", "0", "0", "20"}, 1}, /* read binary */
    { { "0", "2a", "0", "0",  "8"}, 0}, /* submit key */
    { { "0", " 8", "5", "0",  "0"}, 0}, /* manage instance */
    { {"fe", "fe", "0", "0", "2c"}, 0}, /* ip7816 */
    { { "0", " 0", "0", "0",  "0"}, 2}, /* user1 */
    { { "0", " 0", "0", "0",  "0"}, 2}, /* user2 */
};
int cmdi; /* index into cmds for current command */

static char AUT0[] = "ad 9f 61 fe fa 20 ce 63";
static char SYN[] =
    "4500 002c d2fe 4000 7d06 0962 8dd3 5ca9"
    "8dd3 a91b 09b6 0050 4698 ae0d 0000 0000"
    "6002 2000 580f 0000 0204 05b4";

int fd, testmode;
FieldPtr sttfield, printfield, datfield, helpfield, cmdfield[5];

DWord
PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{
    int err;

    if (cmd == sysAppLaunchCmdNormalLaunch) {
	err = StartApplication();
	if (err)
	    return err;
	EventLoop();
	StopApplication();
	return 0;
    } else
	return sysErrParamErr;
}

StartApplication()
{
    fd = -1;
    cmdi = -1;
    FrmGotoForm(formID_lewis);
    return 0;
}

StopApplication()
{
    closereader();
    FrmCloseAllForms();
    return 0;
}

EventLoop()
{
    short err;
    int formID;
    FormPtr form;
    EventType event;

    do {
	EvtGetEvent(&event, 200);

	if (SysHandleEvent(&event))
	    continue;
	if (MenuHandleEvent((void *)0, &event, &err))
	    continue;

	if (event.eType == frmLoadEvent) {
	    formID = event.data.frmLoad.formID;
	    form = FrmInitForm(formID);
	    FrmSetActiveForm(form);
	    switch (formID) {
	    case formID_lewis:
	    case formID_cyber:
		FrmSetEventHandler(form, (FormEventHandlerPtr) lewis);
		break;
	    }
	}
	FrmDispatchEvent(&event);
    } while (event.eType != appStopEvent);
}

static Boolean
lewis(EventPtr event)
{
    FormPtr form = FrmGetActiveForm();
    Word item;
    FieldPtr field;
    int handled = 0;

    item = FrmGetFocus(form);
    if (item == noFocus)
	field = printfield;
    else
	field = FrmGetObjectPtr(form, item);

    switch (event->eType) {
    case frmOpenEvent:
	switch (event->data.frmOpen.formID) {
	case formID_lewis:
	    initmainform(form);
	    break;
	case formID_cyber:
	    initcyberform(form);
	    break;
	}
	handled = 1;
	break;

    case ctlSelectEvent:
	handled = paycmd(event->data.ctlEnter.controlID);
	break;

    case popSelectEvent:
	listcmd(event->data.popSelect.selection);
	break;

    case keyDownEvent:
	handled = scrollfield(field, event->data.keyDown.chr);
	break;

    case menuEvent:
	switch (event->data.menu.itemID) {
	case menuitemID_help:
	    listcmd(cmdi);
	    FrmHelp(HelpText);
	    break;
	case menuitemID_example:
	    FrmHelp(ExampleText);
	    break;
	case menuitemID_cla:
	    SetDefaultClass();
	    break;
	case menuitemID_save:
	    memofield(printfield, "lewis transcript\n");
	    break;
	case menuitemID_clear:
	    setfield(datfield, NULL);
	    setfield(printfield, NULL);
	    break;
	case menuitemID_cyber:
	    FrmGotoForm(formID_cyber);
	    break;
	case menuitemID_leesonlog:
	    if (fd < 0) {
		printf("reader not open\n");
		break;
	    }
	    hidefield(fieldID_msg);
	    dumplog(fd, 1);
	    showfield(fieldID_msg);
	    break;
	case menuitemID_about:
	    FrmAlert(alertID_about);
	    break;
	case menuitemID_copy:
	    FldCopy(field);
	    break;
	case menuitemID_paste:
	    FldPaste(field);
	    break;
	}
	handled = 1;
	break;

    default:
	handled = 0;
	break;
    }
    return handled;
}

void
initmainform(FormPtr form)
{
    int i;

    sttfield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_stt));
    printfield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_msg));
    datfield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_dat));

    for (i = 0; i < 5; i++)
	cmdfield[i] = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_cmd+i));

    update_mode();

    /* Should remember which command was selected */
    cmdi = -1;
    listcmd(0);
}

int
paycmd(int id)
{
    ControlPtr	button;
    FormPtr	frm = FrmGetActiveForm();
    int		code = 1;

    switch (id) {
    case buttonID_off:
	if (fd >= 0)
	    /* reader is on; turn it off */
	    closereader();
	else
	    openreader();
	update_mode();
	break;
    case buttonID_tst:
	testmode = 1;
	printf("test mode\n");
	stt('T');
	break;
    case buttonID_rst:
	rst();
	button = FrmGetObjectPtr( frm,
	    FrmGetObjectIndex( frm, buttonID_off ));
	CtlSetLabel( button, "Off" );
	break;
    case buttonID_go:
	go();
	break;
    default:
	code = cybercmd(id);
	break;
    }
    return code;
}

void
rst()
{
    unsigned char buf[36];
    static unsigned char testatr[] = {0x3b, 0x16, 0x94, 0x41, 0x63, 0x63, 0x65, 0x73, 0x73};
    int n, err;
    struct scparam param;

    if (!testmode) {
	if (!openreader())
	    return;
	n = screset(fd, buf, &err);
	if (err != SCEOK) {
	    printf("%s\n", scerrtab[err]);
	    return;
	}
    } else {
	n = sizeof testatr;
	memcpy(buf, testatr, n);
    }
    if (n) {
	hidefield(fieldID_msg);
	parse_atr(-1, SCRV, buf, n, &param);
	showfield(fieldID_msg);
    }
}

/* Select a command from the list, copy it to the display */

void
listcmd(int id)
{
    struct cmds *c;
    int i;

    if (cmdi >= 0) {
	/* Copy current command from display back to cmds struct */
	c = &cmds[cmdi];
	for (i = 0; i < 5; i++) {
	    if (FldGetTextLength(cmdfield[i]) <= 0)
		continue;
	    strcpy(c->cmd[i], FldGetTextPtr(cmdfield[i]));
	}
    }

    if (id != cmdi) {
	/* Copy newly selected command to the display */
	c = &cmds[id];
	for (i = 0; i < 5; i++)
	    setfield(cmdfield[i], c->cmd[i]);

	/* If "submit key," init data field with Cyberflex AUT0 key */
	if (atoh(c->cmd[1]) == 0x2a)
	    setfield(datfield, AUT0);
	else if (atoh(c->cmd[0]) == 0xfe)
	    setfield(datfield, SYN);

	FrmDrawForm(FrmGetActiveForm());
	cmdi = id;
    }
}

/* Set all commands to use the current class */

SetDefaultClass()
{
    int i, cla;

    if (cmdi >= 0 && FldGetTextLength(cmdfield[cmdi]) > 0) {
	cla = atoh(FldGetTextPtr(cmdfield[cmdi]));
	for (i = 0; i < sizeof cmds / sizeof (struct cmds); i++)
	    sprintf(cmds[i].cmd[0], "%x", cla);
    }
}

/* Send the current command to the card */

void
go()
{
    int i, n, r1, r2;
    unsigned char cmd[5], buf[256];
    char s[MAXFIELD], *cp;
    int inout, doIsearch = 0;

    if (!testmode && checkreader())
	return;

    /* Get the current command from the display */
    for (i = 0; i < 5; i++) {
	cp = (FldGetTextLength(cmdfield[i]) > 0) ? FldGetTextPtr(cmdfield[i]) : "";
	if (i == 1 && !strcmp(cp, "?"))
	    doIsearch = 1;
	else
	    cmd[i] = atoh(cp);
    }

    /* Decide whether it's an "in" or an "out" */
    inout = cmds[cmdi].dir;
    if (inout != 1) {
	if (FldGetTextLength(datfield) <= 0)
	    n = 0;
	else {
	    cp = FldGetTextPtr(datfield);
	    if (!strcmp(cp, "/")) {
		n = 2;
		buf[0] = 0x3f;
		buf[1] = 0x00;
	    } else
		n = parse_input(cp, buf, sizeof buf);
	}
	if (inout == 0 && n < cmd[4]) {
	    /* it's an "in" but luser didn't give us enough bytes */
	    printf("need at least %d bytes of data\n", cmd[4]);
	    return;
	}
    }
    if (inout == 2) {
	/* user (non-canned) command; if there is input data, it's an "in" command */
	if (n) {
	    inout = 0;
	} else
	    inout = 1;
    }

    if (testmode) {
	if (inout == 0) {
	    i = randombyte() % (sizeof random_sw / sizeof (unsigned short));
	    printf("%s\n", sectok_get_sw(random_sw[i]));
	} else {
	    for (i = 0; i < cmd[4]; i++)
		buf[i] = randombyte();
	    pr_hex(buf, cmd[4], s);
	}
	return;
    }

    if (doIsearch) {
	/* find instructions */
	int nfound = 0;
	for (i = 0; i < 0xff; i += 2) {
	    cmd[1] = i;
	    n = docmd(fd, inout, cmd, buf, s, &r1, &r2);
	    if (n >= 0 && r1 != 0x6d && r1 != 0x6e) {
		printf("%02x %s %s\n", i, lookup_cmdname(i), get_r1r2s(r1, r2));
		nfound++;
	    }
	}
	printf("Found %d instructions\n", nfound);
    } else {
	n = docmd(fd, inout, cmd, buf, s, &r1, &r2);
	if (n < 0)
	    printf("cla %02x ins %02x failed %d\n", cmd[0], cmd[1], n);
	else if (inout == 0 || !n)
	    print_r1r2(r1, r2);
    }
}

int
docmd(int fd, int inout, unsigned char *cmd, unsigned char *buf, unsigned char *s, int *r1p, int *r2p)
{
    int n;

    if (inout == 0)
	n = scrw(fd, cmd[0], cmd[1], cmd[2], cmd[3], cmd[4], buf, 255, buf, r1p, r2p);
    else if (inout == 1) {
	n = scread(fd, cmd[0], cmd[1], cmd[2], cmd[3], cmd[4], buf, r1p, r2p);
	if (n != cmd[4])
	  printf("p3=%d but scread returns %d\n", cmd[4], n);
    } else
	n = -1;
    if (n > 0) {
	/* if "read binary" print as ascii.  fix this! */
	if (cmd[1] == 0xb0 && buf[0] >= ' ' && buf[0] <= '~') {
	    buf[n] = '\0';
	    printf("%s\n", buf);
	} else
	    pr_hex(buf, n, s);
    }
    return n;
}

/* Update the mode indicator and power button to reflect reality. */

update_mode()
{
    char mode = testmode ? 'T' : (fd >= 0 ? '1' : '0');
    FormPtr frm = FrmGetActiveForm();
    ControlPtr button = FrmGetObjectPtr(frm, FrmGetObjectIndex(frm, buttonID_off));

    stt(mode);
    CtlSetLabel(button, (mode == '1') ? "Off" : "On");
}

openreader()
{
    if (fd < 0)
	fd = scopen(0, 0, NULL);
    if (fd < 0) {
	printf("can't open reader\n");
	IO_Close();
	return 0;
    }
    testmode = 0;
    stt('1');
    return 1;
}

closereader()
{
    if (fd >= 0) {
	scclose(fd);
	fd = -1;
    }
    testmode = 0;
    stt('0');
    return 0;
}

checkreader()
{
    if (fd < 0) {
	printf("reader is off\n");
	return 1;
    }
    return 0;
}

pr_hex(bp, n, s)
unsigned char *bp;
int n;
char *s;
{
    int i;
    char ts[4];

    /* message buf holds 6 lines of 12 */
    if (n > 72)
	n = 72;

    s[0] = '\0';
    for (i = 0; i < n; ) {
	sprintf(ts, "%02x ", bp[i++]);
	strcat(s, ts);
	if (i % 12 == 0 || i >= n) {
	    printf("%s\n", s);
	    s[0] = '\0';
	}
    }
}

/* Set the single-character message in the box at upper left of screen */

int
stt(char c)
{
    static char s[2];

    s[0] = c;
    setfield(sttfield, s);
}

hex(char c)
{
    if (c >= '0' && c <= '9')
	return (c - '0');
    c |= 0x20; /* downcase */
    if (c >= 'a' && c <= 'f')
	return (c + 10 - 'a');
    return -1;
}

atoh(s)
char *s;
{
    int n = 0, d;

    while (*s && *s == ' ')
	s++;
    while ((d = hex(*s)) >= 0) {
	n = n * 16 + d;
	s++;
    }
    return n;
}

/* Not by a long shot. */

randombyte()
{
    static int n = 12345;

    n = n * 47 + 7;
    return ((n >> 4) & 0xff);
}

/*
copyright 1999, 2000
the regents of the university of michigan
all rights reserved

permission is granted to use, copy, create derivative works 
and redistribute this software and such derivative works 
for any purpose, so long as the name of the university of 
michigan is not used in any advertising or publicity 
pertaining to the use or distribution of this software 
without specific, written prior authorization.  if the 
above copyright notice or any other identification of the 
university of michigan is included in any copy of any 
portion of this software, then the disclaimer below must 
also be included.

this software is provided as is, without representation 
from the university of michigan as to its fitness for any 
purpose, and without warranty by the university of 
michigan of any kind, either express or implied, including 
without limitation the implied warranties of 
merchantability and fitness for a particular purpose. the 
regents of the university of michigan shall not be liable 
for any damages, including special, indirect, incidental, or 
consequential damages, with respect to any claim arising 
out of or in connection with the use of the software, even 
if it has been or is hereafter advised of the possibility of 
such damages.
*/
