########################################################################
#
# This module implements an editor for the Tools dialog
#

#
# Call this routine in order to create a tools editor dialog.
# This routine and FillToolsMenu below are the only routines that
# should be called from outside this file.  There are many other
# private procedure beginning with the prefix "Tse:" and a
# private array variable named "Tse".  The rest of the program
# should take care not to use these names.
#
proc ToolSetEditor {} {
  global Tse
  if {[winfo exists .tse]} {
    raise .tse
    return
  }
  toplevel .tse
  wm title .tse {Tool Set Editor}
  frame .tse.btn -bd 1 -relief sunken
  pack .tse.btn -side bottom -fill x
  button .tse.btn.save -text Save -command Tse:SaveChanges
  button .tse.btn.new -text New -command Tse:AddEntry
  button .tse.btn.sort -text Sort -command Tse:SortTools
  button .tse.btn.exit -text Close -command Tse:ExitApp
  pack .tse.btn.save .tse.btn.sort .tse.btn.new \
       .tse.btn.exit -pady 10 -expand 1 -side left
  label .tse.title -text {Click to edit      Drag to reorder} \
    -bd 1 -relief raised -pady 5
  pack .tse.title -fill x -side top
  catch {source [glob ~/.apps]}
  if {![info exists order]} {
    set Tse(order) [lsort [array names application]]
  } else {
    set Tse(order) $order
  }
  foreach n $Tse(order) {
    if {![info exists application($n)]} continue
    set Tse(tool:$n) $application($n)
  }
  canvas .tse.c -width 400 -height 225 -yscrollcommand {.tse.sb set}
  pack .tse.c -side left -fill both -expand 1
  scrollbar .tse.sb -orient vertical -command {.tse.c yview}
  pack .tse.sb -side left -fill y

  .tse.c create text 0 0 -text X -tags test -anchor nw
  set bbox [.tse.c bbox test]
  set Tse(fontHeight) [lindex $bbox 3]
  incr Tse(fontHeight) 2
  set Tse(fontWidth) [lindex $bbox 2]
  .tse.c delete test
  Tse:Refill
}

#
# A program that wants to use this module should call the following
# routine to fill its "Tools" menu whenever the "Tools" menu button
# is pressed.  This routine and the ToolSetEditor routine above are
# the only procedures in this file that should have external scope.
#
proc FillToolsMenu menu {
  $menu delete 0 end
  catch {source [glob ~/.apps]}
  if {![info exists order]} {
    set order [array names application]
  }
  foreach appname $order {
    $menu add command -label $appname \
       -command "catch {$application($appname)}"
  }
  if {[llength $order]>0} {
    $menu add separator
  }
  $menu add command -label {Edit This Menu...} -command ToolSetEditor
}


#
# Show the user the error message given in $msg
#
proc Tse:ErrMsg {msg} {
  catch {destroy .tse.err}
  toplevel .tse.err
  wm title .tse.err Error
  message .tse.err.msg -aspect 1000 -text $msg
  pack .tse.err.msg -side top
  button .tse.err.ok -text Bummer -command {catch {destroy .tse.err}}
  pack .tse.err.ok -side top -pady 10
  update
  scan [wm geometry .tse.] %dx%d%d%d w h x y
  set rw [winfo reqwidth .tse.err]
  set rh [winfo reqheight .tse.err]
  wm geometry .tse.err +[expr $x+int(($w-$rw)/2)]+[expr $y+int(($h-$rh)/2)]
  wm deiconify .tse.err
  update
  focus .tse.err.btn
}

#
# Call this routine to shut down the tools editor
#
proc Tse:ExitApp {} {
  catch {destroy .tse}
}

# 
# This routine is called when the "Sort" button is pressed on
# the main tools dialog.
#
proc Tse:SortTools {} {
  global Tse
  set Tse(order) [lsort $Tse(order)]
  Tse:Refill
}

# 
# This routine is called when the "Save" button is pressed on
# the main tools dialog.  It writes the list of tools back to
# the disk file.
#
proc Tse:SaveChanges {} {
  set filename "[glob ~]/.apps"
  if {[catch {open $filename w} f]} {
    Tse:ErrMsg "Can't open $filename for writing"
    return
  }
  global Tse
  puts $f "set order [list $Tse(order)]"
  foreach name $Tse(order) {
    set value $Tse(tool:$name)
    puts $f "set [list application($name)] [list $value]"
  }
  close $f
}

#
# This routine is called when the "New" button is pressed on
# the main tool editor dialog.  It creates a new tool with a
# unique name, then immediately invokes an editor on that particular
# tool.
#
proc Tse:AddEntry {} {
  global Tse
  set name NewCommand
  if {[info exists Tse(tool:$name)]} {
    set cnt 1
    set name "NewCommand $cnt"
    while {[info exists "Tse(tool:$name)"]} {
      incr cnt
    }
  }
  set val "exec newcommand &"
  lappend Tse(order) $name
  set Tse(tool:$name) $val
  Tse:Refill
  Tse:Edit $name
}

#
# This routine redraws the list of tools on the canvas.
#
proc Tse:Refill {} {
  global Tse
  .tse.c delete all
  set max 15
  foreach n $Tse(order) {
    if {[string length $n]>$max} {
      set max [string length $n]
    }
  }
  set y 5
  set c1 [expr $Tse(fontWidth)*$max]
  set sp $Tse(fontHeight)
  set i 0
  foreach n $Tse(order) {
    incr i
    .tse.c create text 5 $y -text $n -anchor nw -tags x$i
    .tse.c create text $c1 $y -text $Tse(tool:$n) -anchor nw -tags x$i
    .tse.c bind x$i <ButtonPress-1> "Tse:ButtonDown [list $n] %x %y"
    .tse.c bind x$i <B1-Motion> "Tse:ButtonMotion %x %y"
    .tse.c bind x$i <ButtonRelease-1> "Tse:ButtonUp"
    incr y $sp
  }
  .tse.c config -scrollregion [.tse.c bbox all]
}

#
# This routine is called when the mouse button is pressed over
# an item on the canvas.  All we can do is record the position of
# the press.  We'll take different actions if this is a click or
# a drag.
#
proc Tse:ButtonDown {name x y} {
  global Tse
  set Tse(button:name) $name
  set Tse(button:x) $x
  set Tse(button:y) $y
  set Tse(button:drag) 0
}

#
# This routine is called when the mouse button is released.  This
# will indicate either a click or the end of a drag.  Take
# appropriate action for either case.
#
proc Tse:ButtonUp {} {
  global Tse
  if {!$Tse(button:drag)} {
    Tse:Edit $Tse(button:name)
  } else {
    Tse:DragRelease
  }
}

#
# This routine is called when the mouse moves while the button is
# held down.  This indicates a drag is in process.
#
proc Tse:ButtonMotion {x y} {
  global Tse
  if {$Tse(button:drag)} {
    Tse:DragTo $x $y
  } elseif {abs($y-$Tse(button:y))>4} {
    Tse:DragStart $x $y
  }
}

#
# This routine is called once when a drag is first started
#
proc Tse:DragStart {x y} {
  Tse:DragTo $x $y
}

#
# This routine is called every time the mouse move while executing
# a drag.
#
proc Tse:DragTo {x y} {
  global Tse
  set Tse(button:x) $x
  set Tse(button:y) $y
  set Tse(button:drag) 1
  .tse.c delete insline
  set n [llength $Tse(order)]
  set y [.tse.c canvasy $y]
  set i [expr int(($y+5+$Tse(fontHeight)/2)/$Tse(fontHeight)) - 1]
  if {$i<0} {set i 0}
  if {$i>$n} {set i $n}
  set liney [expr $i*$Tse(fontHeight) + 5]
  .tse.c create line 0 $liney 300 $liney -tags insline
  set Tse(button:insert-index) $i
}

#
# This routine is called when the button is released to complete a
# drag.  Reorder the tools as appropriate.
#
proc Tse:DragRelease {} {
  global Tse
  .tse.c delete insline
  set start [lsearch $Tse(order) $Tse(button:name)]
  set end $Tse(button:insert-index)
  if {$start==$end || $end==$start+1} return
  if {$start>$end} {
    set Tse(order) [lreplace $Tse(order) $start $start]
    set Tse(order) [linsert $Tse(order) $end $Tse(button:name)]
  } else {
    set Tse(order) [linsert $Tse(order) $end $Tse(button:name)]
    set Tse(order) [lreplace $Tse(order) $start $start]
  }
  Tse:Refill
}

#
# This routine is called to invoke a detail editor on the
# tools given by $name
#
proc Tse:Edit name {
  global Tse
  catch {destroy .tse.edit}
  toplevel .tse.edit
  wm title .tse.edit {Edit}
  wm withdraw .tse.edit
  frame .tse.edit.f1
  frame .tse.edit.f2
  pack .tse.edit.f1 -side top -fill x -pady 10 -padx 5
  pack .tse.edit.f2 -side top -fill x -padx 5
  label .tse.edit.f1.l -text Name -anchor e -width 8
  entry .tse.edit.f1.e -bd 1 -relief sunken -width 30
  .tse.edit.f1.e insert end $name
  pack .tse.edit.f1.l .tse.edit.f1.e -side left
  label .tse.edit.f2.l -text Command -anchor e -width 8
  entry .tse.edit.f2.e -bd 1 -relief sunken -width 30
  global application
  .tse.edit.f2.e insert end $Tse(tool:$name)
  pack .tse.edit.f2.l .tse.edit.f2.e -side left
  frame .tse.edit.f3
  pack .tse.edit.f3 -side top -fill x -padx 5 -pady 10
  button .tse.edit.f3.ok -text Ok -width 6 \
       -command "Tse:EditOk [list $name]"
  button .tse.edit.f3.delete -text Delete -width 6 \
       -command "Tse:EditDelete [list $name]"
  button .tse.edit.f3.cancel -text Cancel -width 6 \
       -command "Tse:EditCancel"
  pack .tse.edit.f3.ok .tse.edit.f3.delete .tse.edit.f3.cancel \
      -side left -padx 10
  update
  scan [wm geometry .tse] %dx%d%d%d w h x y
  set rw [winfo reqwidth .tse.edit]
  set rh [winfo reqheight .tse.edit]
  wm geometry .tse.edit +[expr $x+int(($w-$rw)/2)]+[expr $y+int(($h-$rh)/2)]
  wm deiconify .tse.edit
  update
  focus .tse.edit.f2.e
}

#
# This routine is called when the Cancel button on the detail
# editor is pressed.  It cancels the edit operation.
#
proc Tse:EditCancel {} {
  catch {destroy .tse.edit}
}

#
# This routine is called when the Delete button on the detail
# editor is pressed.  It deletes the tool currently being edited
# and shuts down the detail editor.
#
proc Tse:EditDelete name {
  global Tse
  catch {unset Tse(tool:$name)}
  set i [lsearch -exact $Tse(order) $name]
  if {$i>=0} {
    set Tse(order) [lreplace $Tse(order) $i $i]
  }
  Tse:EditCancel
  Tse:Refill
}

#
# This routine is called to complete the editing of tool details.
#
proc Tse:EditOk name {
  set value [.tse.edit.f2.e get]
  if {[string match {exec *} $value]==0} {
    .tse.edit.f2.e insert 0 "exec "
    focus .tse.edit.f2.e
    return
  }
  if {[string match {* &} $value]==0} {
    .tse.edit.f2.e insert end " &"
    focus .tse.edit.f2.e
    return
  }
  set newname [.tse.edit.f1.e get]
  if {[string length $newname]==0} {
    Tse:EditDelete $name
    return
  }
  global Tse
  if {[string compare $name $newname]!=0} {
    catch {unset Tse(tool:$name)}
    set i [lsearch -exact $Tse(order) $name]
    if {$i>=0} {
      if {[info exists Tse(tool:$newname)]} {
        set Tse(order) [lreplace $Tse(order) $i $i]
      } else {
        set Tse(order) [lreplace $Tse(order) $i $i $newname]
      }
    }
  }
  set Tse(tool:$newname) $value
  Tse:EditCancel
  Tse:Refill
}
