/*
 * NAME:
 *	filesystem.c -- Copyright (C) 1998 David Corcoran
 *                 corcordt@cs.purdue.edu
 *	Modifications for Open16K, Copyright (C) 1998 Schlumberger, All Rights Reserved.
 *
 * DESCRIPTION:
 *      This defines some File System specific card functions
 *
 * AUTHOR:
 *	David Corcoran, 3/17/98
 *	Danny Kumamoto, 1-Nov-1998 make it work for Cyberflex Open16K
 *
 * LICENSE: See file LICENSE.
 *
 */

#include <assert.h>
#include <string.h>

#include "filesystem.h"

static char _rcsid[] UNUSED = "$Id$";

bool File_Dir(BYTE dirnum, struct BinData *bindata) { 

  bindata->Header[0] = CARD_CLASS;			/* Class	*/
  bindata->Header[1] = 0xa8; 			/* Instruction	*/
  bindata->Header[2] = 0x00;			/* P1		*/
  bindata->Header[3] = dirnum;			/* P2		*/
  bindata->Header[4] = 0x28;			/* P3		*/
  /* send in dir num and 0x28 bytes for get parameter */
  
  bindata->CommandStatus = ISO_COMMAND_SUCCESS;
  return TRUE;
}

bool File_CreateFile(BYTE init, BYTE numrecords, BYTE *size, BYTE *fileid, BYTE filetype, BYTE *access, BYTE status, BYTE *keys, BYTE rlength, struct BinData *bindata) {
 
  bindata->Header[0] = CARD_CLASS;				/* Class       	  */
  bindata->Header[1] = 0xe0;                    	/* Instruction 	  */
  bindata->Header[2] = 0x00;                    	/* P1          	  */
  bindata->Header[3] = 0xe0;                    	/* Instruction 	  */
  bindata->Header[4] = 0x10;		       /* P3		  */    
  bindata->TxBuffer[7] = numrecords;			/* P2          	  */

  switch (filetype) {

  case ISO_FILE_DEDICATED:
  case ISO_FILE_APPLICATION:
  case ISO_FILE_BINARY:
  case ISO_FILE_PROGRAM:
  case ISO_FILE_VARIABLE_RECORD:
    bindata->TxBuffer[6] = 0;			/* Record Length	  */
    break;

  case ISO_FILE_FIXED_RECORD:
  case ISO_FILE_CYCLIC:
    bindata->TxBuffer[6] = rlength;	       /* Record Length  */
    break;

  default:
    return FALSE;
  }
  
  memcpy(&bindata->TxBuffer[0], size, 2);	/* File Size	  */
  memcpy(&bindata->TxBuffer[2], fileid, 2);	/* File ID	  */

  bindata->TxBuffer[4] = filetype;	/* File Type		  */

  bindata->TxBuffer[5] = status;		/* File Status	     	*/
							/* 0x00 - Invalidated	*/
							/* 0x01 - Validated	*/

  if (access)
    memcpy(&bindata->TxBuffer[8], access, 8);	/* Access Conditions 	*/
  else {
    bindata->TxBuffer[8] = 0xff; /* default */
    bindata->TxBuffer[9] = 0x00; /* CHV1 */
    bindata->TxBuffer[10] = 0x00; /* CHV2 */
    bindata->TxBuffer[11] = 0x00; /* AUT1 */
    bindata->TxBuffer[12] = 0x00; /* AUT2 */
    bindata->TxBuffer[13] = 0x00; /* AUT3 */
    bindata->TxBuffer[14] = 0x00; /* AUT4 */
    bindata->TxBuffer[15] = 0x00; /* Sup */
  }
  
  bindata->CommandStatus = ISO_COMMAND_SUCCESS;
  return TRUE;
}

bool File_CreateRecord(BYTE size, BYTE *data, struct BinData *bindata) {
 
  if (size + 5 > MAX_BUFFER_SIZE) {
 	bindata->CommandStatus = ISO_INVALID_COMMAND_SIZE;
	return FALSE;
  }

  bindata->Header[0] = CARD_CLASS;			/* Class	*/
  bindata->Header[1] = 0xe0;			/* Instruction	*/
  bindata->Header[2] = 0x00;			/* P1		*/
  bindata->Header[3] = 0x00;			/* P2		*/
  bindata->Header[4] = size;			/* P3		*/

  memcpy(bindata->TxBuffer, data, size); /* Data for Record 	  	*/
  
  bindata->CommandStatus = ISO_COMMAND_SUCCESS;
  return TRUE;
}

bool File_DeleteFile(BYTE size, BYTE *data, struct BinData *bindata) { 

  if (size + 5 > MAX_BUFFER_SIZE) {
 	bindata->CommandStatus = ISO_INVALID_COMMAND_SIZE;
	return FALSE;
  }
  bindata->Header[0] = CARD_CLASS;			/* Class	*/
  bindata->Header[1] = 0xe4;			/* Instruction	*/
  bindata->Header[2] = 0x00;			/* P1		*/
  bindata->Header[3] = 0x00;			/* P2		*/
  bindata->Header[4] = size;			/* P3		*/

  memcpy(bindata->TxBuffer, data, size);/* File Id/Cryptogram 		*/
  
  bindata->CommandStatus = ISO_COMMAND_SUCCESS;
  return TRUE;
}

bool File_DeleteAll(struct BinData *bindata) { 

  bindata->Header[0] = CARD_CLASS;			/* Class	*/
  bindata->Header[1] = 0xe4;			/* Instruction	*/
  bindata->Header[2] = 0x80;			/* P1		*/
  bindata->Header[3] = 0x00;			/* P2		*/
  bindata->Header[4] = 0x00;			/* P3		*/

  bindata->CommandStatus = ISO_COMMAND_SUCCESS;
  return TRUE;
}

bool File_ReadBin(BYTE high, BYTE low, BYTE size, struct BinData *bindata) {

	bindata->Header[0] = EXCEPTION_CLASS;	     /* Class	    */
	bindata->Header[1] = 0xb0; 	     /*	Instruction */	
	bindata->Header[2] = high;	     /*	P1	    */
	bindata->Header[3] = low;	     /*	P2	    */
	bindata->Header[4] = size; 	     /*	P3	    */

	return TRUE;
}


bool File_GetData(BYTE object, BYTE identifier, BYTE size, struct BinData *bindata) {
	
	bindata->Header[0] = CARD_CLASS;	     /* Class	    */
	bindata->Header[1] = 0xca; 	     /*	Instruction */
	bindata->Header[2] = object;	     /*	P1	    */
	bindata->Header[3] = identifier;    /*	P2	    */
	bindata->Header[4] = size;	     /*	P3	    */
  
	return TRUE;
}

bool File_Seek(BYTE offset, BYTE mode, BYTE size, BYTE *data, struct BinData *bindata) { 

  if (size + 5 > MAX_BUFFER_SIZE) {
 	bindata->CommandStatus = ISO_INVALID_COMMAND_SIZE;
	return FALSE;
  }					/* Seek Mode			     */
					/* 0x00 - From beginning forward     */
					/* 0x02 - From next location forward */

  if ((mode != 0x00) || (mode != 0x02)) {
	bindata->CommandStatus = ISO_INVALID_SEEK_MODE;
	return FALSE;
  }
  
  bindata->Header[0] = CARD_CLASS;			/* Class	*/
  bindata->Header[1] = 0xa2; 			/* Instruction	*/
  bindata->Header[2] = offset;			/* P1		*/
  bindata->Header[3] = mode;			/* P2		*/
  bindata->Header[4] = size;			/* P3		*/

  memcpy(bindata->TxBuffer, data, size);	/* Seek Pattern		*/
  
  bindata->CommandStatus = ISO_COMMAND_SUCCESS;
  return TRUE;
}

bool File_Increase(BYTE size, BYTE *data, struct BinData *bindata) {
 
  if (size + 5 > MAX_BUFFER_SIZE) {
 	bindata->CommandStatus = ISO_INVALID_COMMAND_SIZE;
	return FALSE;
  }

  bindata->Header[0] = CARD_CLASS;			/* Class	*/
  bindata->Header[1] = 0x32; 			/* Instruction	*/
  bindata->Header[2] = 0x00;			/* P1		*/
  bindata->Header[3] = 0x00;			/* P2		*/
  bindata->Header[4] = size;			/* P3		*/

  memcpy(bindata->TxBuffer, data, size); /* Value to be added	*/
  
  bindata->CommandStatus = ISO_COMMAND_SUCCESS;
  return TRUE;
}

bool File_Decrease(BYTE size, BYTE *data, struct BinData *bindata) {
  
  if (size + 5 > MAX_BUFFER_SIZE) {
 	bindata->CommandStatus = ISO_INVALID_COMMAND_SIZE;
	return FALSE;
  }

  bindata->Header[0] = CARD_CLASS;			/* Class	*/
  bindata->Header[1] = 0x30; 			/* Instruction	*/
  bindata->Header[2] = 0x00;			/* P1		*/
  bindata->Header[3] = 0x00;			/* P2		*/
  bindata->Header[4] = size;			/* P3		*/

  memcpy(bindata->TxBuffer, data, size);	/* Value to be removed	*/
  
  bindata->CommandStatus = ISO_COMMAND_SUCCESS;
  return TRUE;
}

bool File_WriteBin(BYTE high, BYTE low, BYTE size, BYTE *data, struct BinData *bindata) {
	
	if (size + 5 > MAX_BUFFER_SIZE) {
		bindata->CommandStatus = ISO_INVALID_COMMAND_SIZE;
		return FALSE;
	}

	bindata->Header[0] = EXCEPTION_CLASS;	     /* Class	    */
	bindata->Header[1] = 0xd0; 	     /*	Instruction */
	bindata->Header[2] = high;	     /*	P1	    */
	bindata->Header[3] = low;	     /*	P2	    */
	bindata->Header[4] = size;	     /*	P3	    */

	memcpy(bindata->TxBuffer, data, size); /* Data to write		*/

	return TRUE;
}

bool File_UpdateBin(BYTE high, BYTE low, BYTE size, BYTE *data, struct BinData *bindata) {

	if (size + 5 > MAX_BUFFER_SIZE) {
		bindata->CommandStatus = ISO_INVALID_COMMAND_SIZE;
		return FALSE;
	}

	bindata->Header[0] = EXCEPTION_CLASS;	     /* Class	    */
	bindata->Header[1] = 0xd6; 	     /*	Instruction */
	bindata->Header[2] = high;	     /*	P1	    */
	bindata->Header[3] = low;	     /*	P2	    */
	bindata->Header[4] = size;	     /*	P3	    */

	memcpy(bindata->TxBuffer, data, size); /* Data to write		*/
	return TRUE;
}


bool File_EraseBin(BYTE high, BYTE low, BYTE size, BYTE *data, struct BinData *bindata) {

	if (size + 5 > MAX_BUFFER_SIZE) {
		bindata->CommandStatus = ISO_INVALID_COMMAND_SIZE;
		return FALSE;
	}

	bindata->Header[0] = EXCEPTION_CLASS;	     /* Class	    */
	bindata->Header[1] = 0x0e; 	     /*	Instruction */
	bindata->Header[2] = high;	     /*	P1	    */
	bindata->Header[3] = low;	     /*	P2	    */
	bindata->Header[4] = size;	     /*	P3	    */

	memcpy(bindata->TxBuffer, data, size); /* Data to write		*/
	return TRUE;
}

bool File_ReadRec(BYTE recordnumber, BYTE mode, BYTE size, BYTE *data, struct BinData *bindata) {
	
	if (size + 5 > MAX_BUFFER_SIZE) {
		bindata->CommandStatus = ISO_INVALID_COMMAND_SIZE;
		return FALSE;
	}

	if (mode > 0x04)
		return FALSE;
  
	bindata->Header[0] = EXCEPTION_CLASS;	     /* Class	    */
	bindata->Header[1] = 0xb2;	     /*	Instruction */
	bindata->Header[2] = recordnumber;  /*	P1	    */
	bindata->Header[3] = mode;	     /*	P2	    */
	bindata->Header[4] = size;	     /*	P3	    */

	memcpy(bindata->TxBuffer, data, size); /* Data to write		*/
	return TRUE;
}

bool File_WriteRec(BYTE recordnumber, BYTE mode, BYTE size, BYTE *data, struct BinData *bindata) {

	if (size + 5 > MAX_BUFFER_SIZE) {
		bindata->CommandStatus = ISO_INVALID_COMMAND_SIZE;
		return FALSE;
	}

	if (mode > 0x04)
		return FALSE;
  
	bindata->Header[0] = EXCEPTION_CLASS;	     /* Class	    */
	bindata->Header[1] = 0xd2; 	     /*	Instruction */
	bindata->Header[2] = recordnumber;  /*	P1	    */
	bindata->Header[3] = mode;	     /*	P2	    */
	bindata->Header[4] = size;	     /*	P3	    */

	memcpy(bindata->TxBuffer, data, size); /* Data to write		*/
	return TRUE;
}

bool File_AppendRec(BYTE high, BYTE size, BYTE *data, struct BinData *bindata) {
	  
	if (size + 5 > MAX_BUFFER_SIZE) {
		bindata->CommandStatus = ISO_INVALID_COMMAND_SIZE;
		return FALSE;
	}

	bindata->Header[0] = EXCEPTION_CLASS;	     /* Class	    */
	bindata->Header[1] = 0xe2; 	     /*	Instruction */
	bindata->Header[2] = 0x00;	     /*	P1          */
	bindata->Header[3] = high;	     /*	P2	    */
	bindata->Header[4] = size;	     /*	P3	    */

	memcpy(bindata->TxBuffer, data, size); /* Data to write		*/
	return TRUE;
}

bool File_UpdateRec(BYTE recordnumber, BYTE mode, BYTE size, BYTE *data, struct BinData *bindata) {

	if (size + 5 > MAX_BUFFER_SIZE) {
		bindata->CommandStatus = ISO_INVALID_COMMAND_SIZE;
		return FALSE;
	}

	/* FIX: Need enum */

	if (mode > 0x04)
		return FALSE;
  
	bindata->Header[0] = EXCEPTION_CLASS;	     /* Class	    */
	bindata->Header[1] = 0xdc; 	     /*	Instruction */
	bindata->Header[2] = recordnumber;  /*	P1	    */
	bindata->Header[3] = mode;	     /*	P2	    */
	bindata->Header[4] = size;	     /*	P3	    */
  
	memcpy(bindata->TxBuffer, data, size); /* Data to write		*/
	return TRUE;
}

bool File_PutData(BYTE object, BYTE identifier, BYTE size, BYTE *data, struct BinData *bindata) {
	
	if (size + 5 > MAX_BUFFER_SIZE) {
		bindata->CommandStatus = ISO_INVALID_COMMAND_SIZE;
		return FALSE;
	}

	bindata->Header[0] = EXCEPTION_CLASS;	     /* Class	    */
	bindata->Header[1] = 0xda; 	     /*	Instruction */
	bindata->Header[2] = object;	     /*	P1	    */
	bindata->Header[3] = identifier;    /*	P2	    */
	bindata->Header[4] = size;	     /*	P3	    */

	memcpy(bindata->TxBuffer, data, size); /* Data to write		*/
	return TRUE;
}

bool File_Select(BYTE size, BYTE *data, struct BinData *bindata) {

	if (size + 5 > MAX_BUFFER_SIZE) {
		bindata->CommandStatus = ISO_INVALID_COMMAND_SIZE;
		return FALSE;
	}

	bindata->Header[0] = EXCEPTION_CLASS;          /* Class	    */
	bindata->Header[1] = 0xa4; 	     /*	Instruction */
	bindata->Header[2] = 0x00;	     /*	P1	    */
	bindata->Header[3] = 0x00;	     /*	P2	    */
	bindata->Header[4] = size; 	     /*	P3          */

	memcpy(bindata->TxBuffer, data, size); /* Data to write		*/
	return TRUE;  
}

bool File_SelectDF(struct BinData *bindata) {

	bindata->Header[0] = EXCEPTION_CLASS;          /* Class	    */
	bindata->Header[1] = 0xa4; 	     /*	Instruction */
	bindata->Header[2] = 0x03;	     /*	P1	    */
	bindata->Header[3] = 0x00;	     /*	P2	    */
	bindata->Header[4] = 0x00; 	     /*	P3          */

	return TRUE;  
}

bool File_SelectApplet(BYTE size, BYTE *data, struct BinData *bindata) {

	if (size + 5 > MAX_BUFFER_SIZE) {
		bindata->CommandStatus = ISO_INVALID_COMMAND_SIZE;
		return FALSE;
	}

	bindata->Header[0] = EXCEPTION_CLASS;          /* Class	    */
	bindata->Header[1] = 0xa4; 	     /*	Instruction */
	bindata->Header[2] = 0x04;	     /*	P1	    */
	bindata->Header[3] = 0x00;	     /*	P2	    */
	bindata->Header[4] = size; 	     /*	P3          */

	if (size > 0) /* we must be selecting non-default loader */
	  memcpy(bindata->TxBuffer, data, size); /* Data to write		*/
	/* else we assume that we are OK with default loader */
	return TRUE;  
}

bool File_ManageProgramValidate(BYTE size, BYTE *data, struct BinData *bindata) {
	if (size + 5 > MAX_BUFFER_SIZE) {
		bindata->CommandStatus = ISO_INVALID_COMMAND_SIZE;
		return FALSE;
	}

	bindata->Header[0] = CARD_CLASS;          /* Class	    */
	bindata->Header[1] = 0x0a; 	     /*	Instruction */
	bindata->Header[2] = 0x01;	     /*	P1	    */
	bindata->Header[3] = 0x00;	     /*	P2	    */
	bindata->Header[4] = size; 	     /*	P3          */

	if (size > 0) /* we must be using crypto card */
	  memcpy(bindata->TxBuffer, data, size); /* Data to write	*/
	return TRUE;  
}

void File_ManageProgramReset(struct BinData *bindata) {
	bindata->Header[0] = CARD_CLASS;          /* Class	    */
	bindata->Header[1] = 0x0a; 	     /*	Instruction */
	bindata->Header[2] = 0x02;	     /*	P1	    */
	bindata->Header[3] = 0x00;	     /*	P2	    */
	bindata->Header[4] = 0x00; 	     /*	P3          */
}
