/*
 *
 * NAME
 *	dbi2ctapi.c 
 *
 * DESCRIPTION:
 *	Mappingfrom DBIISO interface to CT-API - CT-BCS interface
 *
 * AUTHOR:
 *	Carlos Prados, October 1998
 *	Dave Corcoran, February 2000
 */

#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <assert.h>
#include "dbiiso.h"
#include "ctapi.h"

static char _rcsid[] UNUSED = "$Id$";

char currentReader[MAX_BUFFER_SIZE];

struct ReaderStatus {
  BYTE StatusWord[2];       /* Returned Command Status */
  BYTE Atr[MAX_ATR_SIZE];   /* Returned ATR */
  BYTE AtrSize;             /* Size of ATR             */
} ReaderStatus;

SCARDCONTEXT hContext = 0;
SCARDHANDLE  hCard    = 0;
SCARD_IO_REQUEST sSendPci, sRecvPci;

static BYTE BaudRate;

HANDLE
InitializeReaderPort(int baudrate, int bits, char parity, char *reader) { 
  LONG rv;
  
  strcpy(currentReader, reader);
  rv = SCardEstablishContext( SCARD_SCOPE_SYSTEM, 0, 0, &hContext );
  if ( rv == SCARD_S_SUCCESS ) {
    return 1;
  } else {
    return 0;
  }
}

bool DisposeReaderPort(void) {

  LONG rv;

  rv = SCardReleaseContext( hContext );
  if ( rv == SCARD_S_SUCCESS ) {
    return TRUE;
  } else {
    return FALSE;
  }
}


/***************************************************************************
 *
 * Function: DBI_IsoReset
 * Purpose : Resets and Activates smart card.
 *
****************************************************************************/

int DBI_IsoReset (BYTE *Atr, int *Length) {
  LONG rv;
  DWORD dwProtocol, dwReaderLen;
  DWORD dwAtrLen, dwState;
  UCHAR pbAtr[MAX_ATR_SIZE];
  int fd, i;
  char pcReaders[MAX_BUFFER_SIZE];

  if ( hCard != 0 ) {
    SCardDisconnect(hCard, SCARD_UNPOWER_CARD );
  }

  rv = SCardConnect( hContext, currentReader, SCARD_SHARE_EXCLUSIVE,
		     SCARD_PROTOCOL_T0, &hCard, &dwProtocol );
  if ( rv != SCARD_S_SUCCESS ) {
    return DBI_FAST_INIT_ERROR;
  }

  dwReaderLen = MAX_BUFFER_SIZE;
  rv = SCardStatus( hCard, pcReaders, &dwReaderLen, &dwState, &dwProtocol,
		    pbAtr, &dwAtrLen );
  if ( rv == SCARD_S_SUCCESS && dwAtrLen <= MAX_ATR_SIZE ) {
    memcpy( ReaderStatus.Atr, pbAtr, dwAtrLen );
    ReaderStatus.AtrSize = dwAtrLen;
    memcpy( Atr, pbAtr, dwAtrLen );
    *Length = dwAtrLen;
  }
  else {
    return DBI_FAST_INIT_ERROR;
  }
  return(DBI_FAST_OK); 
}

/***************************************************************************
 *
 * Function: DBI_IsoPts
 * Purpose : Sets protocol and baud rate mode.
 * 
****************************************************************************/

int DBI_IsoPts (int Speed, int Protocol) { 
  /* NOT USED : PROPRIETARY TO LPT READERS */

  return DBI_FAST_OK;
}

/***************************************************************************
 *
 * Function: DBI_IsoOff
 * Purpose : Deactivates smartcard slot.
 * 
****************************************************************************/

int DBI_IsoOff (void) {
  LONG rv;

  rv = SCardDisconnect( hCard, SCARD_UNPOWER_CARD );
  if ( rv != SCARD_S_SUCCESS ) {
    return(DBI_FAST_INIT_ERROR);
  }
  hCard = 0;
  return(DBI_FAST_OK);
}

/***************************************************************************
 *
 * Function: DBI_IsoRaw
 * Purpose : Perform raw communication with smartcard
 * 
****************************************************************************/

int DBI_IsoRaw(int TxLength, BYTE *TxBuffer, int RxLength, BYTE *RxBuffer) {
  LONG rv;
  DWORD dwRxLength;

  dwRxLength = RxLength;
  rv = SCardTransmit( hCard, SCARD_PCI_T0, TxBuffer, TxLength, &sRecvPci,
		      RxBuffer, &dwRxLength );
  if ( rv != SCARD_S_SUCCESS ) {
    return(DBI_FAST_INIT_ERROR);
  }
  memcpy( ReaderStatus.StatusWord,&RxBuffer[dwRxLength-2], 2 );
  return(DBI_FAST_OK); 
}


/***************************************************************************
 *
 * Function: DBI_IsoCase1
 * Purpose : Performs an ISO 7816-4 case 1 transaction:
 *           APDU = CLA,INS,P1,P2
 *           TPDU = CLA,INS,P1,P2
 *           Response = SW1, SW2
 * 
****************************************************************************/

int DBI_IsoCase1 (BYTE *Header) { 
  LONG rv;
  DWORD dwRxLength;
  dwRxLength = 2;

  rv = SCardTransmit( hCard, SCARD_PCI_T0, Header, 5, &sRecvPci,
		      ReaderStatus.StatusWord, &dwRxLength );
  if ( rv != SCARD_S_SUCCESS ) {
    return(DBI_FAST_INIT_ERROR);
  }
  return(DBI_FAST_OK);
}

/***************************************************************************
 *
 * Function: DBI_IsoCase2
 * Purpose : Performs an ISO 7816-4 case 2 transaction:
 *           APDU = CLA,INS,P1,P2,Le
 *           TPDU =CLA,INS,P1,P2,P3
 *           Response = SW1,SW2,Data
 *
****************************************************************************/

int DBI_IsoCase2 (BYTE *Header, BYTE *RxBuffer) {
  LONG rv;
  DWORD dwRxLength;
  UCHAR ucRecvBuffer[MAX_BUFFER_SIZE];
  dwRxLength = Header[4];

  rv = SCardTransmit( hCard, SCARD_PCI_T0, Header, 5, &sRecvPci,
		      ucRecvBuffer, &dwRxLength );
  if ( rv != SCARD_S_SUCCESS ) {
    return(DBI_FAST_INIT_ERROR);
  }
  
  memcpy(RxBuffer, ucRecvBuffer, dwRxLength - 2 );
  memcpy(ReaderStatus.StatusWord, &ucRecvBuffer[dwRxLength-2], 2 );
  return(DBI_FAST_OK);
}

/***************************************************************************
 *
 * Function: DBI_IsoCase3
 * Purpose : Performs an ISO 7816-4 case 3 transaction: 
 *	     APDU = CLA,INS,P1,P2,Lc,Data
 *           TPDU = CLA,INS,P1,P2,P3,Data
 *           Response = SW1, SW2
 * 
****************************************************************************/

int DBI_IsoCase3 (BYTE *Header, BYTE *TxBuffer) { 

  LONG rv;
  DWORD dwRxLength;
  UCHAR ucCommand[MAX_BUFFER_SIZE];

  memcpy( ucCommand, Header, 5 );
  memcpy( &ucCommand[5], TxBuffer, Header[4] );
  dwRxLength = 2;

  rv = SCardTransmit( hCard, SCARD_PCI_T0, ucCommand, 5 + Header[4], 
		      &sRecvPci, ReaderStatus.StatusWord, &dwRxLength );

  if ( rv != SCARD_S_SUCCESS ) {
    return(DBI_FAST_INIT_ERROR);
  }

  return(DBI_FAST_OK); 
}

/***************************************************************************
 *
 * Function: DBI_IsoSet
 * Purpose : Set library operating parameters.
 * 
****************************************************************************/

int DBI_IsoSet (int Parameter, int Value) { 
  /* NOT USED : PROPRIETARY TO LPT READERS */
  return DBI_FAST_OK;
}

/***************************************************************************
 *
 * Function: DBI_IsoGet
 * Purpose : Gets library operating parameters.
 * 
****************************************************************************/

int DBI_IsoGet (int Parameter, BYTE *Value) { 
  if (Value == NULL) {
   return DBI_FAST_NULL_POINTER;
  }
  if (Parameter == ISO_GET_RESET) {
    memcpy(Value,ReaderStatus.Atr,ReaderStatus.AtrSize); /* Return the ATR */
    return DBI_FAST_OK;
  }
  else if (Parameter == ISO_GET_SW) {
    memcpy(Value,ReaderStatus.StatusWord,2); /* Return Command Status      */
    return DBI_FAST_OK;
  }
  else if (Parameter == ISO_GET_ACTUAL_BAUD) {
    *Value = BaudRate;     /* Return the Baud Rate       */
    return DBI_FAST_OK;
  }
  else {
    return FALSE;
  }
  return DBI_FAST_OK;
}
