/*
 *
 * NAME: 
 *	towitoko.c
 *      Copyright (C) 1998 Carlos Prados
 *
 * DESCRIPTION: 
 *	Functions for communication with the Towitoko terminals.
 *
 * AUTHOR: 
 *	Carlos Prados Bocos, October 1998. 
 *      cprados@atos-ods.es
 *
 * LICENSE:
 *      See file COPYING
 */

#include <stdio.h>
#include "towitoko.h"
#include "serial.h"

/*
 * Internal Size of the commands and responses
 */

#define MAX_WRITE_SIZE 15
#define MAX_READ_SIZE 255

/*
 * Internal variable indicating current parity the terminal 
 * 	'O': odd parity
 *	'E': even parity 
 */

static 
char TerminalParity='E';

/*
 * Calculates checksum of a given byte array.
 * Parameters:
 *      buffer: data bytes to include in the checksum calculation.
 *      size: number of bytes.
 *	initial: initial value of checksum.
 * Returns:
 *      the checksum byte.
 */

static 
BYTE Towitoko_CalculateChecksum(BYTE* command, int size, BYTE initial) {

  BYTE checksum, x7;
  int i;

  checksum = initial;
  for (i=0; i < size; i++) {

    // Calculate new checksum
    checksum = checksum ^ command[i];
    x7 = (checksum & 0x80) >> 7;
    checksum = checksum << 1;
    checksum = !x7==0x01?checksum|0x01:checksum&0xFE;
  }

  return(checksum);
}

/*
 * Gets reader type
 * Parameters:
 *      type: returnet reader type
 * Returns:
 *      TRUE if reader type can be determined
 */

bool Towitoko_GetReaderType(BYTE* type) {

  BYTE status[3];
  BYTE buffer[MAX_WRITE_SIZE];

  buffer[0] = 0x00;
  buffer[1] = 0x01;
  
  if(!IO_Write(2,buffer))
    return(FALSE);

  if(!IO_Read(3,status))
    return(FALSE);

  *type=status[0];
  return(TRUE);                
}                                           

/*
 * Forms a card command and sends it to the card.
 * Parameters:
 * 	buffer: data bytes to be sent transparently to the card.
 *	size: number of bytes. 
 * Returns:
 * 	TRUE if size bytes are correctly sent to the card. 
 */

bool Towitoko_CardCommand(BYTE* buffer, int size) {

  BYTE command[MAX_WRITE_SIZE];
  int pointer=0;
  int card_bytes;

  while(pointer<size) {

    // Calculate number of bytes to send to the card
    card_bytes=(MAX_WRITE_SIZE-4)>size-pointer?size-pointer:(MAX_WRITE_SIZE-4);

    // Compose command
    command[0] = 0x6F;
    command[1] = (BYTE)card_bytes;
    command[2] = 0x05;
    command[3] = Towitoko_CalculateChecksum(command,3,0x00);
    memcpy(command+4,buffer+pointer,card_bytes);

    // Send command
    if(!IO_Write(card_bytes+4,command))
      return FALSE;
    
    // Increment pointer
    pointer+=card_bytes;
  }

  return(TRUE);
}

/*
 * Gets reader/card status.
 * Output parameter:
 *	result: 
 * 		0x00: No card / no change
 *		0x80: No card / cahnge
 *		0x40: Card present / no change
 *		0xC0: Card present / change
 * Returns:  
 *      TRUE if reader status can be determined.
 */

bool Towitoko_GetRdrStatus(BYTE* result) {

  BYTE buffer[MAX_WRITE_SIZE];
  BYTE status[2];

  buffer[0] = 0x03;
  buffer[1] = 0x07;

  if(!IO_Write(2,buffer))
    return(FALSE);

  if(!IO_Read(2,status))
    return(FALSE);

  *result=status[0] & 0xF0;

  /* Workaround: sometimes status[0]=0x20 */
  *result=((*result)==0x20?0x00:(*result)); 

  #ifdef DEBUG
  if(*result == 0x00) 
    printf("---- No card / no change \n"); 
  if(*result == 0x80)  
    printf("---- No card / change \n");
  if(*result == 0x40)
    printf ("---- Card present / no change \n");
  if(*result == 0xC0)
    printf ("---- Card present / change \n");
  #endif

  return(TRUE);
}

/*
 * Sets LED red/green/yellow/off.
 * Parameters: 
 *	color:
 *		0x00 - off
 *		0x01 - red
 *		0x02 - green
 *		0x03 - yellow
 * Returns: 
 *	TRUE if LED color can be set.
 */

bool Towitoko_SetLED(BYTE color) {

  BYTE status[1];
  BYTE buffer[MAX_WRITE_SIZE];

  buffer[0] = 0x6F;
  buffer[1] = color;
  buffer[2] = 0x6A;
  buffer[3] = 0x0F;
  buffer[4] = Towitoko_CalculateChecksum(buffer,4,0x00);
  if(!IO_Write(5,buffer))
    return(FALSE);

  if(!IO_Read(1,status))
    return(FALSE);

  return(status[0]==0x01);
}

/*
 * Activates the card.
 * Returns:
 *	TRUE if everithing goes OK.
 */

bool Towitoko_ActivateCard(void) {

  BYTE status[1];
  BYTE buffer[MAX_WRITE_SIZE];

  buffer[0]=0x60;
  buffer[1]=0x0F;
  buffer[2]=0x9C;
  if(!IO_Write(3,buffer))
    return(FALSE);

  if(!IO_Read(1,status))
    return(FALSE);

  return(status[0]==0x01);
}

/*
 * Deactivates the card.
 * Returns:
 *      TRUE if card can be deactivated.
 */

bool Towitoko_DeactivateCard(void) {

  BYTE status[1];
  BYTE buffer[MAX_WRITE_SIZE];

  buffer[0]=0x61;
  buffer[1]=0x0F;
  buffer[2]=0x98;
  if(!IO_Write(3,buffer))
    return(FALSE);

  if(!IO_Read(1,status))
    return(FALSE);

  return(status[0]==0x01);
}

/*
 * Resets card and gets its answer to reset.
 * Output parameters:
 *	Atr: returned ATR
 *	Length: length of the returned ATR
 * Returns:
 *	TRUE if everithing goes OK.
 */

bool Towitoko_ResetCard(BYTE Atr[], int *Length) {

  BYTE buffer1[MAX_WRITE_SIZE];
  BYTE buffer2[MAX_WRITE_SIZE];
  BYTE status[MAX_READ_SIZE];
  BYTE TDi;
  int i;
  bool loop; 

  // First sequence
  buffer1[0] = 0x80;
  buffer1[1] = 0x6F;
  buffer1[2] = 0x00;
  buffer1[3] = 0x05;
  buffer1[4] = 0x76;

  // Second sequence
  buffer2[0] = 0xA0;
  buffer2[1] = 0x6F;
  buffer2[2] = 0x00;
  buffer2[3] = 0x05;
  buffer2[4] = 0x74;

  for(i=0;i<3;i++) {

    #ifdef DEBUG
    printf("---- Sending Active Low Reset \n");
    #endif
    if(!IO_Write(5,buffer2))
      return(FALSE);

    if(IO_Read(1,status))
      break;

    #ifdef DEBUG
    printf("---- Sending Active High Reset \n");
    #endif
    if(!IO_Write(5,buffer1))
      return(FALSE);

    if(IO_Read(1,status))
      break;
  }

  /*
   * Check card convention (inverse/direct) and 
   * terminal/port parity match:
   *
   * Inverse convention: TS=0x3F -> parity=Odd
   * Direct convention: TS=0x3B ->  parity=Even
   *
   */
  if(status[0]==0x3F && TerminalParity=='E') {

    #ifdef DEBUG
    printf("---- Reseting parity to Odd \n");
    #endif

    // Sets terminal parity to Odd
    if(!Towitoko_SetTerminalParity('O'))
      return(FALSE);

    // Sets port parity to Odd
    if(IO_UpdateParity('O')==-1) 
      return(FALSE);

    return(Towitoko_ResetCard(Atr,Length));
  }
  else if(status[0]==0x3B && TerminalParity=='O') {

    #ifdef DEBUG
    printf("---- Reseting parity to Even \n");
    #endif

    // Sets terminal parity to Even
    if(!Towitoko_SetTerminalParity('E'))
      return(FALSE);

    // Sets port parity
    if(IO_UpdateParity('E')==-1)
      return(FALSE);

    return(Towitoko_ResetCard(Atr,Length));
  }

  if(status[0] == 0x3B || status[0] == 0x3F) {

    // Records TS
    Atr[0] = status[0];
    *Length=1;

    // Reads T0 
    if(!IO_Read(1,status))
      return(FALSE); 

    TDi = Atr[(*Length)++] = status[0];

    do {

      loop = FALSE;

      // Reads TAi if present
      if((TDi | 0xEF) == 0xFF) { 
        if(!IO_Read(1,status))
          return(FALSE);
        Atr[(*Length)++] = status[0];
      }

      // Reads TBi if present
      if((TDi | 0xDF) == 0xFF) {
        if(!IO_Read(1,status))
          return(FALSE);
        Atr[(*Length)++] = status[0];
      }

      // Reads TCi if present 
      if((TDi | 0xBF) == 0xFF) {
        if(!IO_Read(1,status))
          return(FALSE);
        Atr[(*Length)++] = status[0];
      }

      // Read TDi if present
      if((TDi | 0x7F) == 0xFF) {
        if(!IO_Read(1,status))
          return(FALSE);
        TDi = Atr[(*Length)++] = status[0];
        loop = TRUE;
      } 

    }
    while(loop);

    // Reads Historical bytes
    if(!IO_Read(Atr[1]&0x0F,status))
      return(FALSE);
    memcpy(Atr+(*Length),status,Atr[1]&0x0F);
    (*Length) += Atr[1]&0x0F;

    #ifdef DEBUG
    printf("---- ATR: ");
    for(i=0;i<(*Length);i++) 
      printf("%X ", Atr[i]);
    printf("\n");
    #endif

    return(TRUE);
  }

  (*Length) = 0;
  return(FALSE);
} 


/*
 *
 * Sets terminal (not COM port) parity to even/odd
 * Parameters:
 *	parity: '0' odd parity, 'E' even parity.
 * Returns: 
 *	TRUE if everithing goes OK.	
 */

bool Towitoko_SetTerminalParity(char parity) {

  BYTE status[1];
  BYTE buffer[MAX_WRITE_SIZE];

  if(parity!='O' && parity!='E') 
    return(FALSE);

  buffer[0] = 0x6F;
  buffer[1] = (parity=='O')?0x80:0x40;
  buffer[2] = 0x6A;
  buffer[3] = 0x0F;
  buffer[4] = Towitoko_CalculateChecksum(buffer,4,0x00);
  if(!IO_Write(5,buffer))
    return(FALSE);

  if(!IO_Read(1,status))
    return(FALSE);

  if(status[0]!=0x01)
    return(FALSE);

  TerminalParity=parity;
  return(TRUE);
}

