/*
 *
 * NAME:
 *      atr.c
 *      Copyright (C) 1998 Carlos Prados
 *
 * DESCRIPTION:
 *      Implementation of ISO-7816-3 answer to reset
 *      decodification procedures
 *
 * AUTHOR:
 *      Carlos Prados Bocos, October 1998.
 *      cprados@atos-ods.es
 *
 * LICENSE:
 *      See file COPYING
 */

#include "atr.h"

/*
 * Returns the specified interface byte
 * Parameters:
 *      atr: Answer to reset of the card
 *      latr: length of the ATR
 *	character: 'A' for TAi, 'B' for TBi, 'C' for TCi, 'D' for TDi 
 *      index: index i of the TXi interface byte
 *      int_byte: returned interface byte
 * Returns:
 *      ATR_OK: Interface byte can be determined
 *      ATR_NOTFOUND: Interface byte cannot be determined
 *      ATR_MALFORMED: Malformed ATR                 
 */

int ATR_InterfaceByte(BYTE atr[ATR_MAX_SIZE], int latr, char character, 
                       int index, BYTE* int_byte) {

  BYTE TDi;
  int pointer;
  int i;

  /* Check size of ATR */
  if(latr<2) 
    return(ATR_MALFORMED);

  /* Record T0 */
  TDi = atr[pointer=1];

  for(i=1;i<=index;i++) {

    /* Check TAi is present */
    if((TDi | 0xEF) == 0xFF) {
      if((++pointer)>=latr)
        return(ATR_MALFORMED);
      if((i==index) && (character=='A')) {
        *int_byte=atr[pointer];
        return(ATR_OK);
      }
    }

    /* Check TBi is present */
    if((TDi | 0xDF) == 0xFF) {
      if((++pointer)>=latr)
        return(ATR_MALFORMED);
      if((i==index) && (character=='B')) {
        *int_byte=atr[pointer];
        return(ATR_OK);
      }
    }

    /* Check TCi is present */
    if((TDi | 0xBF) == 0xFF) {
      if((++pointer)>=latr)
        return(ATR_MALFORMED);
      if((i==index) && (character=='C')) {
        *int_byte=atr[pointer];
        return(ATR_OK);
      }
    }

    /* Read TDi if present */
    if((TDi | 0x7F) == 0xFF) {
      if((++pointer)>=latr)
        return(ATR_MALFORMED);
      if((i==index) && (character=='D')) {
        *int_byte=atr[pointer];
        return(ATR_OK);
      }
      TDi=atr[pointer];
    }
  }

  /* The specified interface byte couldn't be found */
  return(ATR_NOTFOUND);
}

/*
 * Returns the historical bytes of an ATR
 * Parameters:
 *      atr: Answer to reset of the card
 *      latr: length of the ATR
 *      hist: returned historical bytes
 *      lhist: length of returned historical bytes
 * Returns:
 *      ATR_OK: Hystorical bytes can be determined
 *      ATR_NOTFOUND: There is no historycal bytes
 *      ATR_MALFORMED: Malformed ATR             
 */

int ATR_HistoricalBytes(BYTE atr[ATR_MAX_SIZE], int latr,
                         BYTE hist[ATR_MAX_SIZE], int* lhist) {

  BYTE TDi;
  bool loop=TRUE;
  int pointer;

  /* Check size of ATR */
  if(latr<2)
    return(ATR_MALFORMED);

  /* Record T0 and number of historical bytes */
  TDi = atr[pointer=1];
  *lhist = TDi & 0x0F;

  while(loop){

    loop=FALSE;

    // Check TAi is present
    if((TDi | 0xEF) == 0xFF) 
      pointer++; 

    // Check TBi is present
    if((TDi | 0xDF) == 0xFF) 
      pointer++;

    // Check TCi is present
    if((TDi | 0xBF) == 0xFF) 
      pointer++;

    // Read TDi if present
    if((TDi | 0x7F) == 0xFF) {
      if((++pointer)>=latr)
        return(ATR_MALFORMED); 
      TDi=atr[pointer];
      loop=TRUE;
    }
  }

  // Check ATR length
  if((++pointer)+(*lhist)>latr)
    return(ATR_MALFORMED);

  if(*lhist==0) 
    return(ATR_NOTFOUND);
  memcpy(hist,atr+pointer,*lhist);
  return(ATR_OK);
}

/*
 * Determines the transport protocol of a card, given it's ATR
 * Parameter:
 *      atr: Answer to reset of the card
 *      length: length of the ATR
 *      protocol: returned protocol
 * Returns:
 *      ATR_OK: Protocol can be determined
 *      ATR_MALFORMED: Malformed ATR                                       
 */

int ATR_Protocol(BYTE atr[ATR_MAX_SIZE],int length,BYTE* protocol) {

  BYTE TD;
  int status;

  // Get TD0 interface byte
  if((status=ATR_InterfaceByte(atr,length,'D',1,&TD))==ATR_MALFORMED) 
    return(ATR_MALFORMED); 

  // Determine protocol (T=0 by default)
  if(status==ATR_NOTFOUND)
    *protocol=ATR_PROTOCOL_T0;
  else
    *protocol = TD & 0x0F;

  return(ATR_OK);
}

