/******************************************************************

	MUSCLE SmartCard Development ( http://www.linuxnet.com )
	    Title  : winscard.c
	    Package: pcsc lite
            Author : David Corcoran
            Date   : 7/27/99
            Purpose: This handles smartcard reader communications. 
	            
********************************************************************/

#include <winscard.h>
#include <readerfactory.h>
#include <prothandler.h>
#include <ifdhandler.h>
#include <atrhandler.h>
#include <debuglog.h>
#include <configfile.h>
#include <sys_generic.h>

SCARD_IO_REQUEST g_rgSCardT0Pci, g_rgSCardT1Pci, g_rgSCardRawPci;

LONG SCardEstablishContext( DWORD dwScope, LPCVOID pvReserved1, 
			    LPCVOID pvReserved2, LPSCARDCONTEXT phContext) {

  LONG rv;

  /* Only do the folowing when RPC is not being used 
     and a local static link has been made           */

#ifdef PCSCLITE_LOCAL_LINK
  static int sflag = 0;

  if ( sflag == 0 ) {
    rv = DBUpdateReaders ( PCSCLITE_READER_CONFIG ); 
    g_rgSCardT0Pci.dwProtocol  = SCARD_PROTOCOL_T0;
    g_rgSCardT1Pci.dwProtocol  = SCARD_PROTOCOL_T1;
    g_rgSCardRawPci.dwProtocol = SCARD_PROTOCOL_RAW;
    sflag                      = 1;
  }

#endif

  /* Just uses the pointer as a unique identifier 
     instead of using the math rand() function     */

  *phContext = (SCARDCONTEXT)&rv;

  return SCARD_S_SUCCESS;
}

LONG SCardReleaseContext( SCARDCONTEXT hContext ) {
  /* Nothing to do here RPC layer will handle this */
  return SCARD_S_SUCCESS;
}

LONG SCardSetTimeout( SCARDCONTEXT hContext, DWORD dwTimeout ) {
  /* This is only used at the client side
     of an RPC call but just in case someone
     calls it here                           */

  return SCARD_E_UNSUPPORTED_FEATURE;
}

LONG SCardConnect( SCARDCONTEXT hContext, LPCSTR szReader, DWORD dwShareMode,
                   DWORD dwPreferredProtocols, LPSCARDHANDLE phCard,
                   LPDWORD pdwActiveProtocol ) {              
  LONG rv;
  struct ReaderContext *rContext; 
  UCHAR pucAtr[MAX_ATR_SIZE], ucAvailable;
  LPSTR mszReaderNames;
  DWORD dwAtrLength, dwState, dwStatus;
  DWORD dwReaderLen, dwProtocol;

  /* Set hCard to zero */
  *phCard = 0;

#ifdef PCSC_DEBUG
    DebugLogC("SCardConnect: Attempting Connect to %s\n", szReader, 
	     __FILE__, __LINE__);
#endif

  rv = RFReaderInfo( (char *)szReader, &rContext );   

  if ( rv != SCARD_S_SUCCESS ) { 
#ifdef PCSC_DEBUG
    DebugLogC("SCardConnect: Reader %s Not Found\n", szReader, 
	     __FILE__, __LINE__);
#endif
    return rv; 
  }

  /*******************************************/
  /* This section checks for simple errors   */
  /*******************************************/
  
  /* We only use T=0/T=1/RAW for now          */
  if ( (dwPreferredProtocols > 15)||(dwPreferredProtocols < 1 ) ) {
    return SCARD_E_UNSUPPORTED_FEATURE;   
  }

  /* Connect if not exclusive mode    */
  if ( rContext->dwContexts == -1 ) {
#ifdef PCSC_DEBUG
    DebugLogA("SCardConnect: Error Reader Exclusive\n", __FILE__, __LINE__);
#endif
    return SCARD_E_SHARING_VIOLATION;
  }
  
  /*******************************************/
  /* This section tries to load the driver   */
  /*******************************************/
  
#ifndef PCSCLITE_STATIC_DRIVER
  /* Load the driver since first time */
  if ( rContext->dwContexts == 0 ) {
    rv = RFLoadReader( (char *)szReader );
    if ( rv != SCARD_S_SUCCESS ) { 
      return rv; 
    }
  }
#endif

  /* Prepare the SCARDHANDLE identity  */
  *phCard = rContext->dwIdentity + rContext->dwContexts;

#ifdef PCSC_DEBUG
    DebugLogB("SCardConnect: hCard Identity: %x\n", *phCard, 
	     __FILE__, __LINE__);
#endif
  
  /*******************************************/
  /* This section tries to open the port     */
  /*******************************************/

  /* Only do power/port if first time */
  if ( rContext->dwContexts == 0 ) {    
    rv = IFDOpenIFD( rContext, rContext->dwPort );

    if ( rv != IFD_SUCCESS ) { 
#ifdef PCSC_DEBUG
    DebugLogB("SCardConnect: Open Port %x Failed\n", rContext->dwPort, 
	     __FILE__, __LINE__);
#endif
      *phCard = 0;
      RFUnloadReader( (char *)szReader );
      return SCARD_E_NOT_READY; 
    }    
    
    /*******************************************/
    /* This section tries to determine the     */
    /* presence of a card or not               */
    /*******************************************/

    rv = IFDStatusICC( rContext, &dwStatus, &dwProtocol, 
		       pucAtr, &dwAtrLength );

    if ( ((dwStatus & SCARD_ABSENT) == 1 ) && 
	 dwPreferredProtocols != SCARD_IGNORE_CARD ) {  

      rv = IFDCloseIFD( rContext );
      
#ifndef PCSCLITE_STATIC_DRIVER
      RFUnloadReader( (char *)szReader );
#endif         
      
#ifdef PCSC_DEBUG
      DebugLogA("SCardConnect: Card Not Inserted\n", __FILE__, __LINE__);
#endif
      
      *phCard = 0;
      return SCARD_E_NO_SMARTCARD;
    }
    
    /*******************************************/
    /* This section tries to power up the card */
    /*******************************************/

    if ( dwPreferredProtocols != SCARD_IGNORE_CARD ) {  
      rv = IFDPowerICC( rContext, IFD_POWER_UP, 
			pucAtr, &dwAtrLength );

      if ( rv != IFD_SUCCESS ) {    
	rv = IFDCloseIFD( rContext );
#ifndef PCSCLITE_STATIC_DRIVER
	RFUnloadReader( (char *)szReader );
#endif
   
#ifdef PCSC_DEBUG
    DebugLogA("SCardConnect: Power Up Failed\n", __FILE__, __LINE__);
#endif

	*phCard = 0;
	return SCARD_E_NO_SMARTCARD;
      }      
    }
    
    rContext->dwStatus  = 0;
    rContext->dwStatus |= SCARD_POWERED;
    rContext->dwStatus |= SCARD_NEGOTIABLEMODE;    
  }
  
  /*******************************************/
  /* This section tries to decode the ATR    */
  /* and set up which protocol to use        */
  /*******************************************/
  
  if ( rContext->dwContexts == 0 && dwPreferredProtocols 
       != SCARD_IGNORE_CARD ) {
    
    dwReaderLen    = strlen( szReader ) + 1;
    mszReaderNames = (LPSTR)malloc(dwReaderLen);
    SCardStatus( *phCard, mszReaderNames, &dwReaderLen, &dwState, 	      
		 &dwProtocol, pucAtr, &dwAtrLength );
    free(mszReaderNames);

    rContext->dwProtocol = PHGetDefaultProtocol( pucAtr, dwAtrLength );
    ucAvailable          = PHGetAvailableProtocols( pucAtr, dwAtrLength );

    rContext->dwProtocol = PHSetProtocol( rContext, dwPreferredProtocols,
                                          ucAvailable );
  }

  *pdwActiveProtocol = rContext->dwProtocol; 
  
#ifdef PCSC_DEBUG
    DebugLogB("SCardConnect: Active Protocol: %d\n", *pdwActiveProtocol, 
	     __FILE__, __LINE__);
#endif

  /*******************************************/
  /* This section tries to set up the        */
  /* exclusivity modes. -1 is exclusive      */
  /*******************************************/

  if ( dwShareMode == SCARD_SHARE_EXCLUSIVE ) {
    if ( rContext->dwContexts == 0 ) {
      rContext->dwContexts = -1;
      RFLockSharing( *phCard );
    } else {
      return SCARD_E_SHARING_VIOLATION;
    }
  } else {
    rContext->dwContexts += 1;
  }
  
  return SCARD_S_SUCCESS;
}

LONG SCardDisconnect( SCARDHANDLE hCard, DWORD dwDisposition ) {
  
  LONG rv;
  LPBYTE pucAtr[MAX_ATR_SIZE];
  struct ReaderContext *rContext; 
  DWORD dwAction, dwAtrLen;

  if ( hCard == 0 ) { return SCARD_E_INVALID_HANDLE; }
  
  rv = RFReaderInfoById( hCard, &rContext );   
  if ( rv != SCARD_S_SUCCESS ) { return rv; }
  
  RFUnlockSharing( hCard );
  
#ifdef PCSC_DEBUG
  DebugLogB("SCardDisconnect: Active Contexts: %d\n", rContext->dwContexts, 
	    __FILE__, __LINE__);
#endif
  
  switch( dwDisposition ) {
  case SCARD_LEAVE_CARD:     /* Do nothing here */
    break;                    
  case SCARD_UNPOWER_CARD:
    dwAction = IFD_POWER_DOWN;
    break;
  case SCARD_RESET_CARD:
    dwAction = IFD_RESET;
    break;
  case SCARD_EJECT_CARD:
    break;
  default:
    return SCARD_E_INVALID_VALUE;
  };
  
  
  if ( rContext->dwContexts == 1 || rContext->dwContexts == -1 ) 
    {
      if ( dwDisposition != SCARD_LEAVE_CARD && 
	   dwDisposition != SCARD_EJECT_CARD ) 
	{
	  rv = IFDPowerICC( rContext, dwAction, pucAtr, &dwAtrLen );
	}
      
      /* FIX :: Need to take care of eject */
      
      rContext->dwStatus  &= ~SCARD_POWERED;
      
      rv = IFDCloseIFD( rContext );    
#ifndef PCSCLITE_STATIC_DRIVER
      RFUnloadReader( (char *)rContext->lpcReader );
#endif
      
      rContext->dwLockId = 0;
    }
  
  /* For exclusive connection */
  if ( rContext->dwContexts == -1 ) {
    rContext->dwContexts = 0;
    return SCARD_S_SUCCESS;
  }
  
  rContext->dwContexts -= 1;
  return SCARD_S_SUCCESS;
}

LONG SCardBeginTransaction( SCARDHANDLE hCard ) {
  LONG rv;
  if ( hCard == 0 ) { return SCARD_E_INVALID_HANDLE; }
  rv = RFLockSharing( hCard );

#ifdef PCSC_DEBUG
  DebugLogB("SCardBeginTransaction: Status: %d.\n", rv,
            __FILE__, __LINE__);
#endif

  return rv; 
}

LONG SCardEndTransaction( SCARDHANDLE hCard, DWORD dwDisposition ) {
  LONG rv;

  /* Ignoring dwDisposition for now */
  if ( hCard == 0 ) { return SCARD_E_INVALID_HANDLE; }
  rv = RFUnlockSharing( hCard );  

#ifdef PCSC_DEBUG
  DebugLogB("SCardEndTransaction: Status: %d.\n", rv,
            __FILE__, __LINE__);
#endif

  return rv;
}

LONG SCardCancelTransaction( SCARDHANDLE hCard ) {
  LONG rv;

  /* Ignoring dwDisposition for now */
  if ( hCard == 0 ) { return SCARD_E_INVALID_HANDLE; }
  rv = RFUnlockSharing( hCard );  

#ifdef PCSC_DEBUG
  DebugLogB("SCardCancelTransaction: Status: %d.\n", rv,
            __FILE__, __LINE__);
#endif

  return rv;
}

LONG SCardStatus( SCARDHANDLE hCard, LPSTR mszReaderNames, 
		  LPDWORD pcchReaderLen, LPDWORD pdwState, 
		  LPDWORD pdwProtocol, LPBYTE pbAtr,
                  LPDWORD pcbAtrLen ) {  
  LONG rv;
  LONG lReaderSize;
  struct ReaderContext *rContext; 

  *pcbAtrLen      = 0;
  *pdwState       = 0;
  *pdwProtocol    = 0;

  if ( hCard == 0 ) { return SCARD_E_INVALID_HANDLE; }
  
  rv = RFReaderInfoById( hCard, &rContext );   

  /* Cannot find the hCard in this context */
  if ( rv != SCARD_S_SUCCESS ) { 
    *pcchReaderLen  = 0;
    return rv; 
  } else {
    /* Check to see if the buffer is large enough
       if not return the correct size of the buffer */

    lReaderSize = strlen( rContext->lpcReader ) + 1;
    if ( *pcchReaderLen < lReaderSize || mszReaderNames == 0 ) 
      {
	*pcchReaderLen = lReaderSize;
	return SCARD_E_INSUFFICIENT_BUFFER;
      }
  }
  
  rv = IFDStatusICC( rContext, pdwState, pdwProtocol, 
		     pbAtr, pcbAtrLen );

  /* Return the structure values */
  strcpy( mszReaderNames, rContext->lpcReader );
  *pcchReaderLen  = lReaderSize;
  *pdwProtocol    = rContext->dwProtocol;

  return SCARD_S_SUCCESS;
}

LONG SCardGetStatusChange( SCARDCONTEXT hContext, DWORD dwTimeout, 
			   LPSCARD_READERSTATE_A rgReaderStates, 
                           DWORD cReaders ) {

  LONG rv;
  struct ReaderContext *rContext; 
  LPSTR lpcReaderName;
  DWORD dwReaders, dwProtocol;
  DWORD dwEventState, dwAtrSize;
  BYTE pbAtr[MAX_ATR_SIZE];
  DWORD dwCurrentState, dwTime;
  DWORD dwActiveProtocol, dwState;
  SCARDHANDLE hCard;

  dwCurrentState               = rgReaderStates->dwCurrentState;
  rgReaderStates->dwEventState = 0;
  dwEventState                 = 0;
  dwTime                       = 0;

  /* Just ignore this one and return */
  if ( rgReaderStates->dwCurrentState & SCARD_STATE_IGNORE ) {
    rgReaderStates->dwEventState = SCARD_STATE_IGNORE;
    return SCARD_S_SUCCESS;
  }

  lpcReaderName = (char *)rgReaderStates->szReader;

  rv = RFReaderInfo( lpcReaderName, &rContext );   
  if ( rv != SCARD_S_SUCCESS ) {
    rgReaderStates->dwEventState = 
      SCARD_STATE_UNKNOWN | SCARD_STATE_CHANGED | SCARD_STATE_IGNORE;   
    return rv; 
  }

  /* vHandle is zero if no connections are made               */
  /* - No libraries are loaded if not in static mode          */
  /* We must first make sure the libraries are loaded         */  

  if ( rContext->vHandle == 0 ) {
#ifndef PCSCLITE_STATIC_DRIVER 
    rv = SCardConnect( 0, lpcReaderName, SCARD_SHARE_SHARED, 
		       SCARD_IGNORE_CARD, &hCard, &dwActiveProtocol );
    if ( rv != SCARD_S_SUCCESS ) { return rv; }
#endif
  }

  /* Now is where we start our event checking loop */

#ifdef PCSC_DEBUG
    DebugLogA("SCardGetStatusChange: Event Loop Start\n", __FILE__, __LINE__);
#endif

    rContext->dwBlockStatus = hContext; /* This context owns the block */

    rv = RFReaderInfo( (char *)lpcReaderName, &rContext );   
    if ( rv != SCARD_S_SUCCESS ) { 
      rgReaderStates->dwEventState = 
	SCARD_STATE_UNKNOWN | SCARD_STATE_CHANGED | SCARD_STATE_IGNORE;
      rContext->dwBlockStatus = BLOCK_STATUS_RESUME;
      SCardDisconnect( hCard, SCARD_UNPOWER_CARD );
      return rv; 
    } else if ( dwCurrentState & SCARD_STATE_UNAVAILABLE ) {
      dwEventState |= SCARD_STATE_CHANGED;
    }

  do {

    /* Now we check all the Reader States */

    rv = IFDStatusICC( rContext, &dwState, &dwProtocol,
		       pbAtr, &dwAtrSize );

    /* Card is now absent                   */
    if ( dwState & SCARD_ABSENT ) 
      {
	dwEventState |= SCARD_STATE_EMPTY;

	/* After present the rest are assumed */
	if ( dwCurrentState & SCARD_STATE_PRESENT ||
	     dwCurrentState & SCARD_STATE_ATRMATCH ||
	     dwCurrentState & SCARD_STATE_EXCLUSIVE ||
	     dwCurrentState & SCARD_STATE_INUSE ) 
	  {
	    dwEventState |= SCARD_STATE_CHANGED;	
	  }
	
	/* Card is now present              */
      } else {
	dwEventState |= SCARD_STATE_PRESENT;
	
	if ( dwCurrentState & SCARD_STATE_EMPTY ) {
	  dwEventState |= SCARD_STATE_CHANGED;
	}
	
      }

	/* Now figure out sharing modes     */
    if ( rContext->dwContexts == -1 ) {
      dwEventState |= SCARD_STATE_EXCLUSIVE;
      if ( !dwCurrentState & SCARD_STATE_EXCLUSIVE ) {
	dwEventState |= SCARD_STATE_CHANGED;
      }
    } else if ( rContext->dwContexts >= 1 ) {
      dwEventState |= SCARD_STATE_INUSE;
      if ( !dwCurrentState & SCARD_STATE_INUSE ) {
	dwEventState |= SCARD_STATE_CHANGED;
      }
    }
    
    SYS_USleep( PCSCLITE_STATUS_WAIT );
    
    if ( dwTimeout != 0 ) {
      dwTime += PCSCLITE_STATUS_WAIT;
      
      if ( dwTime >= (dwTimeout*10000) ) {
	rgReaderStates->cbAtr        = dwAtrSize;
	rgReaderStates->dwEventState = dwEventState;
	rContext->dwBlockStatus      = BLOCK_STATUS_RESUME;
        memcpy( rgReaderStates->rgbAtr, pbAtr, MAX_ATR_SIZE );
        SCardDisconnect( hCard, SCARD_UNPOWER_CARD );
        return SCARD_E_TIMEOUT;
      }
    }
    
  } while ( (dwEventState & SCARD_STATE_CHANGED) == 0   &&
	    (dwCurrentState != SCARD_STATE_UNAWARE)     &&
             rContext->dwBlockStatus != BLOCK_STATUS_RESUME );

  /* If a card is inserted and no-one else is sharing
     we must power/reset the card to redetermine it's ATR   */

  if ( (dwEventState & SCARD_STATE_PRESENT ) && 
       (rContext->dwContexts == 1) ) {
 
    rv = IFDPowerICC( rContext, IFD_POWER_UP, 
		      pbAtr, &dwAtrSize );    
  }

#ifdef PCSC_DEBUG
    DebugLogA("SCardGetStatusChange: Event Loop End\n", __FILE__, __LINE__);
#endif

    rgReaderStates->cbAtr        = dwAtrSize;
    rgReaderStates->dwEventState = dwEventState;
    rContext->dwBlockStatus      = BLOCK_STATUS_RESUME;
    memcpy( rgReaderStates->rgbAtr, pbAtr, MAX_ATR_SIZE);

    SCardDisconnect( hCard, SCARD_UNPOWER_CARD );
    return SCARD_S_SUCCESS;
}

LONG SCardTransmit( SCARDHANDLE hCard, LPCSCARD_IO_REQUEST pioSendPci,
                    LPCBYTE pbSendBuffer, DWORD cbSendLength,
                    LPSCARD_IO_REQUEST pioRecvPci, LPBYTE pbRecvBuffer, 
		    LPDWORD pcbRecvLength ) {
 
  LONG rv;
  struct ReaderContext *rContext; 
  struct SCARD_IO_HEADER sSendPci, sRecvPci;
  DWORD dwRxLength;

  dwRxLength     = *pcbRecvLength;
  *pcbRecvLength = 0;

  if ( hCard == 0 ) { return SCARD_E_INVALID_HANDLE; }

  /* Make sure no one has a lock on this reader */
  if ( (rv = RFCheckSharing(hCard)) != SCARD_S_SUCCESS ) {
    return rv;
  }

  rv = RFReaderInfoById( hCard, &rContext );   
  if ( rv != SCARD_S_SUCCESS ) { return rv; }

  if ( pioSendPci->dwProtocol != rContext->dwProtocol ) {
    return SCARD_E_PROTO_MISMATCH;
  }

  if ( cbSendLength > MAX_BUFFER_SIZE ) {
    return SCARD_E_INSUFFICIENT_BUFFER;
  }

  /* Quick fix: PC/SC starts at 1 for bit masking
     but the IFD_Handler just wants 0 or 1        */

  if ( pioSendPci->dwProtocol == SCARD_PROTOCOL_T0 ) {
    sSendPci.Protocol = 0;
  } else if ( pioSendPci->dwProtocol == SCARD_PROTOCOL_T1 ) {
    sSendPci.Protocol = 1;
  }

  sSendPci.Length   = pioSendPci->cbPciLength;

  /* Calculate the return length */

  if ( cbSendLength > 5 ) {
    dwRxLength = 2;                      /* CLA P1 P2 Lc Data SW1 SW2 */
  } else {
    dwRxLength = pbSendBuffer[4] + 2;    /* CLA P1 P2 Lr Data SW1 SW2 */
  }

  if ( dwRxLength > MAX_BUFFER_SIZE ) {
    return SCARD_E_INSUFFICIENT_BUFFER;
  }

#ifdef PCSC_DEBUG
  DebugLogB("SCardTransmit: Send Protocol: %d\n", sSendPci.Protocol, 
	    __FILE__, __LINE__);
#endif
  
  rv = IFDTransmit( rContext, sSendPci, pbSendBuffer, cbSendLength,
		    pbRecvBuffer, &dwRxLength, &sRecvPci );

  pioRecvPci->dwProtocol  = sRecvPci.Protocol;
  pioRecvPci->cbPciLength = sRecvPci.Length;

  if ( rv != IFD_SUCCESS ) { 
    *pcbRecvLength = 0;
    return SCARD_E_NOT_TRANSACTED;
  }

  if ( (dwRxLength <= MAX_BUFFER_SIZE) && (dwRxLength >= 0) ) 
    {
      *pcbRecvLength          = dwRxLength;
    } else {
      *pcbRecvLength          = 0;
      return SCARD_E_NOT_TRANSACTED;
    }  

  return SCARD_S_SUCCESS;
}

LONG SCardListReaders( SCARDCONTEXT hContext, LPCSTR mszGroups, 
		       LPSTR mszReaders, LPDWORD pcchReaders ) {
  LONG rv;
  rv = RFListReaders( mszReaders, pcchReaders );
  return rv;
}

LONG SCardCancel( SCARDCONTEXT hContext ) {

  LONG rv;

  /* Issue a cancel request to the
     GetStatusChange function      */

  rv = RFUnblockContext( hContext );
  return rv;
}

/* The following functions extend the Winscard interface
   to add support for memory type storage smart cards    */

LONG SCardReadMemory( SCARDHANDLE hCard, DWORD dwCardType, DWORD dwCardSize, 
                      DWORD dwAddress, LPBYTE pbBuffer, DWORD dwLength ) {
  LONG rv;
  struct ReaderContext *rContext; 

  if ( hCard == 0 ) { return SCARD_E_INVALID_HANDLE; }

  /* Make sure no one has a lock on this reader */
  if ( (rv = RFCheckSharing(hCard)) != SCARD_S_SUCCESS ) {
    return rv;
  }

  rv = RFReaderInfoById( hCard, &rContext );   
  if ( rv != SCARD_S_SUCCESS ) { return rv; }

  if ( dwLength > MAX_BUFFER_SIZE ) {
    return SCARD_E_INSUFFICIENT_BUFFER;
  }

  rv = IFDReadICC( rContext, dwCardType, dwCardSize, dwAddress, 
		   pbBuffer, dwLength );

  if ( rv == IFD_SUCCESS ) { 
    rv = SCARD_S_SUCCESS;
   } else {
    rv = SCARD_E_NOT_READY;
   }

  return rv;
}

LONG SCardWriteMemory( SCARDHANDLE hCard, DWORD dwCardType, DWORD dwCardSize,
                       DWORD dwAddress, LPCBYTE pcbBuffer, DWORD dwLength ) {
  LONG rv;
  struct ReaderContext *rContext; 

  if ( hCard == 0 ) { return SCARD_E_INVALID_HANDLE; }

  /* Make sure no one has a lock on this reader */
  if ( (rv = RFCheckSharing(hCard)) != SCARD_S_SUCCESS ) {
    return rv;
  }

  rv = RFReaderInfoById( hCard, &rContext );   
  if ( rv != SCARD_S_SUCCESS ) { return rv; }

  if ( dwLength > MAX_BUFFER_SIZE ) {
    return SCARD_E_INSUFFICIENT_BUFFER;
  }

  rv = IFDWriteICC( rContext, dwCardType, dwCardSize, dwAddress, 
		    pcbBuffer, dwLength );

  if ( rv == IFD_SUCCESS ) { 
    rv = SCARD_S_SUCCESS;
   } else {
    rv = SCARD_E_NOT_READY;
   }

  return rv;
}
