/******************************************************************

	MUSCLE SmartCard Development ( http://www.linuxnet.com )
	    Title  : winscard.c
	    Package: pcsc lite
            Author : David Corcoran
            Date   : 10/27/99
            Purpose: This handles smartcard reader communications. 
	             This file forwards requests over RPC.

********************************************************************/

#include <winscard.h>
#include <string.h>
#include "winscard_rpc.h"

static struct _psChannelMap {
  SCARDCONTEXT hContext;
  SCARDHANDLE  hCard;
  SCARDHANDLE  hCardTable[PCSCLITE_MAX_CONTEXTS];
  DWORD        dwTableSize;
  CLIENT       *psChannel;
  LPSTR        psHost;
} *psChannelMap[PCSCLITE_MAX_CHANNELS];

static DWORD dwChannelNum = 0;

SCARD_IO_REQUEST g_rgSCardT0Pci, g_rgSCardT1Pci, g_rgSCardRawPci;   

/* These functions allow this file to map SCARDCONTEXT's and
   SCARDHANDLE's over to their appropriate CLIENT structure
   for RPC.  The Add and Remove functions allow the addition
   of multiple SCARDHANDLE's to one SCARDCONTEXT.             */

static LONG SCardAddChannel    ( SCARDCONTEXT );
static LONG SCardRemoveChannel ( SCARDCONTEXT );
static LONG SCardGetContext    ( SCARDCONTEXT, SCARDHANDLE );
static LONG SCardAddContext    ( SCARDCONTEXT, SCARDHANDLE );
static LONG SCardRemoveContext ( SCARDHANDLE );

LONG SCardEstablishContext( DWORD dwScope, LPCVOID pvReserved1, 
			    LPCVOID pvReserved2, LPSCARDCONTEXT phContext) {

  LPSTR host;
  LONG liIndex, rv;
  CLIENT *clnt;
  establish_struct *result_1;
  establish_struct scardestablishcontext_1_arg;      

  /* Get the services host */
  if ( pvReserved1 == 0 ) {
    host = strdup("localhost");
  } else {
    host = strdup((LPSTR)pvReserved1);
  }  

  *phContext = 0;

  g_rgSCardT0Pci.dwProtocol  = SCARD_PROTOCOL_T0;
  g_rgSCardT1Pci.dwProtocol  = SCARD_PROTOCOL_T1;
  g_rgSCardRawPci.dwProtocol = SCARD_PROTOCOL_RAW;

  clnt = clnt_create(host, DISPLAY_PRG, DISPLAY_VER, "tcp");
  
  if ( clnt != 0 ) {
    result_1 = scardestablishcontext_1(&scardestablishcontext_1_arg, clnt);
  } 


  if ( clnt == 0 || result_1 == 0 || result_1->rv != SCARD_S_SUCCESS ) {
    clnt_pcreateerror(host);  
    free ( host );
    return SCARD_E_NO_SERVICE;
  }

  rv = SCardAddChannel( result_1->phContext );

  if ( rv != SCARD_S_SUCCESS ) {
    free ( host );
    return rv;
  }

  liIndex = SCardGetContext( result_1->phContext, 0 );
  psChannelMap[liIndex]->psHost    = strdup(host);
  psChannelMap[liIndex]->psChannel = clnt;
  *phContext                       = result_1->phContext;

  
  free ( host );
  return SCARD_S_SUCCESS;
}

LONG SCardReleaseContext( SCARDCONTEXT hContext ) {

  LONG liIndex;
  CLIENT *clnt;
  release_struct  *result_2;
  release_struct  scardreleasecontext_1_arg;   

  liIndex = SCardGetContext( hContext , 0 );

  if ( liIndex < 0 ) {
    return SCARD_E_INVALID_HANDLE;
  }

  scardreleasecontext_1_arg.hContext = hContext;

  clnt     = psChannelMap[liIndex]->psChannel;
  result_2 = scardreleasecontext_1(&scardreleasecontext_1_arg, clnt);

  SCardRemoveChannel( hContext );
  clnt_destroy( clnt );

  return SCARD_S_SUCCESS;
}

LONG SCardSetTimeout( SCARDCONTEXT hContext, DWORD dwTimeout ) {

  LONG rv;
  CLIENT *clnt;
  LONG liIndex;
  struct timeval rpctime;

  liIndex = SCardGetContext( hContext, 0 );

  if ( liIndex < 0 ) {
    return SCARD_E_INVALID_HANDLE;
  }

  rpctime.tv_sec  = dwTimeout;
  rpctime.tv_usec = 0;

  clnt     = psChannelMap[liIndex]->psChannel;  
  clnt_control( clnt, CLSET_TIMEOUT, (char *)&rpctime );

  return SCARD_S_SUCCESS;
}

LONG SCardConnect( SCARDCONTEXT hContext, LPCSTR szReader, DWORD dwShareMode,
                   DWORD dwPreferredProtocols, LPSCARDHANDLE phCard,
                   LPDWORD pdwActiveProtocol ) {              
  LONG rv;
  CLIENT *clnt;
  LONG liIndex;
  connect_struct  *result_3;
  connect_struct  scardconnect_1_arg;   

  liIndex = SCardGetContext( hContext, 0 );

  if ( liIndex < 0 ) {
    return SCARD_E_INVALID_HANDLE;
  }

  scardconnect_1_arg.hContext                = hContext;
  scardconnect_1_arg.szReader.RPC_LPCSTR_val = strdup((LPSTR)szReader);
  scardconnect_1_arg.szReader.RPC_LPCSTR_len = strlen(szReader) + 1;
  scardconnect_1_arg.dwShareMode             = dwShareMode; 
  scardconnect_1_arg.dwPreferredProtocols    = dwPreferredProtocols;
  scardconnect_1_arg.phCard                  = *phCard;
  scardconnect_1_arg.pdwActiveProtocol       = *pdwActiveProtocol;

  clnt     = psChannelMap[liIndex]->psChannel;
  result_3 = scardconnect_1(&scardconnect_1_arg, clnt);  

  free(scardconnect_1_arg.szReader.RPC_LPCSTR_val);

  if ( result_3 == 0 ) {
    *phCard = 0;
    clnt_perror(clnt, "call failed:"); 
    return SCARD_F_COMM_ERROR;
  }

  *phCard                                 = result_3->phCard;
  *pdwActiveProtocol                      = result_3->pdwActiveProtocol;


  if ( result_3->rv == SCARD_S_SUCCESS ) {
    SCardAddContext( hContext, result_3->phCard );
  }

  return result_3->rv;
}

LONG SCardDisconnect( SCARDHANDLE hCard, DWORD dwDisposition ) {
  
  LONG rv;
  CLIENT *clnt;
  LONG liIndex;
  disconnect_struct  *result_4;
  disconnect_struct  scarddisconnect_1_arg;
  
  liIndex = SCardGetContext( 0, hCard );

  if ( liIndex < 0 ) {
    return SCARD_E_INVALID_HANDLE;
  }

  scarddisconnect_1_arg.hCard         = hCard;
  scarddisconnect_1_arg.dwDisposition = dwDisposition; 

  clnt     = psChannelMap[liIndex]->psChannel;
  result_4 = scarddisconnect_1(&scarddisconnect_1_arg, clnt);

  if ( result_4 == 0 ) {
    clnt_perror(clnt, "call failed:"); 
    return SCARD_F_COMM_ERROR;
  }

  SCardRemoveContext( hCard );
  
  return result_4->rv;
}

LONG SCardBeginTransaction( SCARDHANDLE hCard ) {

  LONG rv;
  CLIENT *clnt;
  LONG liIndex;
  begin_struct  *result_5;
  begin_struct  scardbegintransaction_1_arg;  
  
  liIndex = SCardGetContext( 0, hCard );

  if ( liIndex < 0 ) {
    return SCARD_E_INVALID_HANDLE;
  }
  
  scardbegintransaction_1_arg.hCard = hCard;

  clnt     = psChannelMap[liIndex]->psChannel;
  result_5 = scardbegintransaction_1(&scardbegintransaction_1_arg, clnt);

  if ( result_5 == 0 ) {
    clnt_perror(clnt, "call failed:"); 
    return SCARD_F_COMM_ERROR;
  }

  return result_5->rv;
}

LONG SCardEndTransaction( SCARDHANDLE hCard, DWORD dwDisposition ) {

  LONG rv;
  CLIENT *clnt;
  LONG liIndex;
  end_struct  *result_6;
  end_struct  scardendtransaction_1_arg;  
  
  liIndex = SCardGetContext( 0, hCard );

  if ( liIndex < 0 ) {
    return SCARD_E_INVALID_HANDLE;
  }
  
  scardendtransaction_1_arg.hCard         = hCard;
  scardendtransaction_1_arg.dwDisposition = dwDisposition;

  clnt     = psChannelMap[liIndex]->psChannel;
  result_6 = scardendtransaction_1(&scardendtransaction_1_arg, clnt);

  if ( result_6 == 0 ) {
    clnt_perror(clnt, "call failed:"); 
    return SCARD_F_COMM_ERROR;
  }

  return result_6->rv;
}

LONG SCardCancelTransaction( SCARDHANDLE hCard ) {

  LONG rv;
  CLIENT *clnt;
  LONG liIndex;
  cancel_struct  *result_7;
  cancel_struct  scardcanceltransaction_1_arg;  
  
  liIndex = SCardGetContext( 0, hCard );

  if ( liIndex < 0 ) {
    return SCARD_E_INVALID_HANDLE;
  }
  
  scardcanceltransaction_1_arg.hCard = hCard;

  clnt     = psChannelMap[liIndex]->psChannel;
  result_7 = scardcanceltransaction_1(&scardcanceltransaction_1_arg, clnt);

  if ( result_7 == 0 ) {
    clnt_perror(clnt, "call failed:"); 
    return SCARD_F_COMM_ERROR;
  }

  return result_7->rv;
}

LONG SCardStatus( SCARDHANDLE hCard, LPSTR mszReaderNames, 
		  LPDWORD pcchReaderLen, LPDWORD pdwState, 
		  LPDWORD pdwProtocol, LPBYTE pbAtr,
                  LPDWORD pcbAtrLen ) {
  
  LONG rv;
  LONG liIndex;
  CLIENT *clnt;
  status_struct *result_8;
  status_struct scardstatus_1_arg;

  liIndex = SCardGetContext( 0, hCard );

  if ( liIndex < 0 ) {
    mszReaderNames[0] = 0; *pcchReaderLen    = 0;
    *pcbAtrLen        = 0; *pdwState         = 0;
    *pdwProtocol      = 0;
    return SCARD_E_INVALID_HANDLE;
  }

  scardstatus_1_arg.hCard                        = hCard;
  scardstatus_1_arg.mszReaderNames.RPC_LPSTR_val = 0;
  scardstatus_1_arg.mszReaderNames.RPC_LPSTR_len = 0; 
  scardstatus_1_arg.pbAtr.RPC_LPBYTE_val         = 0;
  scardstatus_1_arg.pbAtr.RPC_LPBYTE_len         = 0;
  scardstatus_1_arg.pcchReaderLen                = *pcchReaderLen;

  clnt     = psChannelMap[liIndex]->psChannel;
  result_8 = scardstatus_1(&scardstatus_1_arg, clnt);

  if ( result_8->rv == SCARD_S_SUCCESS ) {
    memcpy( mszReaderNames, result_8->mszReaderNames.RPC_LPSTR_val,
	    result_8->mszReaderNames.RPC_LPSTR_len );
    memcpy( pbAtr, result_8->pbAtr.RPC_LPBYTE_val, 
	    result_8->pbAtr.RPC_LPBYTE_len );
  }

  *pcchReaderLen = result_8->pcchReaderLen;
  *pdwState       = result_8->pdwState;
  *pdwProtocol    = result_8->pdwProtocol;
  *pcbAtrLen     = result_8->pcbAtrLen;

  return result_8->rv;
}

LONG SCardGetStatusChange( SCARDCONTEXT hContext, DWORD dwTimeout, 
			   LPSCARD_READERSTATE_A rgReaderStates, 
                           DWORD cReaders ) {

  LONG rv;
  CLIENT *clnt;
  LONG liIndex;
  statuschange_struct  *result_9;
  statuschange_struct  scardgetstatuschange_1_arg;   

  liIndex = SCardGetContext( hContext, 0 );

  if ( liIndex < 0 ) {
    return SCARD_E_INVALID_HANDLE;
  }
  
  scardgetstatuschange_1_arg.hContext  = hContext;
  scardgetstatuschange_1_arg.dwTimeout = dwTimeout;
  scardgetstatuschange_1_arg.cReaders  = cReaders;

  scardgetstatuschange_1_arg.rgReaderStates.szReader.RPC_LPCSTR_val = 
    strdup((LPSTR)rgReaderStates->szReader);
  scardgetstatuschange_1_arg.rgReaderStates.szReader.RPC_LPCSTR_len =
    strlen(rgReaderStates->szReader) + 1;
  scardgetstatuschange_1_arg.rgReaderStates.dwCurrentState = 
    rgReaderStates->dwCurrentState;
  scardgetstatuschange_1_arg.rgReaderStates.dwEventState = 
    rgReaderStates->dwEventState;
  scardgetstatuschange_1_arg.rgReaderStates.cbAtr                 = 0;
  scardgetstatuschange_1_arg.rgReaderStates.rgbAtr.RPC_LPBYTE_len = 0;;

  clnt     = psChannelMap[liIndex]->psChannel;
  result_9 = scardgetstatuschange_1(&scardgetstatuschange_1_arg, clnt);

  if ( result_9 == 0 ) {
    free(scardgetstatuschange_1_arg.rgReaderStates.szReader.RPC_LPCSTR_val);
    clnt_perror(clnt, "call failed:"); 
    return SCARD_F_COMM_ERROR;
  }

  rgReaderStates->dwCurrentState = result_9->rgReaderStates.dwCurrentState;
  rgReaderStates->dwEventState   = result_9->rgReaderStates.dwEventState;
  rgReaderStates->cbAtr          = result_9->rgReaderStates.cbAtr; 

  memcpy( rgReaderStates->rgbAtr, 
	  result_9->rgReaderStates.rgbAtr.RPC_LPBYTE_val, 
	  result_9->rgReaderStates.rgbAtr.RPC_LPBYTE_len );

  free(scardgetstatuschange_1_arg.rgReaderStates.szReader.RPC_LPCSTR_val);

  return result_9->rv;
}

LONG SCardTransmit( SCARDHANDLE hCard, LPCSCARD_IO_REQUEST pioSendPci,
                    LPCBYTE pbSendBuffer, DWORD cbSendLength,
                    LPSCARD_IO_REQUEST pioRecvPci, LPBYTE pbRecvBuffer, 
		    LPDWORD pcbRecvLength ) {
 
  LONG rv;
  CLIENT *clnt;
  LONG liIndex;
  transmit_struct  *result_10;
  transmit_struct  scardtransmit_1_arg;  
  
  liIndex = SCardGetContext( 0, hCard );
  
  if ( liIndex < 0 ) {
    *pcbRecvLength = 0;
    return SCARD_E_INVALID_HANDLE;
  }

  scardtransmit_1_arg.hCard                       = hCard;
  scardtransmit_1_arg.cbSendLength                = cbSendLength;
  scardtransmit_1_arg.pcbRecvLength               = *pcbRecvLength;
  scardtransmit_1_arg.pbSendBuffer.RPC_LPCBYTE_len = cbSendLength;
  scardtransmit_1_arg.pbRecvBuffer.RPC_LPBYTE_len = 0;

  scardtransmit_1_arg.pbSendBuffer.RPC_LPCBYTE_val = (char *)malloc
    (sizeof(char)*cbSendLength);

  memcpy( &scardtransmit_1_arg.pioSendPci, pioSendPci, 
	  sizeof(SCARD_IO_REQUEST) );
  
  memcpy( scardtransmit_1_arg.pbSendBuffer.RPC_LPCBYTE_val, 
	  pbSendBuffer, cbSendLength );

  clnt      = psChannelMap[liIndex]->psChannel;
  result_10 = scardtransmit_1(&scardtransmit_1_arg, clnt);
  
  free ( scardtransmit_1_arg.pbSendBuffer.RPC_LPCBYTE_val );

  if ( result_10 == 0 ) {
    clnt_perror(clnt, "call failed:"); 
    return SCARD_F_COMM_ERROR;
  } 

  *pcbRecvLength = result_10->pcbRecvLength;

  memcpy( pbRecvBuffer, result_10->pbRecvBuffer.RPC_LPBYTE_val, 
	  result_10->pcbRecvLength );

  memcpy( pioRecvPci, &result_10->pioRecvPci, 
	  sizeof(SCARD_IO_REQUEST) );

  return result_10->rv;
}

LONG SCardListReaders( SCARDCONTEXT hContext, LPCSTR mszGroups, 
		       LPSTR mszReaders, LPDWORD pcchReaders ) {

  LONG rv;
  CLIENT *clnt;
  LONG liIndex;
  LPSTR mszGroupsB;
  DWORD dwGroupsLen, dwReadersLen;
  list_struct  *result_11;
  list_struct  scardlistreaders_1_arg; 
  
  liIndex = SCardGetContext( hContext, 0 );
  
  if ( liIndex < 0 ) {
    *pcchReaders = 0;
    return SCARD_E_INVALID_HANDLE;
  }

  /*
  if ( mszGroups == 0 ) {
    dwGroupsLen = 0;
  } else {
    dwGroupsLen = strlen(mszGroups)+1;
  }
  */

  /* Groups are not currently used */

  mszGroupsB  = 0;
  dwGroupsLen = 0;  

  if ( mszReaders == 0 ) {
    dwReadersLen = 0;
  } else {
    dwReadersLen = strlen(mszReaders)+1;
  }

  scardlistreaders_1_arg.hContext                  = hContext;
  scardlistreaders_1_arg.mszGroups.RPC_LPCSTR_val  = (LPSTR)mszGroupsB;
  scardlistreaders_1_arg.mszGroups.RPC_LPCSTR_len  = dwGroupsLen;
  scardlistreaders_1_arg.mszReaders.RPC_LPSTR_val  = mszReaders;
  scardlistreaders_1_arg.mszReaders.RPC_LPSTR_len  = dwReadersLen; 
  scardlistreaders_1_arg.pcchReaders               = *pcchReaders;

  clnt       = psChannelMap[liIndex]->psChannel;
  result_11  = scardlistreaders_1(&scardlistreaders_1_arg, clnt); 

  if ( result_11 == 0 ) {
    clnt_perror(clnt, "call failed:");
    *pcchReaders = 0;
    return SCARD_F_COMM_ERROR;
  } 

  if ( mszReaders != 0 ) {
    memcpy( mszReaders, result_11->mszReaders.RPC_LPSTR_val, 
	    *pcchReaders );
  }

  *pcchReaders = result_11->pcchReaders;
  return result_11->rv;
}

LONG SCardCancel( SCARDCONTEXT hContext ) {

  LONG rv;
  CLIENT *clnt;
  LONG liIndex;
  cancellock_struct  *result_12;
  cancellock_struct  scardcancel_1_arg;  
 
  liIndex = SCardGetContext( hContext, 0 );
  
  if ( liIndex < 0 ) {
    return SCARD_E_INVALID_HANDLE;
  }

  scardcancel_1_arg.hContext = hContext;

  clnt      = psChannelMap[liIndex]->psChannel;
  result_12 = scardcancel_1(&scardcancel_1_arg, clnt);  
  
  if ( result_12 == 0 ) {
    clnt_perror(clnt, "call failed:"); 
    return SCARD_F_COMM_ERROR;
  } 

  return result_12->rv;
}

/* The following functions extend the Winscard interface
   to add support for memory type storage smart cards    */

LONG SCardReadMemory( SCARDHANDLE hCard, DWORD dwCardType, DWORD dwCardSize, 
                      DWORD dwAddress, LPBYTE pbBuffer, DWORD dwLength ) {

  LONG rv;
  CLIENT *clnt;
  LONG liIndex;
  read_struct  *result_13;
  read_struct  scardreadmemory_1_arg;   

 
  liIndex = SCardGetContext( 0, hCard );
  
  if ( liIndex < 0 ) {
    return SCARD_E_INVALID_HANDLE;
  }

  scardreadmemory_1_arg.hCard                   = hCard;
  scardreadmemory_1_arg.dwCardType              = dwCardType;
  scardreadmemory_1_arg.dwCardSize              = dwCardSize;
  scardreadmemory_1_arg.dwAddress               = dwAddress;
  scardreadmemory_1_arg.dwLength                = dwLength;
  scardreadmemory_1_arg.pbBuffer.RPC_LPBYTE_len = dwLength;

  clnt      = psChannelMap[liIndex]->psChannel;
  result_13 = scardreadmemory_1(&scardreadmemory_1_arg, clnt);

  if ( result_13 == 0 ) {
    clnt_perror(clnt, "call failed:"); 
    return SCARD_F_COMM_ERROR;
  } 
  
  memcpy( pbBuffer, &result_13->pbBuffer, result_13->dwLength );

  return result_13->rv;  
}

LONG SCardWriteMemory( SCARDHANDLE hCard, DWORD dwCardType, DWORD dwCardSize,
                       DWORD dwAddress, LPCBYTE pcbBuffer, DWORD dwLength ) {

  LONG rv;
  CLIENT *clnt;
  LONG liIndex;
  write_struct  *result_14;
  write_struct  scardwritememory_1_arg;   

 
  liIndex = SCardGetContext( 0, hCard );
  
  if ( liIndex < 0 ) {
    return SCARD_E_INVALID_HANDLE;
  }

  scardwritememory_1_arg.hCard      = hCard;
  scardwritememory_1_arg.dwCardType = dwCardType;
  scardwritememory_1_arg.dwCardSize = dwCardSize;
  scardwritememory_1_arg.dwAddress  = dwAddress;
  scardwritememory_1_arg.dwLength   = dwLength;

  memcpy( &scardwritememory_1_arg.pcbBuffer, pcbBuffer, dwLength );

  clnt      = psChannelMap[liIndex]->psChannel;
  result_14 = scardwritememory_1(&scardwritememory_1_arg, clnt);

  if ( result_14 == 0 ) {
    clnt_perror(clnt, "call failed:"); 
    return SCARD_F_COMM_ERROR;
  } 

  return result_14->rv;  
}

LONG SCardGetContext( SCARDCONTEXT hContext, SCARDHANDLE hCard ) {

  int i, p;

  if ( hCard == 0 ) {
    for ( i=0; i<dwChannelNum; i++ ) {
      if ( hContext == psChannelMap[i]->hContext 
	   && psChannelMap[i] != 0 ) {
	return i;
      }
    }
  } else {
    for ( i=0; i<dwChannelNum; i++ ) {
      for (p=0; p < psChannelMap[i]->dwTableSize; p++) {
	if ( hCard == psChannelMap[i]->hCardTable[p] ) {
	  return i;
	}
      }
    }
  }    
  
  return -1;
}

LONG SCardAddChannel( SCARDCONTEXT hContext ) {
  
  int i;

  if ( dwChannelNum >= PCSCLITE_MAX_CHANNELS ) {
    return SCARD_E_INSUFFICIENT_BUFFER;
  }
  
  /* See if channel already exists */

  for ( i=0; i < dwChannelNum; i++ ) {
    if ( hContext == psChannelMap[i]->hContext ) {
      return SCARD_S_SUCCESS;
    }
  }

  psChannelMap[dwChannelNum] = (struct _psChannelMap *)calloc
    (1,sizeof(struct _psChannelMap));
  
  psChannelMap[dwChannelNum]->hContext    = hContext;
  psChannelMap[dwChannelNum]->dwTableSize = 0;
  psChannelMap[dwChannelNum]->psChannel   = 0;
  psChannelMap[dwChannelNum]->psHost      = 0;

  dwChannelNum += 1;

  return SCARD_S_SUCCESS;
}

LONG SCardRemoveChannel( SCARDCONTEXT hContext ) {
  
  LONG rv, liIndex;
  int i;
  
  liIndex = SCardGetContext( hContext, 0 );
  
  if ( liIndex != -1 ) {
    
    /* The server will do the disconnects
       and cancel requests.               */
    
    if ( psChannelMap[liIndex]->psHost != 0 ) {
      free( psChannelMap[liIndex]->psHost );
    }

    free( psChannelMap[liIndex] );
    dwChannelNum          -= 1;
    psChannelMap[liIndex]  = psChannelMap[dwChannelNum];
    return SCARD_S_SUCCESS;
  } 
  
  return SCARD_E_INVALID_HANDLE;
}

LONG SCardAddContext( SCARDCONTEXT hContext, SCARDHANDLE hCard ) {

  /* Add a SCARDHANDLE to the current Channel
     Increment the Table Size                 */


  LONG rv, liIndex;
  int i;
  
  liIndex = SCardGetContext( hContext, 0 );
  
  if ( liIndex != -1 ) {

    /* See if context already exists */

    for ( i=0; i < psChannelMap[liIndex]->dwTableSize; i++ ) {
      if ( hCard == psChannelMap[liIndex]->hCardTable[i] ) {
	return SCARD_S_SUCCESS;
      }
    }

    if ( psChannelMap[liIndex]->dwTableSize >= PCSCLITE_MAX_CONTEXTS ) {
      return SCARD_E_INSUFFICIENT_BUFFER;
    }
    
    psChannelMap[liIndex]->hCardTable[psChannelMap[liIndex]->
				     dwTableSize] = hCard;
    psChannelMap[liIndex]->dwTableSize += 1;
    return SCARD_S_SUCCESS;
  }

  return SCARD_E_INVALID_HANDLE;
}

LONG SCardRemoveContext ( SCARDHANDLE hCard ) {

  int i,p;

  /* Lookup the hCard in all the Context Structures
     Delete it and decrement the counter.
     Note: There is a very low probability that two
     Channels might have the same hCard value in
     which this might fail.  FIX : There is a 1:65536
     chance of this happenning only if connections
     are made to more than 1 resource manager        */

  for (i=0; i < dwChannelNum; i++) {
    for (p=0; p < psChannelMap[i]->dwTableSize; p++) {
      if ( psChannelMap[i]->hCardTable[p] == hCard ) {
	psChannelMap[i]->dwTableSize -= 1;
	psChannelMap[i]->hCardTable[p] = 
	  psChannelMap[i]->hCardTable[psChannelMap[i]->dwTableSize];
      }
    }
  }

  return SCARD_S_SUCCESS;
}
