/******************************************************************

	MUSCLE SmartCard Development ( http://www.linuxnet.com )
	    Title  : readerfactory.c
	    Package: pcsc lite
            Author : David Corcoran
            Date   : 7/27/99
            Purpose: This keeps track of a list of currently 
	             available reader structures.

********************************************************************/

#include <readerfactory.h>
#include <dyn_generic.h>
#include <sys_generic.h>
/* FIX :: This should go */
#include <fcntl.h>
#include <debuglog.h>
#include <stdlib.h>
#include <string.h>

static struct ReaderContext *sContexts[PCSCLITE_MAX_CONTEXTS];
static DWORD *dwNumContexts = 0;

LONG RFAddReader( LPSTR lpcReader, DWORD dwPort, LPSTR lpcLibrary ) {
  
  LONG rv;
  static int fd = 0;
  int i, psize;

  if ( lpcReader == 0 || lpcLibrary == 0 ) {
    return SCARD_E_INVALID_VALUE;
  }

  if ( dwNumContexts != 0 ) {
    for ( i=0; i < *dwNumContexts; i++ ) {
      if ((strcmp(lpcReader, (sContexts[i])->lpcReader) == 0)) {
	return SCARD_E_INVALID_VALUE;
      }
    }
  }

#ifdef PCSCLITE_LOCAL_LINK
  /* Here a simple malloc can be done since shared memory is 
     not needed to communicate between multiple client uses   */

  if ( fd == 0 ) { 
    dwNumContexts  = (DWORD *)malloc(sizeof(DWORD));
    *dwNumContexts = 0;
    
    for ( i=0; i < PCSCLITE_MAX_CONTEXTS; i++ ) {
      sContexts[i] = (struct ReaderContext *)malloc
	(sizeof(struct ReaderContext));
    }
    
    fd = 1;
  }

#else
  /* Here we must map the sContexts over to a memory map so when
     the server is forked we can continue to keep track of the
     reader's state.
  */

  if ( fd == 0 ) {    
    fd = SYS_OpenFile("./.pcscrpctmp", O_RDWR | O_CREAT, 0644);
    
    if ( fd < 0 ) {
      DebugLogA("Cannot Open Shared File\n", __FILE__, __LINE__);
      exit(1);
    } else {

      /* FIX :: Abstract this system specific call */
      psize = getpagesize();

      SYS_SeekFile( fd, psize * (PCSCLITE_MAX_CONTEXTS + 1) );
      
      SYS_WriteFile( fd, "", 1 );
      dwNumContexts  = (DWORD *)SYS_MemoryMap( sizeof(DWORD), fd, 0 );
      *dwNumContexts = 0;

      for ( i=0; i < PCSCLITE_MAX_CONTEXTS; i++ ) {
	sContexts[i] = (struct ReaderContext *)
	  SYS_MemoryMap( sizeof(struct ReaderContext)*PCSCLITE_MAX_CONTEXTS, 
			 fd, (i+1)*(psize) );
      }
    }
  }

#endif 


  strcpy((sContexts[*dwNumContexts])->lpcReader,  lpcReader);
  strcpy((sContexts[*dwNumContexts])->lpcLibrary, lpcLibrary);
  (sContexts[*dwNumContexts])->dwPort          = dwPort;
  (sContexts[*dwNumContexts])->dwStatus        = 0;
  (sContexts[*dwNumContexts])->dwBlockStatus   = BLOCK_STATUS_RESUME;
  (sContexts[*dwNumContexts])->dwContexts      = 0;
  (sContexts[*dwNumContexts])->dwSlot          = 1;
  (sContexts[*dwNumContexts])->dwLockId        = 0;
  (sContexts[*dwNumContexts])->vHandle         = 0;
  (sContexts[*dwNumContexts])->dwIdentity      =
    ((DWORD)(sContexts[*dwNumContexts])) << (sizeof(DWORD) / 2) * 8;

  /* Check to see if an existing reader is specified on the
     same port and uses the same library - if that is true
     then it must be a reader that supports multiple slots  */
  if ( dwNumContexts != 0 ) {
    for ( i=0; i < *dwNumContexts; i++ ) {
      if (((strcmp(sContexts[i]->lpcLibrary, lpcLibrary) == 0)&&
	   (sContexts[i])->dwPort == dwPort)) {
	if ( (sContexts[i])->dwSlot == (sContexts[*dwNumContexts])->dwSlot ) {
	  (sContexts[*dwNumContexts])->dwSlot += 1;
	}     
      }
    }
  }

  *dwNumContexts += 1;
  return SCARD_S_SUCCESS;
}

LONG RFRemoveReader( LPSTR lpcReader ) {
  
  LONG rv;
  int i;

  if ( lpcReader == 0 ) {
    return SCARD_E_INVALID_VALUE;
  }

  for ( i=0; i < *dwNumContexts; i++ ) {
    if (strcmp(lpcReader, (sContexts[i])->lpcReader) == 0) {
      free((sContexts[i])->lpcReader);
      free((sContexts[i])->lpcLibrary);
      free(sContexts[i]);
      *dwNumContexts -= 1;
      return SCARD_S_SUCCESS;
    }
  }

  return SCARD_E_INVALID_VALUE;
}

LONG RFListReaders( LPSTR lpcReaders, LPDWORD pdwReaderNum ) {

  LONG rv;
  DWORD dwCSize;
  int i,p;
  LPSTR lpcTReaders;

  dwCSize = 0;

  /* Ignore the groups for now, return all readers */
  for ( i=0; i < *dwNumContexts; i++ ) {
    dwCSize += strlen( (sContexts[i])->lpcReader ) + 1;
  }
  
  /* Added for extra NULL byte on MultiString */
  dwCSize += 1;  

  /* If lpcReaders is not allocated then just */
  /* return the amount needed to allocate     */
  if ( lpcReaders == 0 ) {
    *pdwReaderNum = dwCSize;
    return SCARD_S_SUCCESS;
  }
 
  if ( *pdwReaderNum < dwCSize ) {
    return SCARD_E_INSUFFICIENT_BUFFER;
  }

  *pdwReaderNum = dwCSize;
  lpcTReaders   = lpcReaders;
  p = 0;
  
  /* Creating MultiString */
  for ( i=0; i < *dwNumContexts; i++ ) {
    strcpy(&lpcTReaders[p], (sContexts[i])->lpcReader );
    p += strlen((sContexts[i])->lpcReader);              /* Copy       */
    lpcTReaders[p] = 0;                                  /* Add NULL   */
    p += 1;                                              /* Move on    */
  }
  
  lpcTReaders[p] = 0;                                    /* Add NULL   */

  return SCARD_S_SUCCESS;
}  

LONG RFReaderInfo( LPSTR lpcReader, struct ReaderContext** sReader ) {
  
  LONG rv;
  int i;

  if ( lpcReader == 0 ) {
    return SCARD_E_INVALID_VALUE;
  }

  for ( i=0; i < *dwNumContexts; i++ ) {
    if (strcmp(lpcReader, (sContexts[i])->lpcReader) == 0) {
      *sReader = sContexts[i];
      return SCARD_S_SUCCESS;
    }
  }

  return SCARD_E_INVALID_VALUE;
}

LONG RFReaderInfoById( DWORD dwIdentity, struct ReaderContext** sReader ) {
  
  LONG rv;
  int i;

  /* Strip off the lower nibble and get the identity */

  dwIdentity = dwIdentity >> (sizeof(DWORD)/2)*2;
  dwIdentity = dwIdentity << (sizeof(DWORD)/2)*2;

  for ( i=0; i < *dwNumContexts; i++ ) {
    if ( dwIdentity == (sContexts[i])->dwIdentity ) {
      *sReader = sContexts[i];
      return SCARD_S_SUCCESS;
    }
  }

  return SCARD_E_INVALID_VALUE;
}

LONG RFLoadReader( LPSTR lpcReader ) {
  
  LONG rv;
  int i, j;

  /* We must first determine if it is a multi-slot reader
     because it could have another application connected
     to another slot.                                     */
  
  for ( i=0; i < *dwNumContexts; i++ ) {
    if (strcmp(lpcReader, (sContexts[i])->lpcReader) == 0) {
      for ( j=0; j < *dwNumContexts; j++ ) {
	if (((strcmp(sContexts[j]->lpcLibrary, 
		     (sContexts[i])->lpcLibrary) == 0)&&
	     ((strcmp(lpcReader, (sContexts[j])->lpcReader) != 0))&&	   
	     (sContexts[j])->dwPort == (sContexts[i])->dwPort)) {
	  if ( (sContexts[j])->vHandle != 0 ) {
	    (sContexts[i])->vHandle = (sContexts[j])->vHandle;
	    return SCARD_S_SUCCESS;
	  } 
	}
      }
      
      DYN_LoadLibrary( &(sContexts[i])->vHandle, 
		       (sContexts[i])->lpcLibrary );
      
      if ( (sContexts[i])->vHandle == 0 ) {
	return SCARD_E_INVALID_VALUE;
      }
      
      return SCARD_S_SUCCESS;
    }
  }  
  
  return SCARD_E_INVALID_HANDLE;
}

LONG RFUnloadReader( LPSTR lpcReader ) {
  
  LONG rv;
  int i, j;

  /* We must first determine if it is a multi-slot reader
     because it could have another application connected
     to another slot.                                     */

  for ( i=0; i < *dwNumContexts; i++ ) {
    if (strcmp(lpcReader, (sContexts[i])->lpcReader) == 0) {
      for ( j=0; j < *dwNumContexts; j++ ) {
	if (((strcmp(sContexts[j]->lpcLibrary, 
		     (sContexts[i])->lpcLibrary) == 0)&&
	     ((strcmp(lpcReader, (sContexts[j])->lpcReader) != 0))&&	   
	     (sContexts[j])->dwPort == (sContexts[i])->dwPort)) {
	  if ( (sContexts[j])->vHandle != 0 ) {
	    (sContexts[i])->vHandle = 0;
	    return SCARD_S_SUCCESS;
	  } 
	}
      }
      
      DYN_CloseLibrary( &(sContexts[i])->vHandle );
      (sContexts[i])->vHandle = 0;
      return SCARD_S_SUCCESS;
    }
  }
  
  return SCARD_E_INVALID_HANDLE;
}

LONG RFCheckSharing( DWORD hCard ) {

  LONG rv;
  DWORD dwSharing;
  struct ReaderContext *rContext; 

  RFReaderInfoById(hCard, &rContext);
  dwSharing = hCard;

  if ( rContext->dwLockId == 0 || 
       rContext->dwLockId == dwSharing ) {
    return SCARD_S_SUCCESS;
  } else {
    return SCARD_E_SHARING_VIOLATION;
  }

}

LONG RFLockSharing( DWORD hCard ) {
  
  LONG rv;
  struct ReaderContext *rContext; 

  RFReaderInfoById(hCard, &rContext);

  if ( RFCheckSharing(hCard) == SCARD_S_SUCCESS ) {
    rContext->dwLockId = hCard;
  } else {
    return SCARD_E_SHARING_VIOLATION;
  }

  return SCARD_S_SUCCESS;
}

LONG RFUnlockSharing( DWORD hCard ) {

  LONG rv;
  struct ReaderContext *rContext; 

  RFReaderInfoById(hCard, &rContext);

  if ( RFCheckSharing(hCard) == SCARD_S_SUCCESS ) {
    rContext->dwLockId = 0;
  } else {
    return SCARD_E_SHARING_VIOLATION;
  }

  return SCARD_S_SUCCESS;
}

LONG RFUnblockContext( SCARDCONTEXT hContext ) {

  int i;

  for ( i=0; i < *dwNumContexts; i++ ) {
    if ( hContext == (sContexts[i])->dwBlockStatus ) {
      (sContexts[i])->dwBlockStatus = BLOCK_STATUS_RESUME;
    }
  }
  
  return SCARD_S_SUCCESS;
}
